from __future__ import absolute_import

import json
import types
import difflib
import hashlib
import inspect
import functools as ft

from sandbox.common import data as common_data
from sandbox.common import lazy
from sandbox.common import errors as common_errors
from sandbox.common import patterns
from sandbox.common import itertools as common_itertools
from sandbox.common import collections as common_collections

from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt

import six


class InternalValue(object):
    """ Value wrapper used to initialize parameters/requirements without casting """
    def __init__(self, value, decode=False):
        self.__value = value
        self.__decode = decode
        self.__param = None

    def __call__(self, param=None):
        if param is not None:
            self.__param = param
            return self
        assert self.__param is not None, "param must be specified"
        if self.__decode:
            if self.__param.__complex_type__ and isinstance(self.__value, six.string_types):
                return json.loads(self.__value)
        return self.__value


class ServerValue(InternalValue):
    """ Same as InternalValue but for server side """


class ParameterMeta(type):
    __type_id__ = None

    def __new__(mcs, name, bases, namespace):
        frame = inspect.currentframe().f_back.f_back
        if "__names__" not in frame.f_locals:
            frame.f_locals["__names__"] = frame.f_code.co_names
        if not namespace.get("__same_type__"):
            namespace["__type_id__"] = object()
        return type.__new__(mcs, name, bases, namespace)

    def __repr__(cls):
        description = getattr(cls, "description", None)
        return "<{}{}>".format(cls.__name__, ": {!r}".format(description) if description else "")

    def __eq__(cls, other):
        try:
            return cls.__type_id__ is other.__type_id__
        except AttributeError:
            return False

    @property
    def default(cls):
        # noinspection PyUnresolvedReferences
        default_value = cls.default_value
        if isinstance(default_value, lazy.Deferred):
            # noinspection PyAttributeOutsideInit,PyUnresolvedReferences
            cls.default_value = default_value = cls.cast(default_value())
        return default_value

    @property
    def output(cls):
        # noinspection PyUnresolvedReferences
        return cls.__output__


class Parameter(six.with_metaclass(ParameterMeta, object)):
    """
    Base parameter class

    Requirements for overloaded methods:
    - `__decode__`:
        1. Must be recursively idempotent;
        2. Cannot raise exceptions ValueError and TypeError;
        3. cannot call `cast`;
        4. Must call base class method;
        5. Used to construct SDK objects from values saved in DB model or JSON;
    - `cast`:
        1. Must raise exceptions ValueError or TypeError for not valid values, can call `__decode__`;
        2. Used to check values;

    @DynamicAttrs
    """
    class UI(patterns.Abstract):
        """ Specification for UI presentation class. """
        __slots__ = ("type", "modifiers", "context")
        __defs__ = ("void", {}, {})

    def __new__(cls, label=None, **kws):
        cls.__validate_kwargs(kws)

        # `__doc__` has a two-fold role here.
        # In a base class, e.g. Integer, Boolean, etc., it holds a docstring.
        # Once a base class is instantiated, it holds an optional description shown in the UI.
        inherited_description = getattr(cls, "__doc__", None) if cls.__instantiated__ else None
        kws["__doc__"] = kws.pop("description", inherited_description) or ""

        if label is not None:
            kws["description"] = label
        # for backward compatibility
        parameters = kws.pop("__parameters__", None)
        namespace = dict(kws)
        if parameters:
            namespace["__parameters__"] = parameters
        namespace["__same_type__"] = True
        cls = type(cls.__name__, (cls,), namespace)
        default = kws.get("default", ctm.NotExists)
        if default is not ctm.NotExists:
            if isinstance(default, lazy.Deferred):
                kws["default_value"] = default
            else:
                # noinspection PyUnresolvedReferences
                kws["default_value"] = kws["default"] = cls.cast(default) if default is not None else default
        kws["__same_type__"] = True
        kws["__instantiated__"] = True
        return type(cls.__name__, (cls,), kws)

    @classmethod
    def __validate_kwargs(cls, kws):
        valid_kwargs = set(_ for _ in dir(cls) if not _.startswith("__"))
        kwargs = set(_ for _ in kws if not _.startswith("__"))
        kwargs.discard("name")  # TODO: remove assigning names via kwargs from SDK internals

        for arg in kwargs - valid_kwargs:
            similar = {}
            for valid_arg in valid_kwargs:
                ratio = difflib.SequenceMatcher(None, arg, valid_arg).ratio()
                if ratio > 0.8:
                    similar[valid_arg] = ratio

            error_msg = "{}() got an unexpected keyword argument `{}`.".format(cls.__name__, arg)
            if similar:
                most_similar = max(similar, key=similar.get)
                error_msg += " Did you mean `{}`?".format(most_similar)

            raise TypeError(error_msg)

    @classmethod
    def __encode__(cls, value):
        return value

    @classmethod
    def __decode__(cls, value):
        while isinstance(value, InternalValue):
            value = value()
        return value

    __instantiated__ = False
    __parameters__ = None
    __static__ = False
    __output__ = False
    __reset_on_restart__ = False
    __complex_type__ = False
    __lazy_default__ = False
    hint = False

    # to working of autocomplete in IDE
    default = None
    output = None

    # to avoid IDE warnings
    def __call__(self, *_, **__):
        pass


# noinspection PyMethodParameters,PyPropertyDefinition
class ChoiceableMeta(ParameterMeta):
    def __enter__(cls):
        cls.choices = []
        return cls

    def __exit__(cls, *_):
        cls.choices = tuple(cls.choices) if cls.choices else None

    @property
    def values(cls):
        class Values(object):
            def __setitem__(self, key, value):
                # noinspection PyUnresolvedReferences
                if value is None:
                    value = key
                elif isinstance(value, cls.Value):
                    default = value.default
                    value = key if value.value is None else value.value
                    if default:
                        assert not hasattr(cls, "__default_value__"), "Parameter can has only one default value"
                        cls.__default_value__ = key
                        cls.default_value = key
                # noinspection PyUnresolvedReferences
                cls.choices.append((value, key))

            __setattr__ = __setitem__

        return Values()


class Choiceable(six.with_metaclass(ChoiceableMeta, Parameter)):
    class Value(object):
        def __init__(self, value=None, default=False):
            self.value = value
            self.default = default

    # to working of autocomplete in IDE
    values = None


# noinspection PyMethodParameters,PyPropertyDefinition
class SubfieldableMeta(ParameterMeta):
    @staticmethod
    def subfield_cast(value):
        return value

    @property
    def value(cls):
        class Value(object):
            def __getitem__(self, value):
                self.value = type(cls).subfield_cast(value)
                return self

            __getattr__ = __getitem__

            def __enter__(self):
                self.names = set(common_itertools.chain(
                    inspect.currentframe().f_back.f_locals,
                    inspect.currentframe().f_back.f_locals.get("__external_names__", ())
                ))

            def __exit__(self, *_):
                names = set(common_itertools.chain(
                    inspect.currentframe().f_back.f_locals,
                    inspect.currentframe().f_back.f_locals.get("__external_names__", ())
                )) - self.names - {"__external_names__"}
                if not cls.sub_fields:
                    cls.sub_fields = {}
                # noinspection PyUnresolvedReferences
                cls.sub_fields[self.value] = cls.sub_fields.get(self.value, ()) + tuple(names)

        return Value()


class Subfieldable(six.with_metaclass(SubfieldableMeta, Parameter)):
    # to working of autocomplete in IDE
    value = None


class IntegerMeta(SubfieldableMeta):
    @staticmethod
    def subfield_cast(value):
        return common_data.force_int(value, None)


class Integer(six.with_metaclass(IntegerMeta, Subfieldable)):
    pass


class StringMeta(ChoiceableMeta, SubfieldableMeta):
    @staticmethod
    def subfield_cast(value):
        return str(value)


class String(six.with_metaclass(StringMeta, Choiceable, Subfieldable)):
    pass


class BoolMeta(SubfieldableMeta):
    @staticmethod
    def subfield_cast(value):
        return str(value).lower()


class Bool(six.with_metaclass(BoolMeta, Subfieldable)):
    pass


class Resource(Parameter):
    __lazy_default__ = True

    def __new__(cls, label=None, **kws):
        frame = inspect.currentframe().f_back
        if "__names__" not in frame.f_locals:
            frame.f_locals["__names__"] = frame.f_code.co_names
        res_type = kws.get("resource_type")
        if res_type:
            from . import resource as internal_resource
            names = []
            for t in (
                [res_type]
                if inspect.isclass(res_type) and issubclass(res_type, internal_resource.Resource) else
                common_itertools.chain(res_type)
            ):
                assert inspect.isclass(t) and issubclass(t, internal_resource.Resource), (
                    "Value(s) of `resource_type` must be subclass of sdk2.Resource, not a {!r}".format(t)
                )
                names.append(str(t))
            kws["resource_type"] = names[0] if len(names) == 1 else names
        default = kws.get("default", ctm.NotExists)
        if default is not ctm.NotExists and not isinstance(default, lazy.Deferred):
            kws["default"] = lazy.Deferred(lambda: default)
        return super(Resource, cls).__new__(cls, label, **kws)

    @classmethod
    def __encode__(cls, value):
        if not value:
            return value
        from . import resource as internal_resource
        # noinspection PyProtectedMember,PyUnresolvedReferences
        multiple = cls._get_contexted_attribute("multiple", "multiple")
        if multiple:
            if isinstance(value, internal_resource.Resource):
                return [int(value)]
            return [int(v) for v in common_itertools.chain(value)]
        else:
            return int(value)

    @classmethod
    def __decode__(cls, value):
        server_mode = isinstance(value, ServerValue)
        value = super(Resource, cls).__decode__(value)
        if not value:
            return value
        from . import resource as internal_resource

        def safe_int(val):
            try:
                return int(val)
            except (TypeError, ValueError):
                pass

        def safe_resources(val):
            rids = list(six.moves.filter(None, six.moves.map(safe_int, common_itertools.chain(val))))
            if server_mode:
                from sandbox.yasandbox.database import mapping
                if rids:
                    objs = {
                        obj.id: obj for obj in mapping.Resource.objects.exclude("hosts_states").filter(id__in=rids)
                    }
                    for rid in rids:
                        obj = objs.get(rid)
                        if obj is not None:
                            yield internal_resource.Resource.restore(model=obj)
                else:
                    for obj in common_itertools.chain(val):
                        if isinstance(obj, mapping.Resource):
                            yield internal_resource.Resource.restore(model=obj)
                return
            for rid in rids:
                try:
                    yield internal_resource.Resource[rid]
                except common_errors.ResourceNotFound:
                    yield val

        # noinspection PyProtectedMember,PyUnresolvedReferences
        multiple = cls._get_contexted_attribute("multiple", "multiple")
        return lazy.Deferred(
            (
                (lambda: list(safe_resources(value)))
                if multiple else
                (lambda: next(iter(safe_resources(value)), value))
            ), value_repr="Resource:{}".format(value)
        )


class Task(Parameter):
    __lazy_default__ = True

    def __new__(cls, label=None, **kws):
        frame = inspect.currentframe().f_back
        if "__names__" not in frame.f_locals:
            frame.f_locals["__names__"] = frame.f_code.co_names
        task_type = kws.get("task_type")
        if task_type:
            from . import task as internal_task
            names = []
            for t in common_itertools.chain(task_type):
                assert isinstance(t, six.string_types) or issubclass(t, internal_task.Task), (
                    "Value(s) of `task_type` must be subclass of sdk2.Task, not a {!r}".format(t)
                )
                names.append(str(t))
            kws["task_type"] = names[0] if len(names) == 1 else names
        default = kws.get("default", ctm.NotExists)
        if default is not ctm.NotExists and not isinstance(default, lazy.Deferred):
            kws["default"] = lazy.Deferred(lambda: default)
        return super(Task, cls).__new__(cls, label, **kws)

    @classmethod
    def __encode__(cls, value):
        return value and int(value)

    @classmethod
    def __decode__(cls, value):
        server_mode = isinstance(value, ServerValue)
        value = super(Task, cls).__decode__(value)
        if not value:
            return value
        from . import task as internal_task

        def safe_task(val):
            if server_mode:
                from sandbox.yasandbox.database import mapping
                if isinstance(val, internal_task.Task):
                    return val
                elif isinstance(val, mapping.Task):
                    return internal_task.Task.restore(model=val)
                try:
                    val = int(val)
                except (TypeError, ValueError):
                    return
                model = mapping.Task.objects.with_id(val)
                return internal_task.Task.restore(model=model) if model else val
            try:
                return internal_task.Task[val]
            except common_errors.TaskNotFound:
                return val

        # noinspection PyUnresolvedReferences
        return lazy.Deferred(lambda: safe_task(value), "Task:{}".format(value))


# noinspection PyPropertyDefinition
class CheckGroupMeta(ChoiceableMeta):
    # noinspection PyMethodParameters
    @property
    def values(cls):
        class Values(object):
            def __setitem__(self, key, value):
                key = str(key).strip()
                # noinspection PyUnresolvedReferences
                if value is None:
                    value = key
                elif isinstance(value, cls.Value):
                    checked = value.checked
                    value = key if value.value is None else str(value.value).strip()
                    if checked and key:
                        cls.default_value = list(cls.default_value) + [key] if cls.default_value else [key]
                    value = value
                # noinspection PyUnresolvedReferences
                cls.choices.append((value, key))

            __setattr__ = __setitem__

        return Values()


class CheckGroup(six.with_metaclass(CheckGroupMeta, Choiceable)):
    class Value(object):
        def __init__(self, value=None, checked=False):
            self.value = value
            self.checked = checked


class RadioGroupMeta(ChoiceableMeta, SubfieldableMeta):
    @staticmethod
    def subfield_cast(value):
        return str(value).lower() if isinstance(value, bool) else str(value)


class RadioGroup(six.with_metaclass(RadioGroupMeta, Choiceable, Subfieldable)):
    pass


# noinspection PyMethodParameters
class ScopeMeta(ParameterMeta):
    def __enter__(cls):
        cls.__names = set(inspect.currentframe().f_back.f_locals)
        return cls

    def __exit__(cls, *_):
        frame_locals = inspect.currentframe().f_back.f_locals
        names = frame_locals["__names__"]
        frame_locals_keys = sorted(frame_locals.keys(), key=lambda _: names.index(_) if _ in names else -1)
        cls._postprocess(
            frame_locals, sorted(set(frame_locals) - cls.__names, key=lambda _: frame_locals_keys.index(_))
        )

    def _postprocess(cls, frame_locals, names):
        pass


class Repeater(Parameter):
    def __new__(cls, label=None, value_type=None, default=None, required=False, description=None):
        frame = inspect.currentframe().f_back
        if "__names__" not in frame.f_locals:
            frame.f_locals["__names__"] = frame.f_code.co_names
        if value_type is None:
            value_type = cls.default_type
        supported_types, repetition = cls.supported_types, cls.repetition
        assert repetition
        assert issubclass(value_type, supported_types), "{} supports only parameters of types: {}".format(
            cls.__name__, ", ".join(six.moves.map(lambda _: _.__name__, supported_types))
        )
        param = type(value_type.__name__, (value_type,), dict(
            description=label,
            __doc__=description or "",
            default_value=cls.cast(default, cast=value_type.cast),
            required=required,
            cast=classmethod(ft.partial(cls.cast.__func__, cast=value_type.cast)),
            __encode__=classmethod(cls.__encode__.__func__),
            __decode__=classmethod(cls.__decode__.__func__),
            __complex_type__=cls.__complex_type__,
            __instantiated__=True,
            __default_value__=None  # disable default value for choiceable type
        ))
        param.ui = type(param.ui)(*[v for _, v in param.ui])  # "copy" method does not work properly
        param.ui.modifiers = dict(param.ui.modifiers or {})
        param.ui.modifiers["repetition"] = repetition
        return param

    # to avoid IDE warnings
    def __iter__(self):
        pass


# noinspection PyMethodParameters,PyPropertyDefinition
class GroupMeta(ScopeMeta):
    def _postprocess(cls, frame_locals, names):
        from sandbox.sdk2.internal import task as internal_task
        # noinspection PyUnresolvedReferences
        assert (
            id(cls) not in type(internal_task.Parameters).__parameters_registry__
        ), "Modification of group content outside of initial place of definition is denied"
        cls.__locals__ = frame_locals
        cls.__names__ = tuple(list(getattr(cls, "__names__", [])) + list(six.moves.filter(
            lambda _: (lambda _: inspect.isclass(_) and issubclass(_, Parameter) and _ is not cls)(frame_locals[_]),
            names
        )))

    @property
    def names(cls):
        return cls.__names__

    def __iter__(cls):
        """ Iterate across the group parameters """
        for name in cls.__names__:
            yield cls.__locals__[name]


class Group(six.with_metaclass(GroupMeta, Parameter)):
    # to working of autocomplete in IDE
    names = None


# noinspection PyMethodParameters
class OutputMeta(ScopeMeta):
    def __iter_params(cls, frame_locals, names):
        from sandbox.sdk2.internal import task
        ext_params = []
        for name in names:
            param = frame_locals[name]
            if not inspect.isclass(param):
                continue
            if issubclass(param, task.Parameters):
                ext_params.append(param)
                continue
            if not issubclass(param, Parameter) or param is cls:
                continue
            yield param
        for subparam in ext_params:
            for param in subparam:
                yield param

    def _postprocess(cls, frame_locals, names):
        for param in cls.__iter_params(frame_locals, names):
            if param.ui:
                param.ui = type(param.ui)(*[v for _, v in param.ui])  # "copy" method does not work properly
                param.ui.modifiers = param.ui.modifiers or {}
            param.__output__ = True
            # noinspection PyUnresolvedReferences
            param.__reset_on_restart__ = cls.__reset_on_restart__


class Output(six.with_metaclass(OutputMeta, Parameter)):
    def __new__(cls, reset_on_restart=False, **kws):
        kws["__reset_on_restart__"] = reset_on_restart
        return type(cls.__name__, (cls,), kws)


class DeduplicationUniqueness(Parameter):
    ui = None
    dummy = False
    required = False
    default_value = common_collections.ImmutableAttrDict(
        key=None,
        excluded_statuses=(ctt.Status.DELETED,)
    )

    @classmethod
    def __decode__(cls, value):
        value = super(DeduplicationUniqueness, cls).__decode__(value)
        if value is None:
            return cls.default
        return (
            type(DeduplicationUniqueness.default)(
                key=value["key"],
                excluded_statuses=value.get(
                    "excluded_statuses",
                    DeduplicationUniqueness.default.excluded_statuses
                )
            )
            if isinstance(value, dict) else
            type(DeduplicationUniqueness.default)(
                key=value,
                excluded_statuses=(cls.default or {}).get(
                    "excluded_statuses",
                    DeduplicationUniqueness.default.excluded_statuses
                )
            )
        )

    @classmethod
    def cast(cls, value):
        assert isinstance(value, (types.NoneType, dict)), "Wrong value: {!r}".format(value)
        return value

    @classmethod
    def hash(cls, task):
        return hashlib.md5(b" ".join(
            b":".join((six.ensure_binary(n), six.ensure_binary(str(v))))
            for n, v in sorted(task.Parameters.__values__.items())
            if not issubclass(getattr(type(task).Parameters, n), cls)
        )).hexdigest()
