from __future__ import absolute_import

import os
import inspect
import logging
import weakref
import textwrap
import aniso8601

import six
from six.moves import http_client as httplib
from six.moves import urllib_parse as url_parse

from sandbox.common import rest as common_rest
from sandbox.common import errors as common_errors
from sandbox.common import config as common_config
from sandbox.common import system as common_system
from sandbox.common import itertools as common_itertools
from sandbox.common import collections as common_collections

from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.common.types import resource as ctr

from .. import path as sdk2_path

from . import common as internal_common
from . import parameters as internal_parameters

PACKAGE_PREFIX = __package__.rpartition(".")[0] + "."
RESERVED_NAMES = {}


class AttributesMeta(type):
    __registry__ = ()

    def __new__(mcs, name, bases, namespace):
        mcs.__registry__ = tuple(
            value
            for value in six.itervalues(namespace)
            if inspect.isclass(value) and issubclass(value, internal_parameters.Parameter)
        )
        return type.__new__(mcs, name, bases, namespace)

    def __contains__(cls, item):
        return item in type(cls).__registry__


class Attributes(six.with_metaclass(AttributesMeta, object)):
    pass


class SystemAttributes(object):
    __props__ = None

    def __setattr__(self, key, value):
        """ Protect from setting system attribute """

    def __init__(self, linux_platform=None, osx_platform=None, osx_arm_platform=None, win_nt_platform=None):
        object.__setattr__(
            self, "__props__",
            common_collections.AttrDict(
                linux_platform=linux_platform,
                osx_platform=osx_platform,
                osx_arm_platform=osx_arm_platform,
                win_nt_platform=win_nt_platform
            )
        )

    @property
    def linux_platform(self):
        return self.__props__.linux_platform

    @property
    def osx_platform(self):
        return self.__props__.osx_platform

    @property
    def osx_arm_platform(self):
        return self.__props__.osx_arm_platform

    @property
    def win_nt_platform(self):
        return self.__props__.win_nt_platform


class ResourceQuery(internal_common.Query):
    @property
    def count(self):
        # "total" is not returned in APIv2, fallback to v1
        # TODO: SANDBOX-7365: move to dedicated resource count endpoints when they are implemented
        from sandbox import sdk2
        rest = common_rest.Path(sdk2.Task.server)
        rest <<= common_rest.Client.HEADERS({ctm.HTTPHeader.NO_LINKS: "1", ctm.HTTPHeader.RESOURCE_META: "1"})
        return rest.resource[self._constraints, 0:0:self._order]["total"]


class ResourceMeta(internal_common.Type):
    __registry__ = {}
    __base_properties__ = set()
    __cache__ = weakref.WeakValueDictionary()

    def __new__(mcs, name, bases, namespace):
        abstract = namespace.pop("__abstract__", False)
        if bases == (object,):
            mcs.__base_properties__ = frozenset(k for k, v in six.iteritems(namespace) if isinstance(v, property))
            # noinspection PyUnresolvedReferences
            return mcs.__base__.__new__(mcs, name, bases, namespace)
        assert len(bases) == 1, "Multiple inheritance for resource types is not supported"
        base = bases[0]
        attrs = {} if base is object or base is Resource else dict(base.__attrs__)
        default_attribute = namespace.pop("__default_attribute__", base.__default_attribute__)
        new_namespace = {"__default_attribute__": default_attribute}
        for key, value in six.iteritems(namespace):
            assert key not in mcs.__base_properties__ or value is Resource.__dict__.get(key), (
                "Cannot override base property {!r}".format(key)
            )
            attr_name = key.rstrip("_")
            attr = attrs.get(attr_name)
            if attr:
                assert (
                    not inspect.isclass(value) or
                    not issubclass(value, internal_parameters.Parameter)
                ), "Value of attribute {!r} must be of type {!r}".format(key, attr)
                attrs[attr_name] = attr(default=value)
            elif inspect.isclass(value) and issubclass(value, internal_parameters.Parameter):
                assert value in Attributes, "Attribute type {!r} is not supported".format(value)
                attr = attrs[attr_name] = value
            new_namespace[key] = mcs.__attribute__(key, attr) if attr else value
        properties = {
            key: property(lambda _, p=param: p.default)
            for key, param in six.iteritems(attrs)
        }
        attrs = common_collections.ImmutableDict(attrs)
        properties["__attrs__"] = property(lambda _: attrs)
        cls = internal_common.Type.__new__(
            type(mcs.__name__, (mcs,), properties),
            name, bases, new_namespace
        )

        if abstract and cls.__module__.startswith(PACKAGE_PREFIX):
            RESERVED_NAMES[cls.name] = cls
        assert abstract or cls.name not in RESERVED_NAMES, "Resource class name ({}) cannot be in: {}".format(
            cls.name, RESERVED_NAMES
        )
        if not abstract:
            stored_cls = mcs.__registry__.setdefault(cls.name, cls)
            if stored_cls is not cls:
                raise common_errors.ResourceDuplicationError(
                    "Resource name {} duplicated: {}.{}, {}.{}".format(
                        cls.name, cls.__module__, cls.__name__, stored_cls.__module__, stored_cls.__name__
                    )
                )
        return cls

    @staticmethod
    def __attribute__(name, attr):
        def getter(self):
            return self.__attrs__.get(name, attr.default)

        def setter(self, value):
            self.__attrs__[name] = attr.cast(value)

        return property(getter, setter)

    def __str__(cls):
        return cls.name

    def __repr__(cls):
        return "<Resource type {!r}>".format(cls.name)

    def __getitem__(cls, resource_id_or_name):
        """
        Get resource by id or resource class by name

        :param resource_id_or_name: resource id or resource type name
        :return: Resource object or Resource class
        """
        try:
            resource_id = int(resource_id_or_name)
        except (TypeError, ValueError):
            resource_cls = type(cls).__registry__.get(resource_id_or_name)
            if resource_cls is None:
                from sandbox import sdk2
                if sdk2.Task.current is None:
                    raise common_errors.UnknownResourceType(
                        "Unknown resource type {!r}".format(resource_id_or_name)
                    )
                try:
                    resource_cls = cls.__setstate__(
                        sdk2.Task._sdk_server.resource.meta_list[resource_id_or_name].read()
                    )
                except common_rest.Client.HTTPError as ex:
                    if ex.response.status_code == httplib.NOT_FOUND:
                        raise common_errors.UnknownResourceType(
                            "Unknown resource type {!r}".format(resource_id_or_name)
                        )
                    else:
                        raise
            return resource_cls
        return cls.restore(resource_id=resource_id)

    def __getstate__(cls):
        """
        Get resource class dict representation

        :return: Dict with resource class representation
        """
        current_cls = cls
        resource_meta = []
        while current_cls.name not in RESERVED_NAMES:
            parent_class = current_cls.__bases__[0]

            # noinspection PyUnresolvedReferences
            result = {
                "name": str(current_cls),
                "releaseable": current_cls.releaseable,
                "releasable": current_cls.releasable,
                "any_arch": current_cls.any_arch,
                "releasers": current_cls.releasers,
                "executable": current_cls.executable,
                "auto_backup": current_cls.auto_backup,
                "calc_md5": current_cls.calc_md5,
                "on_restart": current_cls.restart_policy,
                "release_subscribers": current_cls.release_subscribers,
                "share": current_cls.share,
                "parent": str(parent_class)
            }
            # noinspection PyUnresolvedReferences
            if current_cls.__default_attribute__ is not None:
                # noinspection PyUnresolvedReferences
                result["default_attribute"] = current_cls.__default_attribute__.__name__

            attributes = {}

            # noinspection PyUnresolvedReferences
            for attr_name, attr_cls in six.iteritems(current_cls.__attrs__):
                # noinspection PyUnresolvedReferences
                if attr_name in parent_class.__attrs__:
                    # noinspection PyUnresolvedReferences
                    if attr_cls is not parent_class.__attrs__[attr_name]:
                        attribute = {"value": attr_cls.default}
                    else:
                        continue
                else:
                    attribute = {
                        "class_name": attr_cls.__name__,
                        "description": attr_cls.description,
                        "default": attr_cls.default,
                        "required": attr_cls.required
                    }
                attributes[attr_name] = attribute
            result["attributes"] = attributes
            resource_meta.append(result)
            current_cls = parent_class
        return resource_meta

    def __set_state(cls, state):
        from sandbox import sdk2
        if state["name"] in sdk2.Resource:
            return sdk2.Resource[state["name"]]
        namespace = state.copy()
        parent = namespace.pop("parent", None) or sdk2.Resource
        attributes = namespace.pop("attributes")
        default_attribute = namespace.pop("default_attribute", None)
        if default_attribute is not None:
            namespace["__default_attribute__"] = getattr(sdk2.parameters, default_attribute)
        for attr_name, attr_cls in six.iteritems(attributes):
            if "value" in attr_cls:
                namespace[attr_name] = attr_cls["value"]
            else:
                namespace[attr_name] = getattr(sdk2.Attributes, attr_cls["class_name"])(
                    attr_cls["description"],
                    default=attr_cls["default"],
                    required=attr_cls["required"]
                )

        parent_cls = RESERVED_NAMES.get(parent) or cls[parent]
        try:
            return type(cls).__new__(
                type(parent_cls),
                str(namespace["name"]),
                (parent_cls,),
                namespace
            )
        except common_errors.ResourceDuplicationError:
            return sdk2.Resource[str(namespace["name"])]

    def __setstate__(cls, state):
        """
        Create new resource class from resource dict representation

        :param state: resource dict representation
        :return: resource class
        """
        from sandbox import sdk2

        if state[0] in sdk2.Resource:
            return cls.__set_state(state[0])

        res_cls = None
        for st in reversed(state):
            res_cls = cls.__set_state(st)
        return res_cls

    def __contains__(cls, type_name):
        """
        Check existence of resource type
        """
        return str(type_name) in type(cls).__registry__

    def __iter__(cls):
        """
        Iterate all resource types if called on the base abstract class
        """
        registry = type(cls).__registry__
        return (_ for _ in (cls,)) if cls.name in registry else six.itervalues(registry)

    def __eq__(cls, other):
        return str(cls) == str(other)

    def __ne__(cls, other):
        return str(cls) != str(other)

    def __hash__(cls):
        return hash(cls.name)

    def from_model(cls, model):
        def resource_meta_from_model(meta):
            return dict(
                name=meta.name,
                parent=meta.parent,
                releaseable=meta.releaseable,
                releasable=meta.releasable,
                any_arch=meta.any_arch,
                releasers=meta.releasers,
                executable=meta.executable,
                auto_backup=meta.auto_backup,
                calc_md5=meta.calc_md5,
                on_restart=meta.on_restart,
                release_subscribers=meta.release_subscribers,
                share=meta.share,
                default_attribute=meta.default_attribute,
                attributes=meta.attributes
            )

        if getattr(model, "resource_meta_objects", None):
            resource_meta = [
                resource_meta_from_model(meta.resource_meta) for meta in model.resource_meta_objects
            ]
            resource_cls = cls.__setstate__(resource_meta)
        else:
            resource_cls = cls[model.type]
        resource = object.__new__(resource_cls).__setstate_from_model__(model)
        return resource

    def from_json(cls, data):
        from sandbox import sdk2

        if sdk2.Task.current is not None and data.get("resource_meta"):
            resource_cls = cls.__setstate__(data["resource_meta"])
        else:
            if "type" not in data:
                logging.error("Resource data does not contain required key. Data content:\n%s", data)
                raise KeyError("Resource data does not contain required key. See more info in logs")
            resource_cls = cls[data["type"]]
        resource = object.__new__(resource_cls).__setstate__(data)
        return resource

    @staticmethod
    def data(resource_id):
        from sandbox import sdk2
        try:
            rest = common_rest.Path(sdk2.Task._sdk_server)
            rest <<= common_rest.Client.HEADERS({ctm.HTTPHeader.NO_LINKS: "1", ctm.HTTPHeader.RESOURCE_META: "1"})
            return rest.resource[resource_id][:]
        except sdk2.Task._sdk_server.HTTPError as ex:
            if ex.status == httplib.NOT_FOUND:
                raise common_errors.ResourceNotFound("Resource #{} not found".format(resource_id))
            else:
                raise

    def restore(cls, data=None, resource_id=None, model=None):
        """
        Restore resource object from data received from REST API

        :param data: JSON
        :param resource_id: resource identifier
        :param model: database model to restore from
        :return: Resource object
        """
        assert len(list(filter(None, (data, resource_id, model)))) == 1, "Only one argument should be specified"
        if data:
            resource_id = data["id"]
        elif model:
            resource_id = model.id
        try:
            resource = cls.__cache__[resource_id]
        except KeyError:
            if model:
                resource = cls.from_model(model)
            else:
                resource = cls.from_json(data or cls.data(resource_id))
            cls.__cache__[resource_id] = resource
        return resource

    def restore_cached(cls, ids):
        """
        Restore resources from cache if possible.

        :param ids: List of resource ids to restore
        :return: List of restored resources and list of unmatched ids
        :rtype: list(sdk2.Resource), list(int)
        """
        found = []
        not_found = []
        for resource_id in ids:
            try:
                resource = cls.__cache__[resource_id]
                found.append(resource)
            except KeyError:
                not_found.append(resource_id)
        return found, not_found

    @staticmethod
    def adapt_query_constraints(constrants):
        from sandbox import sdk2
        rid = constrants.get("id")
        if rid:
            constrants["id"] = ",".join(six.moves.map(str, common_itertools.chain(rid)))

        tids = []
        for task in common_itertools.chain(constrants.pop("task", None)):
            if task is not None:
                assert isinstance(task, sdk2.Task), "`task` must be instance of `Task`"
                tids.append(str(task.id))
        if tids:
            constrants["task_id"] = ",".join(tids)

        state = constrants.get("state")
        if state:
            constrants["state"] = ",".join(common_itertools.chain(state))

        task = constrants.pop("dependant", None)
        if task is not None:
            assert isinstance(task, sdk2.Task), "`dependant` must be instance of `Task`"
            constrants["dependant"] = task.id

        resource_type = constrants.pop("resource_type", None)
        if resource_type is not None:
            assert (
                isinstance(resource_type, six.string_types) or
                inspect.isclass(resource_type) and issubclass(resource_type, Resource)
            ), "`resource_type` must be string or subclass of `Resource`"
            constrants["type"] = str(resource_type)
        # TODO: more parameters and checks

    def find(cls, resource_type=None, **constraints):
        """
        Find resources according to specified constraints

        :param resource_type: find resources of the type, if None then find resources of type of current class
        :param constraints: query constraints, according to :py:class:`sandbox.web.api.v1.resource.ResourceList.Get`
                            with the following exceptions:
                            `id` contains resource id or list of ids,
                            `task` instead of `task_id`, contains task id or list of ids,
                            `state` contains resource state or list of states,
                            `dependant` contains task object
        :rtype: sandbox.sdk2.internal.common.Query
        """
        from sandbox import sdk2
        constraints.setdefault("resource_type", resource_type or (cls in cls or None) and cls)
        cls.adapt_query_constraints(constraints)
        rest = common_rest.Path(sdk2.Task._sdk_server.copy(version=2))
        rest <<= common_rest.Client.HEADERS({ctm.HTTPHeader.NO_LINKS: "1", ctm.HTTPHeader.RESOURCE_META: "1"})
        return ResourceQuery(rest.resource, cls.restore, **constraints)

    @property
    def description(cls):
        """
        HTML description of the resource type generated on base of appropriate class' documentation string.
        """
        try:
            return cls.__description__
        except AttributeError:
            pass
        import docutils.core
        # noinspection PyAttributeOutsideInit
        cls.__description__ = docutils.core.publish_parts(
            textwrap.dedent(cls.__doc__).strip(), writer_name="html"
        )["html_body"] if cls.__doc__ else ""
        return cls.__description__

    @description.setter
    def description(cls, value):
        # noinspection PyAttributeOutsideInit
        cls.__description__ = value

    @property
    def on_restart(cls):
        # noinspection PyUnresolvedReferences
        return cls.restart_policy


class Resource(six.with_metaclass(ResourceMeta, object)):

    __default_attribute__ = None

    __props__ = None
    __attrs__ = None
    __meta__ = None

    def __new__(cls, task, description, path, arch=None, system_attributes=None, **attrs):
        """
        Create new resource
        """
        assert task and (task is task.current or task is task.current.parent), "task must be current task or its parent"
        for_parent = task is not task.current

        if arch is None:
            # noinspection PyUnresolvedReferences
            arch = ctm.OSFamily.ANY if cls.any_arch else common_config.Registry().this.system.family
        assert arch in ctm.OSFamily, "arch must be one of {!r}".format(ctm.OSFamily)
        # set default attributes
        for name, attr_type in six.iteritems(cls.__attrs__):
            if attr_type.__static__:
                continue
            value = attr_type.default
            if value is None:
                continue
            attrs.setdefault(name, value)
        # set attributes
        for name, value in six.iteritems(attrs):
            attr_type = cls.__attrs__.get(name, cls.__default_attribute__)
            if attr_type is None:
                raise ValueError("Unknown attribute {!r}".format(name))
            try:
                attr_type.cast(value)
            except (TypeError, ValueError) as ex:
                raise TypeError(
                    "Value {!r} of attribute {!r} cannot be casted to {!r}: {}".format(value, name, attr_type, ex)
                )

        if for_parent:
            attrs["from_task"] = task.current.id

        from sandbox import sdk2
        # noinspection PyUnresolvedReferences,PyTypeChecker
        resource_data = task.current.agentr.resource_register(
            str(path), str(cls), description, arch, attrs,
            share=cls.share, service=issubclass(cls, sdk2.ServiceResource),
            for_parent=for_parent, resource_meta=cls.__getstate__(),
            system_attributes=system_attributes
        )

        # noinspection PyUnresolvedReferences
        return cls.restore(resource_data)

    def __repr__(self):
        # noinspection PyUnresolvedReferences
        return "<Resource {}:{}>".format(type(self).name, self.id)

    def __getattr__(self, key):
        key = key.rstrip("_")
        if key not in self.__attrs__:
            raise AttributeError("{!r} is not an attribute of {}".format(key, self.type))
        return self.__attrs__[key]

    def __setattr__(self, key, value):
        from sandbox import sdk2

        if key in self.__props__ or key == "releasers":  # TODO: Hotfix for market's hack with releases list changing
            return super(Resource, self).__setattr__(key, value)
        key = key.rstrip("_")
        if key not in self.type.__attrs__:
            if not self.type.__default_attribute__:
                raise AttributeError("{!r} is not an attribute of {}".format(key, self.type))
            value = self.type.__default_attribute__.cast(value)
        new_attr = key not in self.__attrs__
        if key not in self.type.__attrs__:
            self.__attrs__[key] = value
        super(Resource, self).__setattr__(key, value)

        data = dict(name=key, value=value)
        # Try until either new attribute is created or existing one is updated
        while True:
            if new_attr:
                try:
                    sdk2.Task._sdk_server.resource[self.id].attribute.create(data)
                except common_rest.Client.HTTPError as ex:
                    # noinspection PyUnresolvedReferences
                    if ex.status == httplib.CONFLICT:
                        # Attribute already exists, try again with update
                        new_attr = False
                    else:
                        raise
                else:
                    break
            else:
                try:
                    sdk2.Task._sdk_server.resource[self.id].attribute[key].update(data)
                except common_rest.Client.HTTPError as ex:
                    if ex.status == httplib.NOT_FOUND:
                        # Attribute not found, try again with create
                        new_attr = True
                    else:
                        raise
                else:
                    break

    def __delattr__(self, key):
        from sandbox import sdk2
        key = key.rstrip("_")
        if key not in self.__attrs__:
            raise AttributeError("{!r} is not an attribute of {}".format(key, self.type))
        del sdk2.Task._sdk_server.resource[self.id].attribute[key]
        self.__attrs__.pop(key)

    def __iter__(self):
        return six.iteritems(self.__attrs__)

    def __set_props(self, state):
        object.__setattr__(self, "__props__", common_collections.AttrDict(state))
        self.__props__.task_id = state["task"]["id"]
        self.__props__.path = sdk2_path.Path(state["file_name"])
        self.__props__.http_proxy = state["http"]["proxy"]
        system_attributes_dict = state.get("system_attributes", None) or {}
        self.__props__.system_attributes = SystemAttributes(
            linux_platform=system_attributes_dict.get("linux_platform"),
            osx_platform=system_attributes_dict.get("osx_platform"),
            osx_arm_platform=system_attributes_dict.get("osx_arm_platform"),
            win_nt_platform=system_attributes_dict.get("win_nt_platform")
        )
        time_ = state["time"]
        self.__props__.update(
            (_, aniso8601.parse_datetime(time_[_]))
            for _ in ("accessed", "created", "expires", "updated")
            if time_[_]
        )
        for _ in ("attributes", "http", "rsync", "sources", "time", "file_name", "task"):
            self.__props__.pop(_, None)

    def __set_attrs(self, state):
        object.__setattr__(self, "__attrs__", common_collections.AttrDict())
        cls = type(self)
        for name, value in six.iteritems(state["attributes"]):
            attr_type = cls.__attrs__.get(name, cls.__default_attribute__)
            if attr_type is None:
                self.__attrs__[name] = value
            else:
                try:
                    self.__attrs__[name] = attr_type.cast(value)
                except (TypeError, ValueError) as ex:
                    raise TypeError(
                        "Value {!r} of attribute {!r} cannot be casted to {!r}: {}".format(value, name, attr_type, ex)
                    )

    @internal_common.dual_method
    def __setstate__(self, state):
        assert state["type"] == self.type.name
        object.__setattr__(self, "__meta__", dict(state))
        self.__set_props(state)
        self.__set_attrs(state)
        return self

    def __local_http_url(self, model):
        if common_config.Registry().common.installation != ctm.Installation.LOCAL:
            return None
        http_prefix = common_system.get_sysparams()["fileserver"]
        path = (
            os.path.join(*(ctt.relpath(model.task_id) + [url_parse.quote(model.path)]))
        )
        return url_parse.urljoin(http_prefix, path) if http_prefix else ""

    def __setstate_from_model__(self, model):
        assert model.type == type(self).name
        proxy_settings = common_config.Registry().client.fileserver.proxy
        http_proxy = (
            "{}://{}/{}".format(proxy_settings.scheme.http, proxy_settings.host, model.id)
            if proxy_settings.host else
            self.__local_http_url(model)
        )
        object.__setattr__(self, "__props__", common_collections.AttrDict(
            id=model.id,
            path=model.path,
            description=model.name,
            owner=model.owner,
            task_id=model.task_id,
            state=model.state,
            arch=model.arch,
            created=model.time.created,
            accessed=model.time.accessed,
            updated=model.time.updated,
            expires=model.time.expires,
            size=model.size << 10,
            skynet_id=model.skynet_id,
            md5=model.md5,
            http_proxy=http_proxy,
            system_attributes=SystemAttributes(
                linux_platform=model.system_attributes and model.system_attributes.linux_platform,
                osx_platform=model.system_attributes and model.system_attributes.osx_platform,
                osx_arm_platform=model.system_attributes and model.system_attributes.osx_arm_platform,
                win_nt_platform=model.system_attributes and model.system_attributes.win_nt_platform,
            )
        ))
        object.__setattr__(self, "__attrs__", common_collections.AttrDict())
        cls = type(self)
        for attr in model.attributes or ():
            attr_type = cls.__attrs__.get(attr.key, cls.__default_attribute__)
            self.__attrs__[attr.key] = (
                attr.value
                if attr_type is None else
                attr_type.__decode__(attr.value)
            )
        return self

    def __int__(self):
        return self.id

    @internal_common.dual_method
    def __getstate__(self):
        return self.id

    @internal_common.Query.SortedField
    def id(self):
        return self.__props__.id

    @internal_common.Query.SortedField
    def type(self):
        return type(self)

    @property
    def path(self):
        return self.__props__.path

    @path.setter
    def path(self, value):
        from sandbox import sdk2
        current = sdk2.Task.current
        if self.__props__.task_id != current.id and self.__attrs__.get("from_task") != current.id:
            raise ValueError("Cannot change path of non-own resource.")
        if self.__props__.state != ctr.State.NOT_READY:
            raise ValueError("Cannot change path of non-NOT_READY resource.")

        current.agentr.resource_update(self.__props__.id, path=value)
        self.__props__.path = value

    @property
    def description(self):
        return self.__props__.description

    @description.setter
    def description(self, value):
        self.__props__.description = str(value)

    @property
    def owner(self):
        return self.__props__.owner

    @property
    def task_id(self):
        return self.__props__.task_id

    @internal_common.Query.SortedField
    def task(self):
        try:
            return self.__props__.task
        except KeyError:
            from sandbox import sdk2
            self.__props__.task = sdk2.Task[self.__props__.task_id]
            return self.__props__.task

    @internal_common.Query.SortedField
    def state(self):
        return self.__props__.state

    @property
    def arch(self):
        return self.__props__.arch

    @internal_common.Query.SortedField
    def created(self):
        return self.__props__.created

    @internal_common.Query.SortedField
    def accessed(self):
        return self.__props__.accessed

    @internal_common.Query.SortedField
    def updated(self):
        return self.__props__.updated

    @internal_common.Query.SortedField
    def expires(self):
        return self.__props__.expires

    @internal_common.Query.SortedField
    def client(self):
        return None

    @property
    def size(self):
        return self.__props__.size

    @property
    def skynet_id(self):
        return self.__props__.skynet_id

    @property
    def md5(self):
        return self.__props__.md5

    @property
    def rights(self):
        return self.__props__.rights

    @property
    def url(self):
        return self.__props__.url

    @property
    def http_proxy(self):
        return self.__props__.http_proxy

    @property
    def system_attributes(self):
        return self.__props__.system_attributes

    def _is_parent_resource(self):
        from sandbox import sdk2

        task = sdk2.Task.current
        if task.parent and self.task_id == task.parent.id:
            if self.from_task == task.id:
                return True
            for param in task.type.Parameters:
                if issubclass(param, sdk2.parameters.ParentResource):
                    res = getattr(task.Parameters, param.name, None)
                    if isinstance(res, list):
                        if self in res:
                            return True
                    elif self == res:
                        return True

    def reload(self, data=None):
        """ Reload resource info from API """
        return self.__setstate__(data or type(self).data(self.__props__.id))


class ResourceData(object):
    def __init__(self, resource, fastbone=None):
        assert isinstance(resource, Resource), "instance of `Resource` is expected, not an `{!r}`".format(
            type(resource)
        )
        from sandbox import sdk2
        self.__resource = resource
        if self.__resource.state == ctr.State.READY:
            kws = common_collections.AttrDict()
            if fastbone is not None:
                kws.fastbone = fastbone
            # noinspection PyUnresolvedReferences
            self.__path = sdk2.Path(sdk2.Task.current.agentr.resource_sync(self.__resource.id, **kws))
        elif self.__resource.task_id != sdk2.Task.current.id and not resource._is_parent_resource():
            raise common_errors.InvalidResource(
                "Resource #{} from task #{} is in state {} and cannot be synchronized".format(
                    self.__resource.id, self.__resource.task_id, self.__resource.state
                )
            )
        else:
            self.__path = sdk2.Path(common_config.Registry().client.tasks.data_dir).joinpath(
                *ctt.relpath(sdk2.Task.current.id)
            ).joinpath(self.__resource.path)

            cls = type(resource)

            # Add source for resources subclassed from TASK_LOGS (such as TASK_CUSTOM_LOGS).
            # It makes them available via proxy during the task execution.
            add_source = issubclass(cls, sdk2.service_resources.TaskLogs)

            sdk2.Task.current.agentr.resource_register_meta(
                str(self.__path), resource.__meta__,
                share=cls.share, service=issubclass(cls, sdk2.ServiceResource), add_source=add_source
            )

    @property
    def path(self):
        """
        """
        return self.__path

    def ready(self):
        """
        """
        from sandbox import sdk2
        # noinspection PyUnresolvedReferences
        self.__resource.reload(sdk2.Task.current.agentr.resource_complete(self.__resource.id))

    def broken(self):
        """
        """
        from sandbox import sdk2
        # noinspection PyUnresolvedReferences
        self.__resource.reload(sdk2.Task.current.agentr.resource_complete(self.__resource.id, broken=True))

    def mount(self, name=None):
        """
        """
        from sandbox import sdk2
        if not self.__path:
            raise common_errors.InvalidResource(
                "Resource #{} from task #{} is in state {} and cannot be mounted as image".format(
                    self.__resource.id, self.__resource.task_id, self.__resource.state
                )
            )
        return sdk2.Path(sdk2.Task.current.agentr.mount_image(self.__path, dirname=name))
