from __future__ import absolute_import

import os
import sys
import json
import copy
import inspect
import logging
import weakref
import textwrap
import itertools as it
import functools as ft
import collections

import six
from six.moves import http_client as httplib
# noinspection PyPep8Naming
from six.moves import cPickle as pickle
if six.PY2:
    import pathlib2 as pathlib
else:
    # noinspection PyUnresolvedReferences
    import pathlib

import aniso8601

from sandbox.common import lazy
from sandbox.common import rest
from sandbox.common import urls as common_urls
from sandbox.common import errors as common_errors
from sandbox.common import config
from sandbox.common import format
from sandbox.common import encoding
from sandbox.common import patterns
from sandbox.common import projects_handler
from sandbox.common import itertools as common_itertools
from sandbox.common import collections as common_collections

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
import sandbox.common.types.notification as ctn

from sandbox.sdk2.helpers import misc

from . import common as internal_common
from . import parameters as internal_parameters


# Sandbox API version for internal SDK requests.
SDK_API_VERSION = "1.0"


def safe_encode(param, name, value):
    try:
        value = param.__encode__(value)
    except (TypeError, ValueError) as ex:
        logging.warning(
            "Error while encoding value %r of parameter %r of type %r: %s",
            value, name, param, ex
        )
    return value


def safe_decode(param, name, value):
    try:
        value = param.__decode__(value)
    except (TypeError, ValueError) as ex:
        while isinstance(value, internal_parameters.InternalValue):
            value = value()
        logging.warning(
            "Error while decoding value %r of parameter %r of type %r: %s",
            value, name, param, ex
        )
    return value


class RequirementsMeta(type):
    def __new__(mcs, name, bases, namespace):
        if bases == (object,):
            # noinspection PyUnresolvedReferences
            return mcs.__base__.__new__(mcs, name, bases, namespace)

        allowed_req_types = dict(inspect.getmembers(mcs, lambda _: isinstance(_, property)))
        req_types = dict(allowed_req_types)
        for key, value in six.iteritems(namespace):
            if inspect.isclass(value) and issubclass(value, internal_parameters.Parameter):
                prop = allowed_req_types and allowed_req_types.get(key)
                assert (
                    not allowed_req_types or prop and value.__type_id__ is prop.fget(None).__type_id__
                ), "Cannot define new requirement {!r} of type {!r}".format(key, value)
                req_types[key] = property(
                    internal_common.memoized(
                        lambda _, p=value, n=key:
                        p(name=n)
                    )
                )
            else:
                prop = allowed_req_types.get(key)
                if prop is None:
                    continue
                param_type = prop.fget(None)
                if not param_type.__lazy_default__:
                    param_type.cast(value)
                req_types[key] = property(
                    internal_common.memoized(
                        lambda _, p=param_type, v=value, n=key:
                        p(name=n, default=v)
                    )
                )

        for param_name in req_types:
            namespace[param_name] = mcs.__property__(param_name, req_types)
        namespace["__requirements_names__"] = tuple(req_types)
        cls = type.__new__(
            type(mcs.__name__, (mcs,), req_types),
            name, bases, namespace
        )
        return cls

    def __iter__(cls):
        # noinspection PyUnresolvedReferences
        for name in cls.__requirements_names__:
            yield getattr(cls, name)

    def __setattr__(cls, *_):
        raise AttributeError("Unsupported operation")

    @staticmethod
    def __property__(name, params):
        def getter(self):
            value = self.__values__.get(name, ctm.NotExists)
            if value is ctm.NotExists:
                return params[name].fget(None)().default
            while isinstance(value, lazy.Deferred):
                value = self.__values__[name] = value()
            return value

        def setter(self, value):
            param = params[name].fget(None)
            self.__values__[name] = lazy.Deferred(
                lambda: param.__decode__(
                    value(param())
                    if isinstance(value, internal_parameters.InternalValue) else
                    param().cast(value)
                )
            )

        return property(getter, setter)


class Requirements(six.with_metaclass(RequirementsMeta, object)):
    # list of custom parameters names sorted by order of definition
    __requirements_names__ = ()
    # input parameters are readonly
    __locked__ = False
    # fields that do not present in template database model
    __unused_template_requirements__ = {"environments", "resources"}

    def __init__(self, reqs=None, model=None, taskbox=None):
        if reqs and model:
            raise RuntimeError("Can't use both json state and model")

        object.__setattr__(self, "__values__", {})
        # noinspection PyUnresolvedReferences
        object.__setattr__(self, "Caches", type(self).Caches())

        if reqs:
            self._from_json(reqs)
        if model:
            self._from_model(model, taskbox=taskbox)

    def _from_json(self, state):
        for name, value in six.iteritems(state):
            if name == "caches":
                # noinspection PyUnresolvedReferences
                object.__setattr__(self, "Caches", type(self).Caches(value))
                continue
            if name not in self.__requirements_names__:
                continue
            if value:
                if name == "ramdrive":
                    value = value.copy()
                    value["size"] >>= 20
                elif name in ("ram", "disk_space"):
                    value >>= 20
                elif name == "resources":
                    value = value.get("ids")
            setattr(self, name, value)

    def _from_model(self, model, taskbox=None):
        self.host = internal_parameters.ServerValue(model.requirements.host)
        self.privileged = internal_parameters.ServerValue(bool(model.requirements.privileged))
        self.disk_space = internal_parameters.ServerValue(model.requirements.disk_space)
        self.ram = internal_parameters.ServerValue(model.requirements.ram)
        self.cores = internal_parameters.ServerValue(model.requirements.cores or 0)
        self.dns = internal_parameters.ServerValue(model.requirements.dns or ctm.DnsType.DEFAULT)
        if model.requirements.porto_layers:
            porto_layers = (
                taskbox.resources(model.requirements.porto_layers)
                if taskbox else
                model.requirements.porto_layers
            )
            self.porto_layers = internal_parameters.ServerValue(porto_layers)

        self.client_tags = internal_parameters.ServerValue(ctc.Tag.Query.__decode__(model.requirements.client_tags))

        if model.requirements.semaphores:
            # TODO: use ctm.Semaphores?
            self.semaphores = internal_parameters.ServerValue(model.requirements.semaphores.to_mongo())
        else:
            self.semaphores = internal_parameters.ServerValue(None)

        if model.requirements.ramdrive:
            self.ramdrive = internal_parameters.ServerValue(ctm.RamDrive(
                type=model.requirements.ramdrive.type,
                size=model.requirements.ramdrive.size,
                path=None,
            ))
        else:
            self.ramdrive = internal_parameters.ServerValue(None)

        if model.requirements.caches is None:
            object.__setattr__(self, "Caches", type(self).Caches(None))
        else:
            caches_value = {c.key: c.value for c in model.requirements.caches}
            # noinspection PyUnresolvedReferences
            object.__setattr__(self, "Caches", type(self).Caches(caches_value))

        if model.requirements.tasks_resource:
            tasks_resource = (
                (taskbox.resources(model.requirements.tasks_resource.id) or None)
                if taskbox else
                model.requirements.tasks_resource.id
            )
            self.tasks_resource = internal_parameters.ServerValue(tasks_resource)

        if model.requirements.container_resource:
            container_resource = (
                (taskbox.resources(model.requirements.container_resource) or None)
                if taskbox else
                model.requirements.container_resource
            )
            self.container_resource = internal_parameters.ServerValue(container_resource)

        if model.requirements.resources:
            resources = (
                taskbox.resources(model.requirements.resources)
                if taskbox else
                model.requirements.resources
            )
            self.resources = internal_parameters.ServerValue(resources)

        self.resources_space_reserve = internal_parameters.ServerValue(
            model.requirements.resources_space_reserve.to_mongo()
            if model.requirements.resources_space_reserve else
            None
        )

    def __lock__(self):
        object.__setattr__(self, "__locked__", True)

    def __setattr__(self, key, value):
        if key not in self.__requirements_names__:
            raise AttributeError("Unknown requirement {!r}".format(key))
        if self.__locked__ and not getattr(type(self), key).__output__:
            raise AttributeError("Requirements are readonly")
        super(Requirements, self).__setattr__(key, value)

    def __delattr__(self, _):
        raise AttributeError("Unsupported operation")

    def __getstate__(self):
        # noinspection PyTypeChecker
        state = {"requirements": {
            name: value << 20 if name in ("ram", "disk_space") else value
            for name, value in (
                (param_type.name, param_type.__encode__(getattr(self, param_type.name)))
                for param_type in type(self)
                if not param_type.__static__
            )
        }}
        # noinspection PyUnresolvedReferences
        state["requirements"]["caches"] = self.Caches.__getstate__()
        return state


class CachesMeta(type):
    def __new__(mcs, name, bases, namespace):
        if bases == (object,):
            # noinspection PyUnresolvedReferences
            return mcs.__base__.__new__(mcs, name, bases, namespace)

        module_name = namespace.pop("__module__", None)
        qual_name = namespace.pop("__qualname__", None)

        if bases == (Caches,) and not namespace:
            base_values = None
        else:
            base_values = copy.deepcopy(bases[0].__values__ or {})
            base_values.update(namespace)

        namespace = {
            "__module__": module_name,
            "__values__": base_values,
        }

        if qual_name is not None:
            namespace["__qualname__"] = qual_name

        cls = type.__new__(mcs, name, bases, namespace)
        return cls

    def __nonzero__(cls):
        # noinspection PyUnresolvedReferences
        return cls.__values__ is not None

    __bool__ = __nonzero__

    def __iter__(cls):
        # noinspection PyUnresolvedReferences
        return iter(() if cls.__values__ is None else cls.__values__)

    def __getstate__(cls):
        # noinspection PyUnresolvedReferences
        return copy.deepcopy(cls.__values__)


class Caches(six.with_metaclass(CachesMeta, object)):
    """
    Task caches
    @DynamicAttrs
    """
    __values__ = None
    __current_task__ = None

    def __init__(self, values=ctm.NotExists):
        if values is not ctm.NotExists:
            object.__setattr__(self, "__values__", values)

    def __nonzero__(self):
        return self.__values__ is not None

    __bool__ = __nonzero__

    def __iter__(self):
        return iter(() if self.__values__ is None else self.__values__)

    @internal_common.dual_method
    def __getstate__(self):
        return copy.deepcopy(self.__values__)

    def __setattr__(self, name, value):
        if self.__values__ is None:
            object.__setattr__(self, "__values__", {})
        self.__values__[name] = value

    def __delattr__(self, name):
        if name is Ellipsis:
            object.__setattr__(self, "__values__", None)
        elif self.__values__ is not None:
            self.__values__.pop(name, None)

    __delitem__ = __delattr__


# noinspection PyPep8Naming
class _common_property(property):
    pass


class ParametersMeta(type):
    __parameters_registry__ = weakref.WeakValueDictionary()

    def __new__(mcs, name, bases, namespace):
        if bases == (object,):
            # noinspection PyUnresolvedReferences
            return mcs.__base__.__new__(mcs, name, bases, namespace)
        cls = None

        def check_param_dup(name_, param, params):
            assert name_ not in params, "Detected parameter duplication for {!r}: {} and {}".format(
                name_, param, name_ in params and params[name_].fget(None)
            )

        if bases == (Parameters,):
            common_params = {
                key: _common_property(
                    internal_common.memoized(
                        lambda _, p=value, n=key:
                        p(name=n, __parameters__=cls)
                    )
                )
                for key, value in six.iteritems(namespace)
                if inspect.isclass(value) and issubclass(value, internal_parameters.Parameter)
            }
        else:
            common_params = {
                name: (name in namespace and prop.fget(None).cast(namespace[name]), _common_property(
                    internal_common.memoized(
                        lambda _, p=prop, v=namespace.get(name, ctm.NotExists), n=name:
                        p.fget(None)(name=n, default=v, __parameters__=cls)
                    )
                ))[-1]
                for name, prop in inspect.getmembers(mcs, lambda _: isinstance(_, _common_property))
            }
        custom_params = {}
        new_namespace = {}
        parameters_index = {}
        parameters_names = namespace.get("__names__", ())
        for param_name in common_params:
            new_namespace[param_name] = mcs.__common_property__(param_name, common_params)
        reorder = []
        non_params = set()

        for param_name, value in six.iteritems(namespace):
            if param_name in common_params:
                continue
            if inspect.isclass(value) and issubclass(value, internal_parameters.Parameter):
                assert isinstance(
                    value.description, six.string_types
                ), "Parameter '{}' does not have a label, or it is not a string".format(param_name)
                check_param_dup(param_name, value, custom_params)
                # noinspection PyTypeChecker
                new_namespace[param_name] = mcs.__property__(param_name, value)
                try:
                    parameters_index[param_name] = parameters_names.index(param_name)
                except ValueError:
                    raise ValueError(
                        "Error while processing parameter '{}', possibly wrong parameter definition".format(
                            param_name
                        )
                    )
                custom_params[param_name] = property(lambda _, p=value: p)
                if issubclass(type(value), internal_parameters.ScopeMeta):
                    parent_names = getattr(value.__base__, "names", None)
                    if parent_names:
                        for subparam_name in value.names:
                            reorder.append(
                                (subparam_name, parent_names[-1] if not reorder else reorder[-1][-1])
                            )
            # external parameters
            elif inspect.isclass(value) and issubclass(value, Parameters):
                check_param_dup(param_name, value, custom_params)
                custom_params[param_name] = property(lambda _, p=value: p)
                base_index = parameters_names.index(param_name) if parameters_names else 0
                # noinspection PyTypeChecker
                subparams = list(value)
                subnames = collections.OrderedDict()
                if subparams:
                    index_delta = 1. / len(subparams)
                    for i, subparam in enumerate(subparams):
                        subparam = subparam()
                        new_namespace[subparam.name] = mcs.__property__(subparam.name, subparam)
                        subnames[subparam.name] = subparam
                        parameters_index[subparam.name] = base_index + (i * index_delta)
                        check_param_dup(subparam.name, subparam, custom_params)
                        custom_params[subparam.name] = property(lambda _, p=subparam: p)

                # noinspection PyDefaultArgument
                def prop(pp, sn=subnames, pn=param_name):
                    def __iter__(self):
                        for name_ in sn:
                            param_type = sn[name_]
                            if param_type.__static__:
                                continue
                            yield (name_, getattr(self, name_))
                    ns = {
                        n: property(
                            lambda _, n=n: getattr(pp, n),
                            lambda _, v, n=n: setattr(pp, n, v)
                        )
                        for n in sn
                    }
                    ns.update(__iter__=__iter__)
                    return type.__new__(type(pn, (type,), ns), pn, (object,), {})
                new_namespace[param_name] = property(prop)
            else:
                non_params.add(param_name)
                new_namespace[param_name] = value
        common_parameters_names = bases[0].__common_parameters_names__ or tuple(common_params)
        custom_parameters_names = collections.OrderedDict((_, None) for _ in bases[0].__custom_parameters_names__)

        def iter_params_to_remove():
            for _ in bases[0].__custom_parameters_names__:
                if _ in namespace and namespace[_] is None:
                    param = getattr(bases[0], _)
                    if inspect.isclass(param) and issubclass(param, internal_parameters.Group):
                        # noinspection PyTypeChecker
                        _ = list(six.moves.map(lambda _: _.name, param))
                        _.append(param.name)
                    for _ in common_itertools.chain(_):
                        yield _

        # remove parameters from the common list
        for _ in iter_params_to_remove():
            custom_parameters_names.pop(_, None)

        custom_parameters_names.update(
            _
            for _ in sorted(six.iteritems(parameters_index), key=lambda _: _[1])
            if _[0] not in common_parameters_names
        )
        custom_parameters_names = list(custom_parameters_names)
        for i, (name_to_reorder, name_to_insert_after) in enumerate(reorder):
            if name_to_reorder == name_to_insert_after or name_to_insert_after not in custom_parameters_names:
                continue
            custom_parameters_names.remove(name_to_reorder)
            custom_parameters_names.insert(
                custom_parameters_names.index(name_to_insert_after) + i + 1, name_to_reorder
            )

        overriden_with_non_params = set(custom_parameters_names) & non_params
        assert not overriden_with_non_params, (
            "Cannot override parameters with non-parameter values: {}".format(list(overriden_with_non_params))
        )

        new_namespace["__common_parameters_names__"] = common_parameters_names
        new_namespace["__custom_parameters_names__"] = tuple(custom_parameters_names)
        new_namespace["__parameters_names__"] = (
            new_namespace["__common_parameters_names__"] + new_namespace["__custom_parameters_names__"]
        )

        # create new properties for removed parameters
        for _ in iter_params_to_remove():
            def getter(__):
                raise AttributeError("Parameter '{}' was deleted".format(_))
            custom_params[_] = property(getter)

        cls = type.__new__(
            type(mcs.__name__, (mcs,), dict(it.chain(six.iteritems(common_params), six.iteritems(custom_params)))),
            name, bases, new_namespace
        )
        return cls

    def __call__(cls, *args, **kwargs):
        prefix = kwargs.pop("prefix", None)
        suffix = kwargs.pop("suffix", None)
        label_substs = kwargs.pop("label_substs", None)
        assert label_substs is None or isinstance(label_substs, dict), (
            "label_substs can be instance of dict or None"
        )
        if args:
            return type.__call__(cls, *args, **kwargs)
        rename = lambda _: "".join((prefix or "", _, suffix or ""))
        frame = inspect.currentframe().f_back
        if "__names__" not in frame.f_locals:
            frame.f_locals["__names__"] = frame.f_code.co_names
        new_params = {}
        names = []
        groups = []
        # noinspection PyUnresolvedReferences
        for param_name in cls.__custom_parameters_names__:
            param = getattr(cls, param_name)
            param_kws = {}
            if label_substs is not None:
                param_kws["label"] = param.description.format(**label_substs)
            new_param_name = rename(param_name)
            param = new_params[new_param_name] = param(**param_kws)
            names.append(new_param_name)
            if issubclass(param, internal_parameters.Subfieldable) and param.sub_fields:
                # rename sub fields
                param.sub_fields = {
                    value: tuple(map(rename, sub_fields))
                    for value, sub_fields in param.sub_fields.items()
                }
            elif issubclass(param, internal_parameters.Group):
                groups.append(param)
        # rename groups internals
        for group in groups:
            group.__names__ = tuple(map(rename, group.__names__))
            group.__locals__ = {name: new_params[name] for name in group.names}
        new_params["__names__"] = names
        mcs = type(cls)
        cls = mcs.__new__(mcs, cls.__name__, (cls.__bases__[0],), new_params)
        # noinspection PyUnresolvedReferences
        frame.f_locals.setdefault("__external_names__", set()).update(cls.__custom_parameters_names__)
        return cls

    def __iter__(cls):
        # noinspection PyUnresolvedReferences
        for name in cls.__custom_parameters_names__:
            yield getattr(cls, name)

    def __setattr__(cls, *_):
        raise AttributeError("Unsupported operation")

    @staticmethod
    def __common_property__(name, params):
        def getter(self):
            value = self.__values__.get(name, ctm.NotExists)
            if value is ctm.NotExists:
                value = params[name].fget(None)().default
            if isinstance(value, lazy.Deferred):
                value = self.__values__[name] = value()
            return value

        def setter(self, value):
            if name in Parameters.__immutable__:
                raise AttributeError("`{}` is immutable".format(name))
            param = params[name].fget(None)()
            if (
                name in self.__mutable__ and
                self.__current_task__ and
                self.__current_task__() and
                not self.__current_task__().__taskbox__
            ):
                try:
                    # noinspection PyProtectedMember
                    self.__current_task__()._sdk_server.task[self.__current_task__().id].update(
                        {name: param.__encode__(value)}
                    )
                except common_errors.InvalidRESTCall:
                    pass

            self.__values__[name] = param.__decode__(
                value(param)
                if isinstance(value, internal_parameters.InternalValue) else
                param.cast(value)
            )

        return property(getter, setter)

    @classmethod
    def __property__(mcs, name, param):
        assert (
            id(param) not in mcs.__parameters_registry__ and param.__instantiated__
        ), "Cannot use parameter class without instantiation: {}".format(name)
        mcs.__parameters_registry__[id(param)] = param
        param.name = name

        def getter(self):
            value = self.__values__.get(name, ctm.NotExists)
            if value is ctm.NotExists:
                if patterns.is_classproperty(param, "default_value"):
                    return None
                value = param.default
                if not param.__output__:
                    self.__values__[name] = value
            if isinstance(value, lazy.Deferred):
                value = self.__values__[name] = value()
            return value

        def setter(self, value):
            casted_value = (
                value(param)
                if isinstance(value, internal_parameters.InternalValue) else
                param.cast(value)
            )
            normalized = safe_decode(param, name, casted_value)
            if param.__output__ and self.__current_task__ and self.__current_task__():
                if name in self.__values__ and self.__values__[name] != normalized:
                    raise AttributeError(
                        "Output parameter {!r} is already set. Current value is {!r}, given: {!r}".format(
                            name, self.__values__[name], normalized
                        )
                    )
                assert self.__current_task__() is self.__current_task__().current, \
                    "Can set output parameters for current task only"
                try:
                    self.__current_task__()._sdk_server.task.current.output.update([{
                        "name": name,
                        "value": param.__encode__(
                            value(param)
                            if isinstance(value, internal_parameters.InternalValue) else
                            value
                        ),
                        "reset_on_restart": param.__reset_on_restart__,
                    }])
                except common_errors.InvalidRESTCall:
                    pass

            self.__values__[name] = normalized
        return property(getter, setter)


class Parameters(six.with_metaclass(ParametersMeta, object)):
    # parent task
    __current_task__ = None
    # list of common parameters names sorted by order of definition
    __common_parameters_names__ = ()
    # list of custom parameters names sorted by order of definition
    __custom_parameters_names__ = ()
    # list of all parameters names sorted by order of definition
    __parameters_names__ = ()
    # input parameters are readonly
    __locked__ = False
    # common parameters that are updated immediately after being set, allow to reset parameter
    __mutable__ = ("description", "tags", "expires", "score")
    # task hints not derived from parameters values
    __explicit_hints__ = set()
    # common parameters that are set in task class and cannot be updated externally
    __immutable__ = ()
    # fields that not presents in template database model
    __unused_template_parameters__ = {"uniqueness", "release_to"}
    # map from sdk parameters to database model
    __parameters_database_map__ = {"expires": "expires_delta"}

    def __init__(self, task=None, params=None, model=None, taskbox=None):
        assert isinstance(task, Task), "`task` must be instance of `Task`"

        if params and model:
            raise RuntimeError("Can't use both json params and model")
        object.__setattr__(self, "__values__", {})

        if params:
            self._from_json(params)
        if model:
            self._from_model(model, taskbox=taskbox)

        object.__setattr__(self, "__current_task__", weakref.ref(task))

    def _from_json(self, parameters):
        for name, value in list(six.iteritems(parameters)):
            if name == "hints":
                object.__setattr__(self, "__explicit_hints__", set(str(h) for h in common_itertools.chain(value)))
                continue
            if name not in self.__parameters_names__ or value is None:
                continue
            if name in self.__immutable__:
                continue
            setattr(self, name, parameters.pop(name))

    def _from_model(self, model, taskbox=None):
        wrapper = internal_parameters.InternalValue if taskbox else internal_parameters.ServerValue
        # Set custom parameters
        if model.parameters:
            for p in it.chain(model.parameters.input, model.parameters.output):
                if p.key not in self.__parameters_names__ or p.value is None:
                    continue
                setattr(self, p.key, wrapper(p.value, True))

        self.tags = model.tags or []
        object.__setattr__(self, "__explicit_hints__", set(model.hints + model.explicit_hints))
        self.expires = wrapper(model.expires_delta)
        self.priority = wrapper(model.priority)
        self.uniqueness = wrapper({"key": model.unique_key})
        self.fail_on_any_error = wrapper(bool(model.fail_on_any_error))
        self.owner = wrapper(model.owner)
        self.hidden = wrapper(model.hidden)
        self.dump_disk_usage = wrapper(bool(model.dump_disk_usage))
        self.tcpdump_args = wrapper(model.tcpdump_args)
        self.description = wrapper(model.description)
        self.suspend_on_status = wrapper(model.suspend_on_status)
        self.push_tasks_resource = wrapper(model.push_tasks_resource)
        self.score = wrapper(model.score)

        self.max_restarts = wrapper(
            model.max_restarts
            if model.max_restarts is not None else
            config.Registry().common.task.execution.max_restarts
        )

        default_timeout = config.Registry().common.task.execution.timeout
        self.kill_timeout = wrapper(min(model.kill_timeout or default_timeout * 3600, 604800))

        self.notifications = wrapper([
            {
                "statuses": list(ctt.Status.Group.collapse(n.statuses)),
                "recipients": list(n.recipients),
                "transport": n.transport,
                "check_status": getattr(n, "check_status", None),
                "juggler_tags": list(getattr(n, "juggler_tags", []))
            }
            for n in model.notifications
        ])

    def __lock__(self):
        object.__setattr__(self, "__locked__", True)

    def __iter__(self):
        for name in self.__custom_parameters_names__:
            param_type = getattr(type(self), name)
            if param_type.__static__:
                continue
            yield (name, getattr(self, name))

    def __setattr__(self, key, value):
        if key not in self.__parameters_names__:
            raise AttributeError("Unknown parameter {!r}".format(key))
        param_type = getattr(type(self), key)
        if self.__locked__ and not param_type.__output__ and key not in self.__mutable__:
            raise AttributeError("Input parameters are readonly")
        super(Parameters, self).__setattr__(key, value)

    def __delattr__(self, _):
        raise AttributeError("Unsupported operation")

    def __getstate__(self):
        state = {
            name: param_type.__encode__(getattr(self, name))
            for name, param_type in ((name, getattr(type(self), name)) for name in self.__common_parameters_names__)
            if not param_type.__static__
        }
        state["custom_fields"] = []
        state["output"] = []
        for name, param_type in ((name, getattr(type(self), name)) for name in self.__custom_parameters_names__):
            is_output = param_type.__output__
            # noinspection PyUnresolvedReferences
            if is_output and name not in self.__values__:
                continue
            value = getattr(self, name)
            if value is not None:
                item = {"name": name, "value": safe_encode(param_type, name, value)}
                key = "output" if is_output else "custom_fields"
                state[key].append(item)
        return state

    def __gethints__(self, no_explicit=False):
        """ Get task hints based on parameters """
        hints = set() if no_explicit else self.__explicit_hints__.copy()
        for name in self.__custom_parameters_names__:
            param_type = getattr(type(self), name)
            # noinspection PyUnresolvedReferences
            if param_type.hint and (not param_type.__output__ or name in self.__values__):
                value = getattr(self, name)
                if value:
                    hints.add(str(value))
        return hints


class ContextMeta(type):
    __internal_attrs__ = None

    def __new__(mcs, name, bases, namespace):
        if bases == (object,):
            # noinspection PyUnresolvedReferences
            return mcs.__base__.__new__(mcs, name, bases, namespace)
        if not mcs.__internal_attrs__:
            mcs.__internal_attrs__ = frozenset(namespace)
            return type.__new__(mcs, name, bases, namespace)
        new_namespace = {}
        for attr_name in ("__module__", "__qualname__", "__classcell__"):
            if attr_name in namespace:
                new_namespace[attr_name] = namespace.pop(attr_name)
        overrides = set(namespace) & mcs.__internal_attrs__
        assert not overrides, "Cannot override internal attributes: {!r}".format(list(overrides))
        base_values = copy.deepcopy(bases[0].__values__ or {})
        mangle_prefix = "_{}__".format(name)
        base_values.update(
            (k[len(mangle_prefix) - 2:] if k.startswith(mangle_prefix) else k, v)
            for k, v in six.iteritems(namespace)
            if json.dumps(v)
        )
        new_namespace["__values__"] = base_values
        cls = type.__new__(mcs, name, bases, new_namespace)
        return cls

    def __iter__(cls):
        # noinspection PyUnresolvedReferences
        return six.iteritems(cls.__values__) if cls.__values__ is not None else iter(())


class Context(six.with_metaclass(ContextMeta, object)):
    """
    Task context
    @DynamicAttrs
    """
    __values__ = None
    __current_task__ = None
    __mangle_prefix__ = None

    def __init__(self, _, **kws):
        task = _
        assert isinstance(task, Task), "`task` must be instance of `Task`"
        object.__setattr__(self, "__current_task__", weakref.ref(task))
        object.__setattr__(self, "__mangle_prefix__", "_{}__".format(type(task).__name__))
        default_values = type(self).__values__
        if self.__values__ is default_values:
            object.__setattr__(self, "__values__", copy.deepcopy(default_values))
        if self.__values__ is None:
            object.__setattr__(self, "__values__", {})
        self.__values__.update(kws)

    def __unmangle_name(self, name):
        if self.__mangle_prefix__ and name.startswith(self.__mangle_prefix__):
            return name[len(self.__mangle_prefix__) - 2:]
        return name

    def __iter__(self):
        return six.iteritems(self.__values__)

    def __getattr__(self, name):
        return self.__values__.get(self.__unmangle_name(name), ctm.NotExists)

    def __setattr__(self, name, value):
        # noinspection PyUnresolvedReferences
        if name in type(type(self)).__internal_attrs__:
            raise AttributeError("Cannot override internal attribute {!r}".format(name))
        json.dumps(value)
        self.__values__[self.__unmangle_name(name)] = value

    def __delattr__(self, name):
        # noinspection PyUnresolvedReferences
        if name in type(type(self)).__internal_attrs__:
            raise AttributeError("Cannot delete internal attribute {!r}".format(name))
        return self.__values__.pop(self.__unmangle_name(name), None)

    def __getstate__(self):
        return self.__values__

    def __contains__(self, name):
        return self.__unmangle_name(name) in self.__values__

    def save(self):
        assert self.__current_task__ and self.__current_task__() is Task.current,\
            "Can save context of the current task only"
        self.__current_task__()._sdk_server.task.current.context = self.__getstate__()


class TaskMeta(internal_common.Type):
    __registry__ = {}
    __cache__ = weakref.WeakValueDictionary()
    __overridable__ = {"__module__", "__doc__", "__reports__"}
    __current = None
    __current_fixed = False

    def __new__(mcs, name, bases, namespace):
        if bases == (object,):
            # noinspection PyUnresolvedReferences
            return mcs.__base__.__new__(mcs, name, bases, namespace)

        # inherit parents' reports metadata while preserving insertion order
        reports = collections.OrderedDict()
        for dictionary in it.chain(
            (
                getattr(base, "__reports__", {})
                for base in bases
            ),
            (namespace.get("__reports__", {}),)
        ):
            for k, v in six.iteritems(dictionary):
                if k in reports and reports[k].function_name != v.function_name:
                    # a different function is overridden -> it's a new report, which has to be moved to the end
                    reports.pop(k)
                reports[k] = v
        reports_labels = {v.function_name: v.label for v in six.itervalues(reports)}

        for k, v in six.iteritems(namespace):
            if isinstance(v, internal_common.overridable):
                mcs.__overridable__.add(k)
                namespace[k] = v.method
            else:
                assert bases in ((object,), (Task,)) or k not in Task.__dict__ or k in mcs.__overridable__, (
                    "{!r} cannot be overridden".format(k)
                )

            # remove hidden reports
            if v is None and k in reports_labels:
                reports.pop(reports_labels[k])

        namespace["__reports__"] = common_collections.ImmutableAttrOrderedDict(reports)

        cls = internal_common.Type.__new__(mcs, name, bases, namespace)
        module_filename = pathlib.Path(sys.modules[cls.__module__].__file__).name.split(".")[0]
        if module_filename == "__init__" or module_filename.endswith(projects_handler.SBTASK_SUFFIX):
            assert cls.name != "TASK", "Tasks of type `TASK` are forbidden"  # SANDBOX-7444
            assert mcs.__registry__.setdefault(cls.name, cls) is cls, (
                "Task name duplicate {}: {}.{} and {}.{}".format(
                    cls.name,
                    cls.__module__, cls.__name__,
                    mcs.__registry__[cls.name].__module__, mcs.__registry__[cls.name].__name__,
                )
            )
        return cls

    def __str__(cls):
        return cls.name

    def __repr__(cls):
        return "<Task type {!r}>".format(cls.name)

    def __getitem__(cls, task_id_or_name):
        """
        Get task by id or task class by name

        :param task_id_or_name: task id or task type name
        :return: Task object or Task class
        """
        try:
            task_id = int(task_id_or_name)
        except (TypeError, ValueError):
            task_cls = type(cls).__registry__.get(task_id_or_name)
            if task_cls is None:
                old_types = projects_handler.load_project_types(reuse=True)
                task_type = old_types.get(task_id_or_name)
                if task_type:
                    return cls.wrap_type(task_type)
                raise common_errors.UnknownTaskType("Unknown task type {!r}".format(task_id_or_name))
            return task_cls
        return cls.restore(task_id=task_id)

    def __contains__(cls, type_name):
        """
        Check existence of task type
        """
        return type_name in type(cls).__registry__

    def __iter__(cls):
        """
        Iterate all task types if called on the base abstract class
        """
        registry = type(cls).__registry__
        return (_ for _ in (cls,)) if cls.name in registry else six.itervalues(registry)

    def from_model(cls, model, taskbox=None):
        task_cls = cls[model.type]
        task = task_cls.__new__(task_cls).__setstate_from_model__(model, taskbox=taskbox)
        return task

    def from_json(cls, data):
        task_cls = cls[data["type"]]
        task = task_cls.__new__(task_cls).__setstate__(data)
        return task

    def data(cls, task_id):
        try:
            return cls._sdk_server.task[task_id][:]
        except cls._sdk_server.HTTPError as ex:
            if ex.status == httplib.NOT_FOUND:
                raise common_errors.TaskNotFound("Task #{} not found".format(task_id))
            else:
                raise

    def restore(cls, data=None, task_id=None, model=None):
        """
        Restore task object from data received from REST API

        :param data: JSON
        :param task_id: task identifier
        :param model: database model to restore from
        :return: Task object
        """
        assert len(list(six.moves.filter(None, (data, task_id, model)))) == 1, "Only one agrument can be specified"
        if data:
            task_id = data["id"]
        elif model:
            task_id = model.id
        try:
            task = cls.__cache__[task_id]
            if model:
                updated = model.time.updated
            else:
                updated = data and data.get("time", {}).get("updated")
                if updated:
                    updated = aniso8601.parse_datetime(updated)
            task_updated = task.updated
            if task_updated is not None:
                task_updated = task_updated.replace(tzinfo=None)
            if updated is not None and updated.replace(tzinfo=None) != task_updated:
                raise KeyError
        except KeyError:
            if model:
                task = cls.from_model(model)
            else:
                task = cls.from_json(data or cls.data(task_id))
            cls.__cache__[task_id] = task
        return task

    @staticmethod
    def wrap_type(task_type):
        old_type = task_type.cls.type
        class_dict = {"name": old_type}
        from sandbox import sdk2
        return type(old_type, (sdk2.OldTaskWrapper,), class_dict)

    @property
    def type(cls):
        return cls.name

    @property
    def description(cls):
        try:
            return cls.__description__
        except AttributeError:
            pass
        doc = encoding.force_unicode_safe(cls.__doc__ or "SDK2 task")

        html = getattr(cls, "__htmldoc__", None)
        if html is None:
            html = format.rst2html(textwrap.dedent(doc).strip()) if doc else ""
        # docutils converter spent about 3 seconds to convert all tasks' descriptions into HTML,
        # so cache them globally.
        # noinspection PyAttributeOutsideInit
        cls.__description__ = html
        return cls.__description__

    @description.setter
    def description(cls, value):
        # noinspection PyAttributeOutsideInit
        cls.__description__ = value

    @property
    def owners(cls):
        if hasattr(cls, "__owners__"):
            return cls.__owners__
        module_path = cls.__module__.replace(".", os.sep)
        owners_file_path = os.path.join(
            config.Registry().client.tasks.code_dir,
            "projects",
            ctm.EnvironmentFiles.OWNERS_STORAGE
        )
        owners_storage = misc.SingleFileStorage(owners_file_path, autoload=True)
        # noinspection PyAttributeOutsideInit
        cls.__owners__ = owners_storage[module_path]
        return cls.__owners__

    @owners.setter
    def owners(cls, val):
        # noinspection PyAttributeOutsideInit
        cls.__owners__ = val

    @property
    def server(cls):
        """
        :rtype: sandbox.common.rest.Client
        """
        return rest.DispatchedClient()

    @property
    def _sdk_server(cls):
        """
        REST client pointing to the latest API version. It should only be used by SDK internals.
        :rtype: sandbox.common.rest.Client
        """
        return rest.DispatchedClient(version=SDK_API_VERSION)

    @property
    @internal_common.memoized
    def current(cls):
        """
        Returns current task object or `None` in case of the code is running outside of Sandbox environment
        :rtype: sandbox.sdk2.Task or None
        """
        mcs = type(cls)
        if mcs.__current is not None or mcs.__current_fixed:
            return cls.__current
        # noinspection PyProtectedMember
        if not cls._sdk_server._external_auth:
            return
        mcs.__current_fixed = True
        try:
            task_data = cls._sdk_server.task.current[:]
        except cls._sdk_server.HTTPError as ex:
            if ex.status == httplib.FORBIDDEN:
                return
            raise
        task_data["full_context"] = task_data.get("context", {})
        mcs.__current = cls.restore(task_data)
        return mcs.__current

    @current.setter
    def current(cls, task):
        mcs = type(cls)
        if mcs.__current is not None or mcs.__current_fixed:
            raise AttributeError("This property is immutable")

        mcs.__current = task
        mcs.__current_fixed = True

    def adapt_query_constraints(cls, constraints):
        input_parameters = constraints.get("input_parameters", {})
        for key in list(six.iterkeys(input_parameters)):
            # noinspection PyUnresolvedReferences
            if key in cls.Parameters.__common_parameters_names__:
                constraints[key] = input_parameters.pop(key)

        description = constraints.pop("description", None)
        if description:
            constraints["desc_re"] = description

        tid = constraints.get("id")
        if tid:
            constraints["id"] = ",".join(six.moves.map(str, common_itertools.chain(tid)))

        parent = constraints.get("parent")
        if parent is not None:
            assert isinstance(parent, Task), "`parent` must be instance of `Task`"
            constraints["parent"] = parent.id

        task_type = constraints.pop("task_type", None)
        if task_type is not None:
            assert inspect.isclass(task_type) and issubclass(task_type, Task), \
                "`task_type` must be subclass of `Task`"
            constraints["type"] = task_type.name

        required_fields = {
            "type", "author", "owner", "parent", "requirements", "description",
            "status", "max_restarts", "priority", "se_tag", "notifications", "time",
            "input_parameters", "output_parameters", "tags", "hints", "scheduler"
        }
        extra_fields = constraints.pop("extra_fields", None)
        required_fields.update(extra_fields or set())
        constraints["fields"] = ",".join(required_fields)

    def _restore_adapter(cls, data=None, task_id=None, model=None):
        if data is not None:
            data["full_context"] = data.get("context")
        return cls.restore(data=data, task_id=task_id, model=model)

    def find(cls, task_type=None, **constraints):
        """
        Find tasks according to specified constraints

        :param task_type: find tasks of the type, if None then find tasks of type of current class
        :param constraints: query constraints, according to :py:class:`sandbox.web.api.v1.task.TaskList.Get`
                            with the following exceptions:
                            `description` instead of `desc_re`,
                            `parent` contains :py:class:`~sandbox.sdk2.task.Task` object,
                            `id` contains task id or list of ids
        :rtype: sandbox.sdk2.internal.common.Query
        """
        constraints.setdefault("task_type", task_type or (cls if cls.name in cls else None))
        cls.adapt_query_constraints(constraints)
        return internal_common.Query(cls._sdk_server.task, cls._restore_adapter, **constraints)


class Task(six.with_metaclass(TaskMeta, object)):
    """
    Base task class
    @DynamicAttrs
    """
    __taskbox__ = None

    def __init__(self, parent, __requirements__=None, **parameters):
        """
        Create new task or subtask
        """
        assert parent is None or parent is parent.current, "`parent` must be None or current task object"
        self.__parent = parent
        self.__current = None
        self.__agentr = None
        self.__log_resource = None
        self.__ramdrive = None
        self.__scheduler = ctm.NotExists

        __requirements__ = dict(__requirements__ or {})

        if parent:
            parameters.setdefault("description", parent.Parameters.description)
            parameters.setdefault("owner", parent.Parameters.owner)
            parameters.setdefault("priority", parent.Parameters.priority)

            from sandbox import sdk2

            if (
                parent.Parameters.push_tasks_resource and
                isinstance(parent.Requirements.tasks_resource, sdk2.service_resources.SandboxTasksBinary) and
                "tasks_resource" not in __requirements__
            ):
                __requirements__["tasks_resource"] = parent.Requirements.tasks_resource.id
                parameters.setdefault("push_tasks_resource", parent.Parameters.push_tasks_resource)

        req = self.Requirements()
        for attr, value in six.iteritems(__requirements__):
            setattr(req, attr, value)

        provided = set(parameters.keys())
        self.Parameters = self.Parameters(self, parameters)
        self.Context = self.Context(self, **parameters)

        data = self.Parameters.__getstate__()

        if __requirements__:
            data["requirements"] = {}
            all_requirements = req.__getstate__()["requirements"]
            # Only send requirements provided by user
            for attr in __requirements__:
                data["requirements"][attr] = all_requirements[attr]

        # this returns all parameters, we want to send only the ones that were provided by user
        data["custom_fields"] = [kv for kv in data["custom_fields"] if kv["name"] in provided]
        # set hints provided by the user
        if self.Parameters.__explicit_hints__:
            data["hints"] = list(self.Parameters.__explicit_hints__)

        response = self._sdk_server.task(
            type=type(self).name,
            context=self.Context.__getstate__(),
            children=self.__parent is not None,
            **data
        )

        # noinspection PyUnresolvedReferences
        self.__set_internal_fields(response)
        self.Requirements = self.Requirements(reqs=response["requirements"])

        type(self).__cache__[self.__id] = self

    def __repr__(self):
        # noinspection PyUnresolvedReferences
        return "<Task {}:{}>".format(type(self).name, self.id)

    def __setattr__(self, key, value):
        assert key not in (
            Requirements.__name__, Parameters.__name__, Context.__name__
        ) or getattr(self, key) is getattr(type(self), key)
        return super(Task, self).__setattr__(key, value)

    def __set_internal_fields(self, state):
        parent_id = (state.get("parent") or {}).get("id")
        # noinspection PyUnresolvedReferences
        self.__parent = parent_id or None
        self.__current = None
        self.__agentr = None
        self.__log_resource = None
        self.__ramdrive = None
        self.__id = state["id"]
        self.__status = state["status"]
        self.__author = state["author"]
        self.__info = state.get("results", {}).get("info") or ""
        time_ = state.get("time", {})
        _ = time_.get("created")
        self.__created = aniso8601.parse_datetime(_) if _ else None
        _ = time_.get("updated")
        self.__updated = aniso8601.parse_datetime(_) if _ else None
        _ = state.get("scheduler")
        self.__scheduler = _["id"] if _ else None

    def __lock_requirements_and_parameters(self):
        if self.status != ctt.Status.DRAFT:
            self.Parameters.__lock__()
        if self.status != ctt.Status.ENQUEUING and not ctt.Status.can_switch(self.status, ctt.Status.ENQUEUING):
            self.Requirements.__lock__()

    def __setstate__(self, state):
        # noinspection PyUnresolvedReferences
        assert state["type"] == type(self).name
        self.__set_internal_fields(state)

        # setup requirements
        self.Requirements = self.Requirements(state.get("requirements"))

        from sandbox import sdk2
        is_sdk2 = not isinstance(self, sdk2.OldTaskWrapper)

        parameters = {}

        if is_sdk2 and "input_parameters" in state:
            parameters.update(
                (name, (None if value is None else internal_parameters.InternalValue(value)))
                for name, value in it.chain(
                    six.iteritems(state.get("input_parameters", {})),
                    six.iteritems(state.get("output_parameters", {}))
                )
            )

        # extract common parameters from state
        for name in self.Parameters.__common_parameters_names__:
            if not getattr(self.Parameters, name).__static__:
                value = state.get(name)
                parameters[name] = None if value is None else internal_parameters.InternalValue(value)

        self.Parameters = self.Parameters(self, parameters)

        if "hints" in state:
            param_hints = self.Parameters.__gethints__(no_explicit=True)
            self.Parameters.__explicit_hints__.update(h for h in state["hints"] if h not in param_hints)

        # setup context
        context = state.get("full_context")
        if context is None:
            context_cls = self.Context
            self.Context = lazy.DeferredProxy(
                lambda: context_cls(self, **self._sdk_server.task[self.id].context[:])
            )
        else:
            self.Context = self.Context(self, **(context or {}))

        self.__lock_requirements_and_parameters()
        return self

    def __setstate_from_model__(self, model, taskbox=None):
        assert model.type == type(self).name

        self.__current = self
        self.__agentr = None
        self.__log_resource = None
        self.__ramdrive = None
        self.__scheduler = getattr(model, "scheduler", None)

        self.__id = model.id
        self.__parent = getattr(model, "parent_id", None)
        self.__status = model.execution.status if hasattr(model, "execution") else ctt.Status.DRAFT
        self.__author = model.author
        self.__info = model.description or ""
        self.__created = model.time.created
        self.__updated = model.time.updated

        self.Requirements = self.Requirements(model=model, taskbox=taskbox)
        self.Context = self.Context(self, **pickle.loads(model.context))

        from sandbox import sdk2
        if isinstance(self, sdk2.OldTaskWrapper):
            self.Parameters = self.Parameters(self, {})
        else:
            self.Parameters = self.Parameters(self, model=model, taskbox=taskbox)

        self.__lock_requirements_and_parameters()
        self.__taskbox__ = taskbox
        return self

    def __int__(self):
        return self.id

    def __getstate__(self):
        return self.id

    def __release_info(self, author, comments):
        """
        Construct plain text release form

        :param author: releaser login
        :param comments: comments to release
        :return: info about release (author, comments, resources etc)
        :rtype: str
        """
        if not comments:
            comments = " "
        info_string = "Author: {}\nSandbox task: {}/task/{}/view\n".format(author, common_urls.server_url(), self.id)
        resource_string = "Resources:\n"
        from sandbox import sdk2
        for resource in sdk2.Resource.find(task=self, state=ctr.State.READY).limit(0):
            if resource.type.releasable:
                b = " * {} - id:{} [{}]".format(resource.type, resource.id, resource.arch or ctm.OSFamily.ANY)
                binary_info = "{}:\n     File name:         {}, {}KB (md5: {})".format(
                    b,
                    resource.path.name,
                    resource.size,
                    resource.md5
                )
                padding = "   "
                sandbox_info = "{}  Resource page:     {}".format(padding, resource.url)
                http_info = "{}  HTTP download URL: {}".format(padding, resource.http_proxy)
                skynet_info = "{}  Skynet ID:         {}".format(padding, resource.skynet_id or "N/A")

                resource_string = "{}{}\n\n".format(
                    resource_string,
                    "\n".join([binary_info, sandbox_info, http_info, skynet_info])
                )
        sign_string = "--\nsandbox \n https://sandbox.yandex-team.ru/ \n"
        return "\n".join([info_string, comments, resource_string, sign_string])

    def _send_release_info_to_email(self, parameters):
        author = parameters["releaser"]
        release_status = parameters["release_status"]
        message_subject = parameters.get("release_subject")
        message_body = parameters.get("release_comments")
        release_to = parameters["email_notifications"]["to"]
        release_cc = parameters["email_notifications"]["cc"]
        if not release_to:
            release_to, release_cc = release_cc, release_to
            if not release_to:
                logging.info("No recipients specified for the release.")
                return

        mail_subj = "[{}] {}".format(release_status, message_subject) if release_status else message_subject
        mail_body = self.__release_info(author, message_body)
        # noinspection PyBroadException
        try:
            self._sdk_server.notification(
                subject=mail_subj,
                body=mail_body,
                recipients=release_to,
                transport=ctn.Transport.EMAIL,
                type=ctn.Type.TEXT,
                charset=ctn.Charset.UTF,
                task_id=self.id,
                view=ctn.View.RELEASE_REPORT
            )
        except Exception:
            logging.exception("")

    @property
    def server(self):
        """
        :rtype: sandbox.common.rest.Client
        """
        return rest.DispatchedClient()

    @property
    def _sdk_server(self):
        """
        REST client pointing to the latest API version. It should only be used by SDK internals.
        :rtype: sandbox.common.rest.Client
        """
        return rest.DispatchedClient(version=SDK_API_VERSION)

    @property
    def agentr(self):
        """
        :rtype: sandbox.agentr.client.Session
        """
        assert self.__agentr, "Agentr instance is not defined"
        return self.__agentr

    @agentr.setter
    def agentr(self, value):
        from sandbox.agentr import client
        assert isinstance(value, client.Session), (
            "Value {!r} must be an instance of {}.{}".format(value, client.Session.__module__, client.Session.__name__)
        )
        # if self.__agentr is not None:
        #     raise AttributeError("This property is immutable")
        self.__agentr = value

    @property
    def current(self):
        """
        :rtype: None
        """
        if self.__current is None:
            self.__current = type(self).current
        return self.__current

    @current.setter
    def current(self, task):
        if self.__current is not None:
            raise AttributeError("This property is immutable")
        self.__current = task

    @property
    def log_resource(self):
        return self.__log_resource

    @log_resource.setter
    def log_resource(self, resource):
        if self.__log_resource is not None:
            raise AttributeError("This property is immutable")
        self.__log_resource = resource

    def log_path(self, *path):
        return self.path(six.ensure_str(self.agentr.logdir), *path)

    @internal_common.dual_method
    def find(self, task_type=None, **constraints):
        assert "parent" not in constraints, "`parent` cannot be specified for this method, use `<TaskClass>.find()`"
        if task_type is not None:
            constraints["task_type"] = task_type
        constraints["parent"] = self
        constraints["children"] = True
        cls = type(self)
        cls.adapt_query_constraints(constraints)
        query = internal_common.Query(self._sdk_server.task, cls._restore_adapter, **constraints)
        return query.limit(0)

    @internal_common.Query.SortedField
    def id(self):
        return self.__id

    @internal_common.Query.SortedField
    def type(self):
        return type(self)

    @internal_common.Query.SortedField
    def author(self):
        return self.__author

    @internal_common.Query.SortedField
    def owner(self):
        return self.Parameters.owner

    @internal_common.Query.SortedField
    def status(self):
        return self.__status

    @internal_common.Query.SortedField
    def parent(self):
        if isinstance(self.__parent, int):
            self.__parent = type(self)[self.__parent]
            self.__parent.__current = self.__current
        return self.__parent

    @internal_common.Query.SortedField
    def host(self):
        return self.Requirements.host

    @internal_common.Query.SortedField
    def hidden(self):
        return self.Parameters.hidden

    @internal_common.Query.SortedField
    def created(self):
        return self.__created

    @internal_common.Query.SortedField
    def updated(self):
        return self.__updated

    @property
    def info(self):
        return self.__info

    @property
    def ramdrive(self):
        return self.__ramdrive

    @ramdrive.setter
    def ramdrive(self, value):
        if self.__ramdrive is not None:
            raise AttributeError("This property is immutable")
        from sandbox import sdk2
        self.__ramdrive = sdk2.parameters.RamDrive.cast(value)

    @property
    def scheduler(self):
        return self.__scheduler

    @scheduler.setter
    def scheduler(self, value):
        if self.__scheduler is not ctm.NotExists:
            raise AttributeError("This property is immutable")
        self.__scheduler = value

    def reload(self):
        """ Reload task info from API """
        try:
            del self.Requirements
        except AttributeError:
            pass
        try:
            del self.Parameters
        except AttributeError:
            pass
        try:
            del self.Context
        except AttributeError:
            pass
        return self.__setstate__(type(self).data(self.__id))


class VaultMeta(type):
    @ft.partial(property, None)
    def vault_key(cls, key):
        # noinspection PyUnresolvedReferences
        assert not cls._decrypt.__func__.__defaults__[-1], "Cannot modify vault_key"
        # noinspection PyUnresolvedReferences
        cls._decrypt.__func__.__defaults__ = cls._decrypt.__func__.__defaults__ + (key,)


class Vault(six.with_metaclass(VaultMeta, object)):
    pass
