from __future__ import absolute_import

import re
import json

import six
from six.moves.urllib import parse as urlparse

from sandbox.common import rest as common_rest
from sandbox.common import errors as common_errors
from sandbox.common import patterns
from sandbox.common import platform as common_platform
from sandbox.common import itertools as common_itertools

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr

from .internal import parameters
from .vcs import svn


# Global variable to store object of current task. It is set by executor before start of task.
current_task = None


def _resource_server():
    from sandbox import sdk2
    rest = common_rest.Path(sdk2.task.Task._sdk_server.copy(version=2))
    rest <<= common_rest.Client.HEADERS({ctm.HTTPHeader.NO_LINKS: "1"})
    return rest


class SandboxParameter(object):
    UI = parameters.Parameter.UI

    name = None
    description = None
    required = False
    do_not_copy = False
    resource_type = None
    group = None
    ui = UI()
    dummy = False
    default_value = None

    @classmethod
    def cast(cls, value):
        return value

    @classmethod
    def get_custom_parameters(cls):
        """ Additional parameters to transfer to GUI """
        return None

    @classmethod
    def _get_contexted_attribute(cls, attr_name, context_name):
        attr_value = getattr(cls, attr_name, None)
        return cls.ui.context.get(context_name) if (cls.ui and attr_value is None) else attr_value


class SandboxStringParameter(SandboxParameter):

    default_value = ""
    multiline = False
    choices = None
    sub_fields = None
    ui = SandboxParameter.UI(ctt.ParameterType.STRING)

    @classmethod
    def cast(cls, value):
        if value is None:
            return None if cls.required else u""
        elif isinstance(value, six.string_types):
            return six.ensure_text(value).strip()
        else:
            return six.text_type(value).strip()

    @classmethod
    def get_custom_parameters(cls):
        result = {}
        if cls.sub_fields and isinstance(cls.sub_fields, dict):
            result["sub_fields"] = cls.sub_fields
        choices = cls._get_contexted_attribute("choices", "values")
        if choices is not None:
            result["values"] = choices
        return result


class SandboxInfoParameter(SandboxParameter):

    ui = SandboxParameter.UI(ctt.ParameterType.BLOCK)
    dummy = True
    collapse = None

    # noinspection PyMethodParameters
    @patterns.classproperty
    def name(cls):
        return cls.__name__.lower()


class SandboxIntegerParameter(SandboxParameter):

    ui = SandboxParameter.UI(ctt.ParameterType.INTEGER)
    default_value = 0
    sub_fields = None

    @classmethod
    def cast(cls, value):
        return value if value is None else int(value)


class SandboxFloatParameter(SandboxParameter):

    ui = SandboxParameter.UI(ctt.ParameterType.FLOAT)
    default_value = 0.0

    @classmethod
    def cast(cls, value):
        return value if value is None else float(value)


class SandboxBoolParameter(SandboxParameter):

    ui = SandboxParameter.UI(ctt.ParameterType.BOOLEAN)
    default_value = False
    sub_fields = None
    hidden = False

    @classmethod
    def get_custom_parameters(cls):
        if cls.sub_fields and isinstance(cls.sub_fields, dict):
            return {"sub_fields": cls.sub_fields}
        return None


class SandboxBoolGroupParameter(SandboxParameter):

    choices = None
    description = "Check group:"
    ui = SandboxParameter.UI(ctt.ParameterType.MULTISELECT, {"checkboxes": True})

    @classmethod
    def cast(cls, value):
        return (
            " ".join(filter(None, (_.strip() for _ in value)))
            if isinstance(value, (list, tuple)) else
            value
        )

    @classmethod
    def get_custom_parameters(cls):
        result = {}
        choices = cls._get_contexted_attribute("choices", "values")
        if choices is not None:
            result["values"] = choices
        return result


class SandboxUrlParameter(SandboxParameter):

    ui = SandboxParameter.UI(ctt.ParameterType.STRING)

    @classmethod
    def cast(cls, value):
        if not value:
            return None if cls.required else ""
        urlparse.urlparse(value)
        return value


class SandboxSvnUrlParameter(SandboxUrlParameter):

    name = "svn_url"
    description = "Svn url:"
    ui = SandboxParameter.UI(ctt.ParameterType.STRING)
    required = True

    @classmethod
    def cast(cls, value):
        value = super(SandboxSvnUrlParameter, cls).cast(value)
        if not value:
            return None if cls.required else ""
        # noinspection PyProtectedMember
        value = svn.Arcadia._get_rw_url(value)
        return value


class SandboxArcadiaUrlParameter(SandboxSvnUrlParameter):

    name = "checkout_arcadia_from_url"
    description = "Svn url for arcadia:"
    ui = SandboxParameter.UI(ctt.ParameterType.STRING)
    required = True

    HG_REQUEST_TIMEOUT = 10

    # noinspection PyMethodParameters
    @patterns.classproperty
    def default_value(cls):
        return svn.Arcadia.trunk_url()

    @classmethod
    def cast(cls, value):
        value = value.strip() if value else value  # value could be string or None
        if value and value.startswith(svn.Arcadia.ARCADIA_HG_SCHEME):
            parsed_url = urlparse.urlparse(value)
            if parsed_url.scheme != svn.Arcadia.ARCADIA_HG_SCHEME:
                raise ValueError("Invalid Arcadia Hg URL")
            # TODO: SANDBOX-6069
        elif value and value.startswith(svn.Arcadia.ARCADIA_ARC_SCHEME):
            parsed_url = urlparse.urlparse(value)
            if parsed_url.scheme != svn.Arcadia.ARCADIA_ARC_SCHEME:
                raise ValueError("Invalid Arcadia Arc URL")
        else:
            value = super(SandboxArcadiaUrlParameter, cls).cast(value)
        if value:
            value = svn.Arcadia.normalize_url(value)
        return value


class SandboxRadioParameter(SandboxParameter):

    ui = SandboxParameter.UI(ctt.ParameterType.RADIO, context={"values": []})
    # List of pairs (title, value) for radio buttons.
    choices = None
    # Amount of lines to show the buttons.
    per_line = 1
    # A dictionary with value-dependent fields specification.
    sub_fields = None

    @classmethod
    def cast(cls, value):
        if not value:
            return "" if not cls.required else None
        if not any(value == _[1] for _ in cls._get_contexted_attribute("choices", "values")):
            raise ValueError("Value {!r} is not in list of the parameter class values".format(value))
        return value

    @classmethod
    def get_custom_parameters(cls):
        ret = {
            "per_line": cls.per_line,
            "values": cls._get_contexted_attribute("choices", "values"),
        }
        if cls.sub_fields and isinstance(cls.sub_fields, dict):
            ret["sub_fields"] = cls.sub_fields
        return ret


class TaskSelector(SandboxParameter):
    ui = SandboxParameter.UI(ctt.ParameterType.TASK)
    task_type = None

    @classmethod
    def cast(cls, value):
        value = value and int(value)
        if value:
            from sandbox import sdk2
            try:
                sdk2.Task[value]
            except common_errors.TaskNotFound:
                raise TypeError("Unknown task: #{}".format(value))
        return value

    @classmethod
    def get_custom_parameters(cls):
        return {
            "type": ",".join(common_itertools.chain(cls.task_type or ())),
            "status": str(ctt.Status.Group.FINISH),
            "limit": 20,
        }


class ResourceSelector(SandboxParameter):

    multiple = None
    resource_type = None
    register_dependency = True
    ui = SandboxParameter.UI(ctt.ParameterType.RESOURCE, context={"multiple": False, "states": [ctr.State.READY]})
    state = None
    attrs = {}

    @classmethod
    def _get_resource_types(cls):
        rt = cls._get_contexted_attribute("resource_type", "types")
        if isinstance(rt, (list, tuple)):
            return list(six.moves.map(str, rt))
        return [str(rt)] if rt else []

    @classmethod
    def cast(cls, value):
        value = super(ResourceSelector, cls).cast(value)

        multiple_value = cls._get_contexted_attribute("multiple", "multiple")
        if multiple_value:
            value = re.split(r"[, ]+", value) if isinstance(value, six.string_types) else value
        else:
            value = next(iter(common_itertools.chain(value)), None)
            if isinstance(value, (list, tuple)):
                if not value:
                    raise TypeError("Empty list supplied instead of a single resource")
                raise TypeError("This parameter does not accept multiple values")
        resource_types = set(cls._get_resource_types())
        if value:
            from sandbox import sdk2
            try:
                ids = map(int, common_itertools.chain(value) if multiple_value else [value])
            except (TypeError, ValueError):
                raise ValueError("Invalid resource id(s): {!r}".format(value))

            cached_resources, ids = sdk2.Resource.restore_cached(ids)
            not_found = set(ids)

            resources = [{"id": res.id, "type": res.type} for res in cached_resources]
            if ids:
                resources.extend(_resource_server().resource.read(id=ids, order="id", limit=len(ids))["items"])

            for resource in resources:
                if resource_types:
                    try:
                        resource_cls = sdk2.Resource[resource["type"]]
                    except common_errors.UnknownResourceType:
                        resource_parents = {resource["type"]}
                    else:
                        resource_parents = set(
                            six.moves.map(
                                str,
                                six.moves.filter(lambda _: issubclass(_, sdk2.Resource), resource_cls.__mro__)
                            )
                        )
                    if not (resource_parents & resource_types):
                        raise TypeError(
                            "Resource #{} has type {!s}, but must be {!s} or its subclass".format(
                                resource["id"],
                                resource["type"],
                                (
                                    next(iter(resource_types))
                                    if len(resource_types) == 1 else
                                    "one of {}".format(resource_types)
                                ),
                            )
                        )
                not_found.discard(resource["id"])
            if not_found:
                ids = ["#{}".format(_) for _ in sorted(not_found)]
                raise TypeError("Unknown resources: {}".format(", ".join(ids)))
        return value

    @classmethod
    def get_custom_parameters(cls):
        result = {
            "multiple": bool(cls._get_contexted_attribute("multiple", "multiple")),
            "states": list(common_itertools.chain(cls._get_contexted_attribute("state", "states"))),
            "types": cls._get_resource_types(),
        }
        if cls.attrs:
            result["attrs"] = json.dumps(cls.attrs)
        return result


class LastReleasedResource(ResourceSelector):
    """ Resource selector parameter defaults to latest released resource of given type. """

    required = True
    owner = None

    # noinspection PyMethodParameters
    @patterns.classproperty
    def default_value(cls):
        resource_types = cls._get_resource_types()
        if len(resource_types) != 1:
            return None
        from sandbox import sdk2
        try:
            kws = {"resource_type": resource_types[0], "limit": 1}
            if cls.owner is not None:
                kws["owner"] = cls.owner
            releases = sdk2.task.Task._sdk_server.release.read(**kws)["items"]
            task_id = releases[0]["task_id"]
            resources = _resource_server().resource.read(
                task_id=task_id, type=resource_types[0], limit=1
            )["items"]
            return resources[0]["id"]
        except LookupError:
            return None


class Container(LastReleasedResource):
    """
    Container selector for the task - the task will be executed in container
    which is packaged in the target resource. In case of container will be specified for the task,
    the "platform" selector will take no effect anymore. But "platform" parameter is very important here:
    there's should be "SANDBOX_DEPS" resource for that platform. Also, the latest released resource with that
    platform will be used by default.
    """
    ui = SandboxParameter.UI(ctt.ParameterType.CONTAINER)
    required = True
    name = "_container"
    platform = "linux_ubuntu_12.04_precise"
    resource_type = "LXC_CONTAINER"
    multiple = False
    state = [ctr.State.READY]

    # noinspection PyMethodParameters
    @patterns.classproperty
    def attrs(cls):
        return {"released": ctt.ReleaseStatus.STABLE, "platform": cls.platform}

    # noinspection PyMethodParameters
    @patterns.classproperty
    def default_value(cls):
        try:
            kws = {
                "type": cls.resource_type,
                "state": cls.state,
                "attrs": cls.attrs,
                "limit": 1,
            }
            if cls.owner is not None:
                kws["owner"] = cls.owner
            return _resource_server().resource.read(**kws)["items"][0]["id"]
        except LookupError:
            return None

    @classmethod
    def cast(cls, value):
        value = super(Container, cls).cast(value)
        if value:
            # noinspection PyTypeChecker
            res_id = int(value)
            resource = _resource_server().resource[res_id].read()
            platform = resource["attributes"].get("platform")
            if not platform:
                raise TypeError("Attribute 'platform' is not defined for resource #{}".format(res_id))
            if "linux" not in common_platform.get_platform_alias(platform):
                raise TypeError("Unknown linux platform {!r} of resource #{}".format(platform, res_id))
        return value
