import logging

import six

from sandbox.agentr import errors as agentr_errors

from . import path as sdk_path
from . import task as sdk_task


def mount_overlay(mount_point, lower_dirs, upper_dir="", work_dir=""):
    """
    Mounts OverlayFS to mount_point using lower_dirs, upper_dir and work_dir as options of `mount`-command.
    upper_dir and work_dir have to be in task's working directory or in mounted tmpfs.

    :param mount_point: Directory for merged filesystem.
    :param lower_dirs: List of paths of readonly layers or singular path.
    :param upper_dir: Path to directory where the changes will be saved.
    :param work_dir: Empty directory on the same filesystem as upper_dir. It's required for proper upper_dir work.
    :return sdk2.path.Path: path to mounted filesystem.
    """

    if issubclass(type(lower_dirs), six.string_types):
        lower_dirs = [lower_dirs]
    lower_dirs = list(six.moves.map(str, lower_dirs))
    mount_point, upper_dir, work_dir = map(str, (mount_point, upper_dir, work_dir))

    for path in (mount_point, upper_dir, work_dir):
        sdk_path.Path(path).mkdir(0o755, parents=True, exist_ok=True)

    agentr = sdk_task.Task.current.agentr
    return sdk_path.Path(agentr.mount_overlay(mount_point, lower_dirs, upper_dir, work_dir))


def umount(mount_point):
    """
    Unmounts mounted earlier mount_point.

    :param mount_point: Directory to unmount.
    """
    sdk_task.Task.current.agentr.umount(mount_point)


def _switch_turbo_boost(enable):
    try:
        sdk_task.Task.current.agentr.turbo_boost(enable)
        return True
    except agentr_errors.InvalidPlatform as er:
        logging.error("Impossible to %sable turbo boost: %s", "en" if enable else "dis", er)
        return False


def enable_turbo_boost():
    """
    Enables turbo boost on host
    :return bool: True on success
    """
    return _switch_turbo_boost(True)


def disable_turbo_boost():
    """
    Disables turbo boost on host
    :return bool: True on success
    """
    return _switch_turbo_boost(False)
