from __future__ import absolute_import

import re
import copy
import json
import datetime as dt
import distutils.util

import six

from sandbox.common import yav
from sandbox.common import patterns
from sandbox.common import itertools as common_itertools

import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr

from . import legacy as sdk1_parameters
from . import internal


class Integer(internal.parameters.Integer, sdk1_parameters.SandboxIntegerParameter):
    """ Integer parameter

    :rtype: `int` or `None`
    """

    # avoid IDE warning
    def __int__(self):
        return 0


class Float(internal.parameters.Parameter, sdk1_parameters.SandboxFloatParameter):
    """ Float parameter

    :rtype: `float` or `None`
    """

    # avoid IDE warning
    def __float__(self):
        return .0

    @classmethod
    def __decode__(cls, value, raise_errors=False):
        value = super(Float, cls).__decode__(value)
        try:
            if value is not None:
                value = float(value)
        except (ValueError, TypeError):
            if raise_errors:
                raise
        return value


class String(internal.parameters.String, sdk1_parameters.SandboxStringParameter):
    """ String parameter

    :rtype: `str` or None
    """


class JSON(internal.parameters.Parameter, sdk1_parameters.SandboxParameter):
    """ JSON structure with special representation in UI.

    :rtype: `dict`, `list` or `None`
    """
    ui = internal.parameters.Parameter.UI(ctt.ParameterType.STRING, {"format": "json"})
    multiline = True
    required = False
    dummy = False
    do_not_copy = False
    default_value = None
    __complex_type__ = True

    @classmethod
    def cast(cls, value):
        if value and isinstance(value, six.string_types):
            json.loads(value)
        else:
            json.dumps(value)
        return value


class Staff(String):
    """ Parameter for staff usernames

    :rtype: str or None
    """
    ui = internal.parameters.Parameter.UI(ctt.ParameterType.STRING, {"format": "staff"})


class Vault(String):
    """ Parameter for vault data. Accepts the following values:

    - `sdk2.VaultItem`
    - sequence of at least 1 element ([owner], name, ... )
    - string that is either "name" or "owner:name"

    :rtype: `sdk2.VaultItem`
    """
    @classmethod
    def cast(cls, value):
        decoded_value = cls.__decode__(value)
        if value and not decoded_value:
            raise ValueError("Vault key value in unrecognized format")
        return decoded_value

    @classmethod
    def __decode__(cls, value):
        from sandbox import sdk2

        value = super(Vault, cls).__decode__(value)
        if not value or isinstance(value, sdk2.VaultItem):
            return value
        if isinstance(value, six.string_types):
            return sdk2.VaultItem(*value.split(":", 1))
        if hasattr(value, "__iter__"):
            return sdk2.VaultItem(*value)

    @classmethod
    def __encode__(cls, value):
        return value if value is None else str(cls.cast(value))


class YavSecret(String):
    """
    Parameter for Yav secret uuid and optional version and default key

    :rtype: `sdk2.yav.Secret` or None
    """

    default_value = None
    ui = internal.parameters.Parameter.UI(ctt.ParameterType.STRING, {"format": "yav"})

    @classmethod
    def cast(cls, value, raise_errors=True):
        decoded_value = cls.__decode__(value)

        if value and not decoded_value:
            raise ValueError("Yav secret value is not recognized")

        if not decoded_value:
            return None

        if raise_errors:
            # validate secret and version
            secret = decoded_value.secret
            yav.Secret.create(secret.secret_uuid, secret.version_uuid, secret.default_key)

        return decoded_value

    @classmethod
    def __decode__(cls, value):
        from sandbox import sdk2

        value = super(YavSecret, cls).__decode__(value)

        if isinstance(value, sdk2.yav.Secret):
            return value

        if not value or not isinstance(value, six.string_types):
            return None

        return sdk2.yav.Secret.__decode__(value)

    @classmethod
    def __encode__(cls, value):
        return str(cls.cast(value, raise_errors=False)) if value else None


class YavSecretWithKey(YavSecret):
    """
    Parameter for YavSecret with required default_key argument
    """

    @classmethod
    def cast(cls, value, raise_errors=True):
        # type: (Union[String, sdk2.yav.Secret], bool) -> Optional[sdk2.yav.Secret]
        from sandbox import sdk2

        decoded_value = super(YavSecretWithKey, cls).cast(value, raise_errors)

        if isinstance(decoded_value, sdk2.yav.Secret) and not decoded_value.default_key:
            raise ValueError("Key value was not found in secret data")

        return decoded_value


class StrictString(String):
    """ String parameter validated by regexp

    :rtype: `str` or None
    """
    regexp = None

    @classmethod
    def cast(cls, value):
        if cls.regexp is None:
            raise ValueError("regexp argument required")

        if value is None:
            ret_val = None if cls.required else u""
        elif isinstance(value, six.string_types):
            ret_val = six.ensure_text(value)
        else:
            ret_val = six.text_type(value)

        if ret_val is not None and not re.match(cls.regexp, ret_val):
            raise ValueError("'{}' doesn't match '{}'".format(value, getattr(cls.regexp, "pattern", cls.regexp)))

        return ret_val


class Info(internal.parameters.Parameter, sdk1_parameters.SandboxInfoParameter):
    """ Used to declare a block of parameters. """


class Bool(internal.parameters.Bool, sdk1_parameters.SandboxBoolParameter):
    """ Boolean parameter

    :rtype: `bool`
    """
    @classmethod
    def cast(cls, value):
        return bool(distutils.util.strtobool(str(value)))


class Resource(internal.parameters.Resource, sdk1_parameters.ResourceSelector):
    """ Resource parameter

    :rtype: `sdk2.Resource` or `list[sdk2.Resource]` or `None`
    """
    @classmethod
    def cast(cls, value):
        return super(Resource, cls).cast(
            value
            if cls._get_contexted_attribute("multiple", "multiple") else
            [value]
        )


class LastResource(Resource):
    """ Last resource """

    state = ctr.State.READY
    owner = None

    # noinspection PyMethodParameters
    @patterns.classproperty
    def default_value(cls):
        assert cls.resource_type, "resource_type is required"
        try:
            kws = {"type": cls.resource_type, "state": cls.state, "limit": 1}
            if cls.owner:
                kws["owner"] = cls.owner
            if cls.attrs:
                kws["attrs"] = cls.attrs
            return sdk1_parameters._resource_server().resource.read(**kws)["items"][0]["id"]
        except LookupError:
            return None


class LastReleasedResource(Resource, sdk1_parameters.LastReleasedResource):
    """ Resource parameter defaults to latest released resource of given type.

    This parameter type is deprecated. Use sdk2.Resource with custom default_value property instead.

    :rtype: `sdk2.Resource` or `None`
    """


class Container(LastReleasedResource, sdk1_parameters.Container):
    """ Container parameter. If present, the resource is used as LXC container for the task.

    :rtype: `sdk2.Resource` or `None`
    """


class ParentResource(Resource):
    """
    This parameter allows a parent task to pass a NOT_READY resource into subtask,
    which then can finish the resource.

    :rtype: `sdk2.Resource` or `None`
    """
    ui = None
    do_not_copy = True


class Task(internal.parameters.Task, sdk1_parameters.TaskSelector):
    """ Task parameter

    :rtype: `sdk2.Task` or `None`
    """


class Url(internal.parameters.Parameter, sdk1_parameters.SandboxUrlParameter):
    """ URL parameter

    :rtype: `str` or `None`
    """


class SvnUrl(Url, sdk1_parameters.SandboxSvnUrlParameter):
    """ SVN URL parameter

    :rtype: `str` or `None`
    """


class ArcadiaUrl(SvnUrl, sdk1_parameters.SandboxArcadiaUrlParameter):
    """ Arcadia URL parameter

    :rtype: `str` or `None`
    """


class CheckGroup(internal.parameters.CheckGroup, sdk1_parameters.SandboxBoolGroupParameter):
    """ A group of checkboxes.

    :Example:

    .. code-block:: python

            with sdk2.parameters.CheckGroup("Just a multiselect field") as multiselect:
                multiselect.values.option1 = "Option One"
                multiselect.values.option2 = multiselect.Value("Option Two", checked=True)
                multiselect.values.option3 = "Option Three"

    :rtype: `list[str]`
    """

    default_value = ()

    @classmethod
    def cast(cls, value):
        return (
            list(six.moves.filter(None, value))
            if isinstance(value, (list, tuple)) else
            (value.split() if value is not None else None)
        )


class RadioGroup(internal.parameters.RadioGroup, sdk1_parameters.SandboxRadioParameter):
    """ Radio button parameter (select one of mutually exclusive options)

    :Example:

    .. code-block:: python

            with sdk2.parameters.RadioGroup("Just a radio field group") as radio_group:
                radio_group.values.value1 = None
                radio_group.values.value2 = radio_group.Value(default=True)
                radio_group.values.value3 = None

    :rtype: `str`
    """


class Repeater(internal.parameters.Repeater):
    supported_types = (String, Integer, Float, Bool, Url)
    repetition = ""
    default_type = String
    default_value = None


class List(Repeater):
    """ List parameter

    :rtype: `list[str]`
    """

    repetition = "list"

    @classmethod
    def cast(cls, value, cast=None):
        assert cast
        if value is None:
            return copy.copy(cls.default) or []
        return list(six.moves.map(cast, common_itertools.chain(value)))


class Dict(Repeater):
    """ Dict parameter

    :rtype: `dict[str, str]`
    """

    __complex_type__ = True
    repetition = "dict"

    @classmethod
    def cast(cls, value, cast=None):
        assert cast
        if value is None:
            return copy.copy(cls.default) or {}
        if isinstance(value, six.string_types):
            value = json.loads(value)
        elif not hasattr(value, "__iter__"):
            raise TypeError("Value {!r} should be a dictionary or a list of key-value pairs".format(value))
        ret = {}
        items = (
            six.iteritems(value)
            if isinstance(value, dict) else
            (value if isinstance(value[0], (list, tuple)) else [value])
        )
        for k, v in items:
            k = String.cast(k)
            if not k:
                raise ValueError("Key cannot be empty")
            ret[k] = cast(v)
        return ret


class Group(internal.parameters.Group, sdk1_parameters.SandboxInfoParameter):
    """ A group of parameters (used as context manager)"""


class Output(internal.parameters.Output):
    """ A group of output parameters (used as context manager)"""


# Special parameters


class Priority(internal.parameters.Parameter):
    """ Priority of a task

    :rtype: `sandbox.common.types.task.Priority`
    """

    ui = None
    dummy = True
    required = True
    default_value = None

    @classmethod
    def cast(cls, value):
        return cls.__decode__(value, raise_errors=True)

    @classmethod
    def __decode__(cls, value, raise_errors=False):
        value = super(Priority, cls).__decode__(value)
        try:
            return ctt.Priority.make(value)
        except (ValueError, TypeError):
            if raise_errors:
                raise


class Notifications(internal.parameters.Parameter):
    """ Task notifications

    :rtype: `list[sdk2.Notification]` or `None`
    """

    ui = None
    dummy = True
    required = False
    default_value = None
    __complex_type__ = True

    @classmethod
    def cast(cls, value):
        return cls.__decode__(value, raise_errors=True)

    @classmethod
    def __decode__(cls, value, raise_errors=False):
        from sandbox import sdk2
        value = super(Notifications, cls).__decode__(value)
        try:
            return [
                nt if isinstance(nt, sdk2.Notification) else sdk2.Notification(**nt)
                for nt in common_itertools.chain(value)
            ] if value else value
        except (ValueError, TypeError):
            if raise_errors:
                raise

    @classmethod
    def __encode__(cls, value):
        return [
            {
                "statuses": nt.statuses, "recipients": nt.recipients,
                "transport": nt.transport, "check_status": getattr(nt, "check_status", None),
                "juggler_tags": getattr(nt, "juggler_tags", [])
            }
            for nt in value
        ] if value else value


class ClientTags(internal.parameters.Parameter, sdk1_parameters.SandboxStringParameter):
    """ Client tags of a task. Defaults to `ctc.Tag.GENERIC`.
    WARNING: this parameter type is internal. For input parameters use `CustomClientTags` instead.

    :rtype: `sandbox.common.types.client.Tag.Query`
    """

    default_value = ctc.Tag.GENERIC

    @classmethod
    def cast(cls, value):
        value = super(ClientTags, cls).cast(value)
        # TODO: SANDBOX-6479
        return ctc.Tag.Query(value or cls.default)

    @classmethod
    def __decode__(cls, value, raise_errors=False):
        value = super(ClientTags, cls).__decode__(value)
        if not value:
            return value
        return ctc.Tag.Query.__decode__(value, raise_errors)

    @classmethod
    def __encode__(cls, value):
        return str(value) if value else value


class CustomClientTags(ClientTags):
    """ Client tags expression parameter

    :rtype: `sandbox.common.types.client.Tag.Query` or None
    """
    default_value = None

    @classmethod
    def cast(cls, value):
        value = super(ClientTags, cls).cast(value)
        return ctc.Tag.Query(value) if value else value


# TODO
class Environments(internal.parameters.Parameter):
    ui = None
    dummy = True
    __static__ = True
    required = False
    default_value = ()

    @classmethod
    def cast(cls, value):
        return value


class RamDrive(internal.parameters.Parameter):
    """ Task ramdrive requirement

    :rtype: `sandbox.common.types.misc.RamDrive` or `None`
    """
    ui = None
    dummy = False
    required = False
    default_value = None
    __complex_type__ = True

    @classmethod
    def cast(cls, value):
        if value is not None and not isinstance(value, (dict, ctm.RamDrive)):
            raise ValueError("Wrong value: {!r}".format(value))
        return cls.__decode__(value)

    @classmethod
    def __decode__(cls, value):
        value = super(RamDrive, cls).__decode__(value)
        # noinspection PyProtectedMember
        return (
            ctm.RamDrive(*((None,) * len(ctm.RamDrive._fields)))._replace(**value)
            if isinstance(value, dict) else
            value
        )

    @classmethod
    def __encode__(cls, value):
        return dict(type=value.type, size=value.size) if value else value


class DnsType(String):
    """ DNS setting for a task

    :rtype: `sandbox.common.types.misc.DnsType`
    """

    ui = None
    required = False
    default_value = ctm.DnsType.DEFAULT
    choices = list((t, t) for t in ctm.DnsType)

    @classmethod
    def cast(cls, value):
        if value not in ctm.DnsType:
            raise ValueError("Wrong value: {!r}".format(value))
        return value


class ReleaseTo(internal.parameters.Parameter):
    ui = None
    dummy = True
    __static__ = True
    required = False
    default_value = None

    @classmethod
    def cast(cls, value):
        return list(six.moves.map(str, common_itertools.chain(value)))


class Semaphores(internal.parameters.Parameter):
    """Semaphores used by task

    :rtype: `sandbox.common.types.task.Semaphores`
    """

    ui = None
    dummy = False
    required = False
    default_value = None
    __complex_type__ = True

    @classmethod
    def cast(cls, value):
        if not value:
            return cls.default_value
        return cls.__decode__(value, raise_errors=True)

    @classmethod
    def __decode__(cls, value, raise_errors=False):
        value = super(Semaphores, cls).__decode__(value)
        try:
            return ctt.Semaphores(**value) if isinstance(value, dict) else ctt.Semaphores(*value)
        except (ValueError, TypeError):
            if raise_errors:
                raise

    @classmethod
    def __encode__(cls, value):
        return value.to_dict() if value else None

    @staticmethod
    def release():
        from sandbox import sdk2
        del sdk2.Task._sdk_server.task.current.semaphores


class TaskTag(String):
    """ Task tags

    :rtype: `list[str]`
    """

    @classmethod
    def cast(cls, value):
        ctt.TaskTag.test(value)
        return value


class Timedelta(Integer):
    """ Timedelta parameter. Allows to assign datetime, in which case it's converted to timedelta from now.

    :rtype: `dt.timedelta`
    """
    default_value = None

    @classmethod
    def cast(cls, value):
        if isinstance(value, six.integer_types + six.string_types):
            return dt.timedelta(seconds=int(value))
        elif isinstance(value, dt.timedelta):
            return value
        elif isinstance(value, dt.datetime):
            return dt.datetime.utcnow() - value

    @classmethod
    def __encode__(cls, value):
        if isinstance(value, six.integer_types):
            return value
        elif isinstance(value, dt.timedelta):
            return int(value.total_seconds())
        elif isinstance(value, dt.datetime):
            return int((dt.datetime.utcnow() - value).total_seconds())


class ResourcesSpaceReserveValue(dict):
    """
    Dict with bucket name as keys and size in bytes as values
    Examples:
        {"sandbox-469": 1073741824}  # reserve 1GiB in bucket sandbox-469
        {"default": 44040192}  # reserve 42MiB for default bucket for task owner
    """

    DEFAULT_BUCKET = "default"

    def __init__(self, *args, **kws):
        super(ResourcesSpaceReserveValue, self).__init__(*args, **kws)
        for bucket, size in self.items():
            if not isinstance(bucket, six.string_types):
                raise ValueError("bucket must be string or None")
            if not (isinstance(size, six.integer_types) and size >= 0):
                raise ValueError("value must be non negative integer")

    def not_implemented(self, *args, **kws):
        raise NotImplementedError

    for method in ("__setitem__", "__delitem__", "update", "clear", "pop", "popitem", "setdefault"):
        locals()[method] = not_implemented
    del locals()["method"]
    del locals()["not_implemented"]


class ResourcesSpaceReserve(internal.parameters.Parameter):
    """
    Space to reserve in MDS buckets for resources the task going to create

    :rtype: `ResourcesSpaceReserveValue`
    """

    ui = None
    dummy = False
    required = False
    default_value = None
    __complex_type__ = True

    @classmethod
    def cast(cls, value):
        if value is None:
            return cls.default_value
        return cls.__decode__(value, raise_errors=True)

    @classmethod
    def __decode__(cls, value, raise_errors=False):
        value = super(ResourcesSpaceReserve, cls).__decode__(value)
        try:
            if isinstance(value, six.integer_types):
                return ResourcesSpaceReserveValue({ResourcesSpaceReserveValue.DEFAULT_BUCKET: value})
            elif isinstance(value, list):
                ResourcesSpaceReserveValue((item["bucket"], item["size"]) for item in value)
            return ResourcesSpaceReserveValue(value)
        except (ValueError, TypeError):
            if raise_errors:
                raise

    @classmethod
    def __encode__(cls, value):
        return (
            [
                {"bucket": bucket, "size": size}
                for bucket, size in value.items()
            ]
            if value else
            None
        )
