from __future__ import absolute_import

import os
import stat
import shutil
import logging
import itertools

import six

from sandbox.common import fs


logger = logging.getLogger(__name__)


def get_logs_folder():
    """
    Get path to directory with logs from logger.
    If current logger has no `logging.FileHandler`, return current working directory.
    Else return path to directory of first `logging.FileHandler` with file name 'common.log'
    """
    for log_handler in logging.getLogger().handlers:
        if isinstance(log_handler, logging.FileHandler):
            main_log_file_path = log_handler.baseFilename
            if os.path.basename(main_log_file_path) == "common.log":
                return os.path.dirname(os.path.abspath(log_handler.baseFilename))
    return os.getcwd()


# noinspection PyShadowingBuiltins
def list_dir(path, filter=None, files_only=False, folders_only=False, abs_path=False):
    """
    Get list of files and directories for given path
    """
    result = []
    path = os.path.abspath(path)
    if filter:
        filter = str(filter)
    logger.info("List dir %s with filter %s", path, filter)
    all_folder_items = os.listdir(path)
    for item in all_folder_items:
        if filter:
            if filter not in item:
                continue
        item_path = os.path.join(path, item)
        result_item = None
        if files_only:
            if os.path.isfile(item_path):
                result_item = item
        if folders_only:
            if os.path.isdir(item_path):
                result_item = item
        if not files_only and not folders_only:
            result_item = item
        if result_item:
            if abs_path:
                result.append(item_path)
            else:
                result.append(item)
    return result


def copytree3(src, dst, symlinks=False, ignore=None, copy_function=shutil.copy2, ignore_dangling_symlinks=False):
    """Recursively copy a directory tree.

    The copytree3 is a port of shutil.copytree function from python-3.2.
    It has additional useful parameters and may be helpful while we are
    on python-2.x.It has to be removed as soon as sandbox is moved to
    python-3.2 or higher.

    The destination directory must not already exist.
    If exception(s) occur, an Error is raised with a list of reasons.

    If the optional symlinks flag is true, symbolic links in the
    source tree result in symbolic links in the destination tree; if
    it is false, the contents of the files pointed to by symbolic
    links are copied. If the file pointed by the symlink doesn't
    exist, an exception will be added in the list of errors raised in
    an Error exception at the end of the copy process.

    You can set the optional ignore_dangling_symlinks flag to true if you
    want to silence this exception. Notice that this has no effect on
    platforms that don't support os.symlink.

    The optional ignore argument is a callable. If given, it
    is called with the `src` parameter, which is the directory
    being visited by copytree3(), and `names` which is the list of
    `src` contents, as returned by os.listdir():

        callable(src, names) -> ignored_names

    Since copytree3() is called recursively, the callable will be
    called once for each directory that is copied. It returns a
    list of names relative to the `src` directory that should
    not be copied.

    The optional copy_function argument is a callable that will be used
    to copy each file. It will be called with the source path and the
    destination path as arguments. By default, copy2() is used, but any
    function that supports the same signature (like copy()) can be used.

    """
    names = os.listdir(src)
    if ignore is not None:
        ignored_names = ignore(src, names)
    else:
        ignored_names = set()

    os.makedirs(dst)
    errors = []
    for name in names:
        if name in ignored_names:
            continue
        srcname = os.path.join(src, name)
        dstname = os.path.join(dst, name)
        try:
            if os.path.islink(srcname):
                linkto = os.readlink(srcname)
                if symlinks:
                    # We can't just leave it to `copy_function` because legacy
                    # code with a custom `copy_function` may rely on copytree3
                    # doing the right thing.
                    os.symlink(linkto, dstname)
                else:
                    # ignore dangling symlink if the flag is on
                    if not os.path.exists(linkto) and ignore_dangling_symlinks:
                        continue
                    # otherwise let the copy occurs. copy2 will raise an error
                    copy_function(srcname, dstname)
            elif os.path.isdir(srcname):
                copytree3(srcname, dstname, symlinks, ignore, copy_function)
            else:
                # Will raise a SpecialFileError for unsupported file types
                copy_function(srcname, dstname)
        # catch the Error from the recursive copytree3 so that we can
        # continue with other files
        except shutil.Error as err:
            errors.extend(err.args[0])
        except EnvironmentError as why:
            errors.append((srcname, dstname, six.binary_type(why)))
    try:
        shutil.copystat(src, dst)
    except OSError as why:
        if WindowsError is not None and isinstance(why, WindowsError):
            # Copying file access times may fail on Windows
            pass
        else:
            errors.extend((src, dst, six.binary_type(why)))
    if errors:
        raise shutil.Error(errors)


def copy_path(source, destination, copy_function=shutil.copy2, symlinks=False):
    """
    Copy file or directory from `source` to `destination`.
    If `source` is a directory and `copy_function` passed,
    it will be used for copying of each file of the directory,
    instead of default one used by `copy2`

    :return: path to the copied directory
    """
    source = os.path.abspath(source)
    destination = os.path.abspath(destination)
    logger.info("Copy %s to %s", source, destination)
    dir_ = make_folder(os.path.dirname(destination))
    if os.path.isfile(source):
        copy_function(source, destination)
    else:
        # copy as a directory
        copytree3(source, destination, copy_function=copy_function, symlinks=symlinks)

    return dir_


remove_path = fs.remove_path
make_folder = fs.make_folder
get_unique_file_name = fs.get_unique_file_name


def chmod(path, mode, recursively=True):
    """
    Change permissions for the path

    :param path: path to file or directory
    :param mode: permissions to set, for example 0o755
    :type mode: int
    :param recursively: whether recursively change permissions for directory
    :return: passed path
    """
    path = os.path.abspath(path)
    logger.info('Chmod for %s to 0%o', path, mode)
    os.chmod(path, mode)
    for dirpath, dirnames, filenames in os.walk(path):
        for fname in itertools.chain(filenames, dirnames):
            os.chmod(os.path.join(dirpath, fname), mode)
        if not recursively:
            del dirnames[:]
    return path


def add_write_permissions_for_path(path, recursively=True):
    """
    Add permissions on write for path

    :param path: path to file or directory
    :param recursively: whether recursively change permissions for directory
    :return: passed path
    """
    path = os.path.abspath(path)
    mask = stat.S_IWUSR | stat.S_IWGRP | stat.S_IWOTH
    return chmod(path, stat.S_IMODE(os.stat(path).st_mode) | mask, recursively)


def remove_write_permissions_for_path(path, recursively=True):
    """
    Remove permissions on write for path

    :param path: path to file or directory
    :param recursively: whether recursively change permissions for directory
    :return: passed path
    """
    path = os.path.abspath(path)
    mode = stat.S_IMODE(os.stat(path).st_mode)
    mode &= ~(stat.S_IWUSR | stat.S_IWGRP | stat.S_IWOTH)
    return chmod(path, mode, recursively)


def add_read_permissions_for_path(path, recursively=True):
    """
    Add permissions on read for path

    :param path: path to file or directory
    :param recursively: whether recursively change permissions for directory
    :return: passed path
    """
    path = os.path.abspath(path)
    curent_path_permissions = stat.S_IMODE(os.stat(path).st_mode)
    new_permissions = stat.S_IRUSR | stat.S_IRGRP | stat.S_IROTH | curent_path_permissions
    return chmod(path, new_permissions, recursively)


def add_executable_permissions_for_path(path, recursively=True):
    """
    Add permissions on execute for path

    :param path: path to file or directory
    :param recursively: whether recursively change permissions for directory
    :return: passed path
    """
    path = os.path.abspath(path)
    curent_path_permissions = stat.S_IMODE(os.stat(path).st_mode)
    new_permissions = stat.S_IXUSR | stat.S_IXGRP | stat.S_IXOTH | curent_path_permissions
    return chmod(path, new_permissions, recursively)


def which(program):
    """
    Check whether program is in the PATH
    Implementation from https://github.com/amoffat/pbs/blob/master/pbs.py

    :return: string with program name if it is included in the system path
    """
    is_exe = lambda _: os.path.exists(_) and os.access(_, os.X_OK)
    fpath, _ = os.path.split(program)
    if fpath:
        if is_exe(program):
            return program
    else:
        for path in os.environ["PATH"].split(os.pathsep):
            exe_file = os.path.join(path, program)
            if is_exe(exe_file):
                return exe_file
    return None
