from __future__ import absolute_import

import struct
import logging
import collections

from sandbox.common import rest

import sandbox.agentr.client


class Phazotron(collections.namedtuple("PhazotronBase", ("script", "socket"))):
    # Binary representation of data block size for socket transfer.
    SIZE_ST = struct.Struct("!I")
    # Binary representation of return code for socket transfer.
    RET_ST = struct.Struct("!b")
    # Script template.
    SCRIPT_TMPL = None

    # noinspection PyInitNewSignature
    def __new__(cls, phazotron):
        return super(Phazotron, cls).__new__(cls, phazotron.script(cls), phazotron.sock_name)

    def __str__(self):
        return self.script


class Synchrophazotron(Phazotron):
    """ Client's information about syncrophazotron based on AgentR location (script, socket path and session token). """
    SCRIPT_TMPL = """
    #!{executable}
    import os
    import sys
    import json
    import logging

    import six
    if six.PY2:
        import pathlib2 as pathlib
    else:
        import pathlib

    sys.path = ["/skynet", os.path.dirname({code_root!r}), {code_root!r}] + sys.path
    os.environ[{config_env_var!r}] = {config_path!r}

    import sandbox.agentr.client


    a = sandbox.agentr.client.Session("{session_token}", {iteration}, 0, logging.getLogger())
    if sys.argv[1] == "-":
        data = json.load(sys.stdin)
        rid = a.resource_register(
            *map(data.get, ("path", "type", "description", "arch", "attributes"))
        )["id"]
        print(six.binary_type(a.resource_complete(rid)["id"]))
        sys.exit(0)

    if sys.argv[1] == "mount_overlay":
        data = json.load(sys.stdin)
        lower_dirs = data["lower_dirs"]
        if issubclass(type(lower_dirs), six.string_types):
            lower_dirs = [lower_dirs]
        lower_dirs = list(six.moves.map(str, lower_dirs))
        mount_point, upper_dir, work_dir = map(
            str, (data["mount_point"], data.get("upper_dir", ""), data.get("work_dir", ""))
        )
        for path in (mount_point, upper_dir, work_dir):
            pathlib.Path(path).mkdir(0o755, parents=True, exist_ok=True)
        print(a.mount_overlay(mount_point, lower_dirs, upper_dir, work_dir))
        sys.exit(0)

    if sys.argv[1] == "umount_overlay":
        mount_point = sys.argv[2]
        a.umount(mount_point)
        sys.exit(0)

    if sys.argv[1] == "mount_image":
        image_path = sys.argv[2]
        try:
            dirname = sys.argv[3] or None
        except IndexError:
            dirname = None

        print(a.mount_image(image_path, dirname))
        sys.exit(0)

    try:
        rid = int(sys.argv[1])
    except (IndexError, TypeError, ValueError) as ex:
        print("Error reading resource ID argument: {{}}".format(ex))
        sys.exit(-1)
    print(a.resource_sync(rid))
    """

    @staticmethod
    def sync_resource(resource_id):
        """ Synchronize the resource via synchrophazotron. """
        # noinspection PyProtectedMember
        return sandbox.agentr.client.Session(
            rest.Client._external_auth.task_token, None, 0, logging.getLogger()
        ).resource_sync(resource_id)


class Arcaphazotron(Phazotron):
    """ Client's information about arcophazotron location (script and and socket path). """
    SCRIPT_TMPL = """
    #!{executable}
    import os
    import sys
    import json
    import struct
    import socket
    import argparse
    import itertools as it
    import functools as ft
    import distutils.spawn

    import six

    size_st = struct.Struct("{size_st}")
    ret_st = struct.Struct("{ret_st}")


    class Option(object):
        def __init__(self, *args, **kws):
            self.__args = args
            self.__kws = kws

        def add(self, parser):
            parser.add_argument(*self.__args, **self.__kws)


    def _call(method, args, opts):
        kws = {{opt: val for opt, val in ((opt, getattr(args, opt)) for opt in opts) if val is not None}}
        sock = socket.socket(socket.AF_UNIX)
        sock.connect("{sockname}")
        sock.sendall("".join(it.chain.from_iterable((
            (size_st.pack(len(_)), _)
            for _ in (method, json.dumps(()), json.dumps(kws))
        ))))
        raw = sock.recv(ret_st.size)
        rc = ret_st.unpack(raw)[0]
        msg = []
        while True:
            chunk = sock.recv(0xFFF)
            if not chunk:
                break
            msg.append(chunk)
        print("".join(msg))
        sys.exit(rc)

    options = {{
        "depth": Option("--depth", dest="depth", action="store"),
        "set_depth": Option("--set-depth", dest="set_depth", action="store"),
        "revision": Option("-r", "--revision", dest="revision", action="store"),
        "ignore_externals": Option("--ignore-externals", dest="ignore_externals", action="store_true"),
        "parents": Option("--parents", dest="parents", action="store_true"),
        "url": Option("url"),
        "path": Option("path")
    }}

    commands = dict(it.chain.from_iterable(
        six.moves.map(lambda _: it.product(*_), (
            (cmds, (opts,))
            for cmds, opts in six.iteritems({{
                ("co", "checkout"): (
                    ft.partial(_call, "Arcadia.checkout"),
                    ("depth", "revision", "url", "path")
                ),
                ("up", "update"): (
                    ft.partial(_call, "Arcadia.update"),
                    ("depth", "set_depth", "revision", "ignore_externals", "parents", "path")
                ),
                ("export",): (
                    ft.partial(_call, "Arcadia.export"),
                    ("depth", "revision", "url", "path")
                )
            }})
        ))
    ))

    parser = argparse.ArgumentParser(description="Arcaphazotron")
    subparsers = parser.add_subparsers(metavar="<command>")

    for name, (func, opts) in six.iteritems(commands):
        subparser = subparsers.add_parser(name)
        subparser.set_defaults(func=func)
        for opt in opts:
            options[opt].add(subparser)

    # FIXME: Temporary proxy __all__ requests to subversion binary
    if 0 and len(sys.argv) > 1 and sys.argv[1] in commands:
        args = parser.parse_known_args()[0]
        args.func(args, commands[sys.argv[1]][1])
    else:
        svn_binary = distutils.spawn.find_executable("svn")
        if not svn_binary:
            print("svn binary not found")
            sys.exit(1)
        os.execv(svn_binary, sys.argv)
    """
