from __future__ import absolute_import

import math

import six

from sandbox.common import patterns as common_patterns
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr

from . import internal
from . import parameters

__all__ = ("Resource", "ServiceResource", "Attributes", "ResourceData")


class Attributes(internal.resource.Attributes):
    """ Namespace, contains allowed types for resource attributes """
    String = parameters.String(default=None)
    Bool = parameters.Bool(default=None)
    Integer = parameters.Integer(default=None)
    Float = parameters.Float(default=None)

    class RestartPolicy(parameters.String):
        __static__ = True

        @classmethod
        def cast(cls, value):
            if value not in ctr.RestartPolicy:
                # noinspection PyTypeChecker
                raise ValueError("Value must be one of {!r}".format(list(iter(ctr.RestartPolicy))))
            return value

    class Released(parameters.String):
        @classmethod
        def cast(cls, value):
            if value not in ctt.ReleaseStatus:
                # noinspection PyTypeChecker
                raise ValueError("Value must be one of {!r}".format(list(iter(ctt.ReleaseStatus))))
            return value

    class TTL(parameters.Float):
        MAX_VALUE = 999

        @classmethod
        def cast(cls, value):
            value = super(Attributes.TTL, cls).cast(value)
            if not value or not (math.isinf(value) or 0 < value <= cls.MAX_VALUE):
                raise ValueError("Value should be either 'inf' or a positive number < {}".format(cls.MAX_VALUE + 1))
            if not math.isinf(value):
                value = int(value)
            return value

    class MdsPolicy(parameters.String):
        @classmethod
        def cast(cls, value):
            if value not in ctt.ReleaseStatus:
                # noinspection PyTypeChecker
                raise ValueError("Value must be one of {!r}".format(list(iter(ctt.ReleaseStatus))))
            return value


class Resource(internal.resource.Resource):
    """
    An abstract base for any Sandbox resource type, which provides defaults and allowed options.

    @DynamicAttrs
    """
    __abstract__ = True
    __default_attribute__ = parameters.String

    # FIXME: added for backward compatibility, remove after all its usage will be removed
    OnRestart = ctr.RestartPolicy

    def __new__(cls, task, description, path, arch=None, system_attributes=None, **attrs):
        """
        Create new resource
        """
        return super(Resource, cls).__new__(cls, task, description, path, arch, system_attributes, **attrs)

    @internal.common.dual_method
    def find(self, **constraints):
        """
        Find resources according to specified constraints

        :param constraints: query constraints, according to :py:class:`sandbox.web.api.v1.resource.ResourceList.Get`
                            with the following exceptions:
                            `resource_type` instead of `type`, contains resource class, by default is the current,
                            `id` contains resource id or list of ids,
                            `task` instead of `task_id`, contains task id or list of ids,
                            `state` contains resource state or list of states,
                            `dependant` contains task object
        :rtype: sandbox.sdk2.internal.common.Query
        """
        return type(self).find(**constraints)

    #: Resource id (RO)
    id = internal.resource.Resource.id
    #: Type (RO)
    type = internal.resource.Resource.type
    #: Data path (RO)
    path = internal.resource.Resource.path
    #: Description (RW)
    description = internal.resource.Resource.__dict__["description"]
    #: Owner (RO)
    owner = internal.resource.Resource.owner
    #: Task id (RO)
    task_id = internal.resource.Resource.task_id
    #: Task object (RO)
    task = internal.resource.Resource.task
    #: State (READY, NOT_READY, etc.) (RO)
    state = internal.resource.Resource.state
    #: Architecture of data (RO)
    arch = internal.resource.Resource.arch
    #: Creation time (datetime.datetime) (RO)
    created = internal.resource.Resource.created
    #: Last access time (datetime.datetime) (RO)
    accessed = internal.resource.Resource.accessed
    #: Last update time (datetime.datetime) (RO)
    updated = internal.resource.Resource.updated
    #: Expiration time (datetime.datetime) (RO)
    expires = internal.resource.Resource.expires
    #: Fake field, only for sorting (RO)
    client = internal.resource.Resource.client
    #: Data size (RO)
    size = internal.resource.Resource.size
    #: Skynet id (RO)
    skynet_id = internal.resource.Resource.skynet_id
    #: MD5 sum (RO)
    md5 = internal.resource.Resource.md5
    #: Rights of current user to modify resource (RO)
    rights = internal.resource.Resource.rights
    #: Resource url in UI (RO)
    url = internal.resource.Resource.url
    #: Http link to resource data (RO)
    http_proxy = internal.resource.Resource.http_proxy
    #: Resource system attributes (RO)
    system_attributes = internal.resource.Resource.system_attributes

    # for backward compatibility
    @classmethod
    def to_dict(cls):
        result = {
            "name": six.binary_type(cls),
            "releaseable": cls.releaseable,
            "releasable": cls.releasable,
            "any_arch": cls.any_arch,
            "releasers": cls.releasers,
            "executable": cls.executable,
            "auto_backup": cls.auto_backup,
            "calc_md5": cls.calc_md5,
            "on_restart": cls.restart_policy,
            "release_subscribers": cls.release_subscribers,
            "share": cls.share,
            "description": cls.description,
        }
        return result

    releaseable = False

    @common_patterns.classproperty
    def releasable(cls):
        return cls.releaseable

    any_arch = True
    executable = False
    auto_backup = False
    calc_md5 = True
    deprecated = False
    releasers = None
    release_subscribers = None
    share = True
    shard = False
    service = False  # FIXME: deprecated

    restart_policy = Attributes.RestartPolicy("Restart policy", default=ctr.RestartPolicy.RESET, required=False)
    ttl = Attributes.TTL("Time to live", default=None, required=False)
    released = Attributes.Released("Release type", default=None, required=False)
    backup_task = Attributes.String("Backup task", required=False)
    ttl_on_release = Attributes.TTL("TTL to set to resource after its release", default=None, required=False)
    mds = Attributes.String("Link to MDS", required=False)
    sync_upload_to_mds = Attributes.Bool("Upload to MDS synchronously", required=False)
    from_task = Attributes.Integer("From task", required=False)
    pack_tar = Attributes.Integer("Pack file or directory to tar (1 - tar, 2 - tgz)")


class ServiceResource(Resource):
    """
    For internal use by Sandbox team
    """
    __abstract__ = True
    service = True


class AbstractResource(Resource):
    """
    This base class has no validation for attribute declaration - it is allowed to set any attribute on instance of it.
    """
    __abstract__ = True
    __default_attribute__ = parameters.String


class ResourceData(internal.resource.ResourceData):
    """ The class aggregates methods for resource data manipulation """

    def __init__(self, resource, fastbone=None):
        """
        Create resource data object and synchronize it locally

        :param resource: Resource object which data should be synchronized
        :param fastbone: If true or `None` (default defined by AgentR), use fastbone network, backbone otherwise
        """
        super(ResourceData, self).__init__(resource, fastbone)

    @property
    def path(self):
        """
        Resource data path
        :rtype: sandbox.sdk2.Path
        """
        return super(ResourceData, self).path

    def ready(self):
        """ Mark resource as ready """
        return super(ResourceData, self).ready()

    def broken(self):
        """ Mark resource as broken """
        return super(ResourceData, self).broken()

    def mount(self, name=None):
        """
        Mount resource as SquashFS image

        :param name: target subdirectory name, if None, will be uses unique number
        """
        return super(ResourceData, self).mount(name=name)
