from __future__ import absolute_import

import abc
import base64
import socket
import inspect
import logging
import datetime as dt
import functools as ft
import threading
import contextlib
import collections

import six
from six.moves import http_client as httplib

from sandbox.common import log
from sandbox.common import lazy
from sandbox.common import rest
from sandbox.common import crypto
from sandbox.common import format as common_format
from sandbox.common import config as common_config
from sandbox.common import errors as common_errors
from sandbox.common import patterns
from sandbox.common import encoding
from sandbox.common import itertools as common_itertools
from sandbox.common import threading as common_threading

import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
import sandbox.common.types.notification as ctn

from . import path as sdk2_path
from . import helpers
from . import internal
from . import resource as sdk2_resource
from . import parameters
from . import service_resources


__all__ = (
    "Requirements", "Parameters", "Context", "Notification", "OldTaskWrapper",
    "Task", "ServiceTask", "ReleaseTemplate", "Vault", "VaultItem",
    "Wait", "WaitTime", "WaitTask", "WaitOutput",
    "report", "header", "footer",
)


ReportInfo = collections.namedtuple("ReportInfo", ("title", "label", "function_name"))


def report(title=None, label=None):
    """
    Mark task class' method as a report to display its return value
    on the task view page (it's also accessible at
    `/task/{id}/reports/{report_label} <https://sandbox.yandex-team.ru/media/swagger-ui/#/task/task_report_get>`_
    API endpoint). Report location depends on its type:

    - ``sdk2.header()``: top of the page, before task info (fetched on page load);

    - ``sdk2.footer()``: bottom of the page, after task's parameters (fetched on page load);

    - ``sdk2.report()``: a separate tab (or a list of tabs) put after "Hardware". User-defined tabs
        are marked with an icon for better visibility.

    Once again, a report can be anything you want to see on the view page:
    performance tests results, execution stages duration, an informational block
    and so on. If you're familiar with "custom footer" feature, know that report is the same thing,
    except it's not necessarily located in the bottom of the page.

    .. note:: Report functions are called on the server side. This leads to a few conclusions:

        - **Do not** make heavy (time-consuming) calls in reports, otherwise page load time will degrade;

        - The task's state may and **will not** be actual before and during execution.

        The latter means that a) instance properties set in runtime do not exist, and
        b) task context is oftentimes outdated, since it's only updated on status change;

    You can override parent class' report if you specify the same label;
    tabs are ordered by methods' appearance order in task class,
    therefore you can re-order them if you try hard enough (hint: use different function names)

    Return value of **the function being decorated** is expected to be in one of the following forms:

    - raw value, or

    - raw HTML, in case you need something tricky and/or pretty.

    The third form, which has a table-like structure, is deprecated.

    .. note:: You can't call this function directly, as it expects the call to happen
        in a task class' namespace to fill class' ``__reports__`` field.

    Usage (you can find more examples on `this page <https://wiki.yandex-team.ru/sandbox/tasks/report>`_):

    .. code-block:: python

        class Reporter(sdk2.Task):

            @sdk2.report(label="almonds")
            def anything(self):
                return 1

            # produces a report called "peanuts" whose tab has this long title slapped on it
            @sdk2.report(title="I don't want to waste my time on labels")
            def peanuts(self):
                return 2

            # a special case of report, shown in the bottom of the task view. Has "footer" label
            @sdk2.footer()
            def copyright(self):
                return 3

            # a special case of report, shown at the top (well, almost) of the task view. Has "header" label
            @sdk2.header()
            def on_execute(self):
                return "You can even decorate on_* hooks"


        class ReporterSuccessor(Reporter):

            # reports are overridden on label match, therefore
            # GET /task/{id}/reports/almonds returns [{"content": 0}] for this task type instead of 1
            # oh, and it'll appear after parent's reports, because the function name's different
            @sdk2.report(label="almonds")
            def does_not_matter(self):
                return 0

            peanuts = None  # hide the report (useful if you subclass from another task)
            copyright = sdk2.report()(None)  # another way to hide a report

    :param title: report title, a string put on the report's individual tab
    :param label: report name (case-insensitive; cast to lowercase),
        in case it's omitted, function's ``__name__`` goes here. Has to be unique within a task class.
    :return: inner wrapper function (why do you need to know this!)
    """

    reports = inspect.currentframe().f_back.f_locals.setdefault("__reports__", collections.OrderedDict())

    def wrapper(f=None):
        if f is None:
            return None

        label_ = (f.__name__ if label is None else label).lower()
        assert label_ not in reports, "There's already a report with label {}".format(label_)
        reports[label_] = ReportInfo(title, label_, f.__name__)

        def wrapper_(*a, **kw):
            return f(*a, **kw)

        return wrapper_

    return wrapper


footer = ft.partial(report, title="Footer", label="footer")
header = ft.partial(report, title="Header", label="header")
for fn in (footer, header):
    fn.__doc__ = "{} is a special case of `report` decorator; refer to `sdk2.report` function's documentation".format(
        fn.func.__name__
    )


# noinspection PyAbstractClassRequirements
class Requirements(internal.task.Requirements):
    """
    Task requirements
    """
    #: Execute task only on the host
    host = parameters.String("Required host")
    #: Use privileged container for execution. Reset on task copy.
    privileged = parameters.Bool("Required privileged container for execution")
    #: Minimum disk space in MiB
    disk_space = parameters.Integer(
        "Required disk space",
        default=lazy.Deferred(lambda: common_config.Registry().common.task.execution.disk_required)
    )
    #: Minimum RAM size in MiB
    ram = parameters.Integer(
        "Required RAM size",
        default=lazy.Deferred(lambda: common_config.Registry().common.task.execution.required_ram)
    )
    #: RAM drive in MiB. Reset on task copy.
    ramdrive = parameters.RamDrive("Required RAM drive")
    #: Minimum number of cores
    cores = parameters.Integer("Required CPU cores", default=None)
    #: Client tags
    client_tags = parameters.ClientTags("Client tags", default=ctc.Tag.GENERIC)
    #: Type of name servers in DNS config
    dns = parameters.DnsType("DNS config type")
    #: Task environments
    environments = parameters.Environments("Task environments")
    #: Semaphores. Reset on task copy.
    semaphores = parameters.Semaphores("Semaphores")
    #: Porto layers.
    porto_layers = parameters.Resource(
        "Porto layers",
        resource_type=service_resources.BasePortoLayer,
        multiple=True,
        default=[]
    )
    #: Use specific resource with tasks code.
    tasks_resource = parameters.Resource(
        "Resource with tasks code",
        resource_type=[
            service_resources.SandboxTasksArchive,
            service_resources.SandboxTasksBinary,
            service_resources.SandboxTasksImage
        ]
    )
    #: Use specific resource with LXC container.
    container_resource = parameters.Container("Resource with LXC container", default=None)
    #: Required resources.
    resources = parameters.Resource("Required resources", multiple=True, default=[])
    #: Resources space reserve.
    resources_space_reserve = parameters.ResourcesSpaceReserve("Resources space reserve", default=None)

    class Caches(internal.task.Caches):
        pass


# noinspection PyAbstractClass
class Parameters(internal.task.Parameters):
    """
    Task parameters
    """
    # Common parameters
    #: Description
    description = parameters.String("Description", required=True)
    #: Maximum restarts
    max_restarts = parameters.Integer(
        "Maximum restarts",
        required=True,
        default=lazy.Deferred(lambda: common_config.Registry().common.task.execution.max_restarts)
    )
    #: Kill timeout. Reset on task copy.
    kill_timeout = parameters.Integer(
        "Kill timeout",
        required=True,
        default=lazy.Deferred(lambda: common_config.Registry().common.task.execution.timeout * 3600)
    )
    #: Go to state FAILURE on any error
    fail_on_any_error = parameters.Bool("Go to state FAILURE on any error", required=True)
    #: Task is hidden in list by default
    hidden = parameters.Bool("Task is hidden in list by default", required=True)
    #: Desired task priority
    priority = parameters.Priority("Desired task priority")
    #: Notification settings
    notifications = parameters.Notifications("Notification settings")
    #: Task owner. Reset on task copy.
    owner = parameters.String("Task owner")
    #: Release to. Reset on task copy.
    release_to = parameters.ReleaseTo("Release to")
    #: Detailed disk usage statistics
    dump_disk_usage = parameters.Bool("Detailed disk usage statistics", required=True, default=True)
    #: Tcpdump arguments that are used for network packets logging. No logging on empty string.
    tcpdump_args = parameters.String("Log network packets using tcpdump that is run with these arguments")
    #: Task tags
    tags = parameters.List("Task tags", value_type=parameters.TaskTag)
    #: Deduplication uniqueness. Reset on task copy.
    uniqueness = internal.parameters.DeduplicationUniqueness("Deduplication uniqueness")
    #: Expire task this time from now. Reset on task copy.
    expires = parameters.Timedelta("Expires at")
    #: suspend on one of the selected statuses
    suspend_on_status = parameters.List("Suspend task on transition to any of specified statuses")
    #: push self resource with binary tasks to subtasks
    push_tasks_resource = parameters.Bool("Pushing tasks resource")
    #: Task priority score
    score = parameters.Integer("Task priority score", default=0)


class Context(internal.task.Context):
    """
    Task context

    @DynamicAttrs
    """

    def save(self):
        """
        Save task context
        """
        return super(Context, self).save()


class Notification(object):
    """
    Task notification setting

    @DynamicAttrs
    """

    def __init__(self, statuses, recipients, transport, check_status=None, juggler_tags=None):
        statuses = ctt.Status.Group.expand(statuses)
        self.__statuses = [status for status in statuses if status in ctt.Status]
        assert transport in ctn.Transport
        if check_status:
            assert check_status in ctn.JugglerStatus
        self.__transport = transport
        self.__recipients = list(recipients)
        self.__check_status = check_status
        self.__juggler_tags = list(juggler_tags or [])

    @property
    def statuses(self):
        return self.__statuses

    @property
    def transport(self):
        return self.__transport

    @property
    def recipients(self):
        return self.__recipients

    @property
    def check_status(self):
        return self.__check_status

    @property
    def juggler_tags(self):
        return self.__juggler_tags

    def __repr__(self):
        return "<Notification on {!r} for {!r} via {!r}> with check status {!r}".format(
            self.statuses, self.recipients, self.transport, self.check_status
        )


class Task(internal.task.Task):
    """
    Base task class

    @DynamicAttrs
    """
    @internal.common.overridable
    class Requirements(Requirements):
        pass

    @internal.common.overridable
    class Parameters(Parameters):
        pass

    @internal.common.overridable
    class Context(Context):
        pass

    def __init__(self, parent, __requirements__=None, **parameters_):
        """
        Create new task or subtask

        :param parent: parent task or None
        :param __requirements__: optional task requirements
        :param parameters: task parameters
        """
        super(Task, self).__init__(parent, __requirements__, **parameters_)

    #: Instance of client to Sandbox REST API
    server = internal.task.Task.__dict__["server"]

    _sdk_server = internal.task.Task.__dict__["_sdk_server"]

    #: Object of the task currently executing
    current = internal.task.Task.__dict__["current"]
    #: Resource with logs of the task currently executing
    log_resource = internal.task.Task.__dict__["log_resource"]

    def log_path(self, *path):
        """
        Build path relative to current logs

        :param path: path components
        :rtype: pathlib2.Path
        """
        return super(Task, self).log_path(*path)

    @internal.common.dual_method
    def find(self, task_type=None, **constraints):
        """
        Find tasks according to specified constraints

        :param task_type: find tasks of the type, if None then find tasks of type of current class
        :param constraints: query constraints, according to :py:class:`sandbox.web.api.v1.task.TaskList.Get`
                            with the following exceptions:
                            `description` instead of `desc_re`,
                            `parent` contains :py:class:`~sandbox.sdk2.task.Task` object,
                            `id` contains task id or list of ids
        :rtype: sandbox.sdk2.internal.common.Query
        """
        return super(Task, self).find(task_type, **constraints)

    #: Task id (RO)
    id = internal.task.Task.id
    #: Type (RO)
    type = internal.task.Task.__dict__["type"]
    #: Author (RO)
    author = internal.task.Task.author
    #: Owner (RO)
    owner = internal.task.Task.owner
    #: Status (RO)
    status = internal.task.Task.status
    #: Parent task (RO)
    parent = internal.task.Task.parent
    #: Required host (RO)
    host = internal.task.Task.host
    #: Hidden flag (RO)
    hidden = internal.task.Task.hidden
    #: Creation time (datetime.datetime) (RO)
    created = internal.task.Task.created
    #: Update time (datetime.datetime) (RO)
    updated = internal.task.Task.updated
    #: Execution info message (RO)
    info = internal.task.Task.info
    #: Ram drive (RW)
    ramdrive = internal.task.Task.ramdrive
    #: Scheduler (RO)
    scheduler = internal.task.Task.scheduler
    #: Current task's container metadata if any. Will be provided on task execution with the instance of
    #: `sandbox.common.types.client.Container` class.
    container = None

    @internal.common.overridable
    @property
    @footer()
    def footer(self):
        """
        Return custom task's footer data for new UI. The return value should be JSON serializable.
        In case of the return value is :class:`list`, it will be returned as is,
        in case of :class:`tuple`, the first value will be used as helper name, the second - as context,
        in other cases, it will be placed as context value.
        """
        return None

    @internal.common.overridable
    @property
    def release_template(self):
        """
        Return custom task's release template for new UI.
        The return value should be an instance of :class:`ReleaseTemplate`. The default implementation
        will use legacy :py:meth:`arcadia_info` to fill the template.
        """
        return ReleaseTemplate()

    # noinspection PyMethodMayBeStatic
    @internal.common.overridable
    def on_create(self):
        """ Called when task creation """

    # noinspection PyMethodMayBeStatic
    @internal.common.overridable
    def on_save(self):
        """ Called when updating task in status DRAFT """

    # noinspection PyMethodMayBeStatic
    @internal.common.overridable
    def on_enqueue(self):
        """ Called before task enqueued """

    # noinspection PyMethodMayBeStatic
    @internal.common.overridable
    def on_prepare(self):
        """ Called when task preparing to execution on the host """

    # noinspection PyMethodMayBeStatic
    @internal.common.overridable
    def on_execute(self):
        """ Called when task executing on the host """

    def __finished(self, mark_as_ready=None):
        from sandbox.agentr import errors
        try:
            self.agentr.finished(drop_session=False, mark_as_ready=mark_as_ready)
            for res in self.server.resource.read(task_id=self.id, limit=internal.common.Query.MAX_LIMIT)["items"]:
                sdk2_resource.Resource[res["id"]].reload(res)
        except errors.InvalidResource as ex:
            raise common_errors.TaskError(repr(ex))

    # noinspection PyMethodMayBeStatic
    @internal.common.overridable
    def on_finish(self, prev_status, status):
        """
        Called when task going to finish

        :param prev_status: status from which the task is going to switch
        :param status: status in which the task is going to switch
        """

    # noinspection PyUnusedLocal
    @internal.common.overridable
    def on_success(self, prev_status):
        """
        Called when task going to status SUCCESS

        :param prev_status: status from which the task is going to switch
        """
        self.__finished(mark_as_ready=True)

    # noinspection PyUnusedLocal
    @internal.common.overridable
    def on_failure(self, prev_status):
        """
        Called when task going to status FAILURE

        :param prev_status: status from which the task is going to switch
        """
        self.__finished(mark_as_ready=False)

    # noinspection PyUnusedLocal
    @internal.common.overridable
    def on_break(self, prev_status, status):
        """
        Called when task going to switch to status from group BREAK

        :param prev_status: status from which the task is going to switch
        :param status: status in which the task is going to switch
        """
        self.__finished(mark_as_ready=False)

    # noinspection PyMethodMayBeStatic
    @internal.common.overridable
    def on_timeout(self, prev_status):
        """
        Called when task going to switch to status TIMEOUT

        :param prev_status: status from which the task is going to switch
        """

    @internal.common.overridable
    def on_before_timeout(self, seconds):
        """
        Called before task executor is killed by timeout.

        :param seconds: seconds left until timeout
        """
        logging.warning("[TC] %s seconds left until task timeout", seconds)
        if seconds == min(self.timeout_checkpoints()):
            logging.warning("[TC] Dumping threads tracebacks for debug purpose:")
            common_threading.dump_threads()

    # noinspection PyUnusedLocal
    @internal.common.overridable
    def on_terminate(self):
        """
        Called in signal handler when task executing is being stopped forcibly.
        """

    @internal.common.overridable
    def timeout_checkpoints(self):
        """
        This method returns a list of intervals (in seconds) before timeout
        when on_before_timeout method is called.

        :rtype: list(int)
        """
        return [10, 30, 60, 60 * 3, 60 * 5]

    # noinspection PyUnusedLocal
    @internal.common.overridable
    def on_wait(self, prev_status, status):
        """
        Called when task going to switch to status from group WAIT

        :param prev_status: status from which the task is going to switch
        :param status: status in which the task is going to switch
        """
        self.__finished()

    def mark_released_resources(self, status, ttl="inf"):
        resources = [
            r for r in sdk2_resource.Resource.find(task=self, state=ctr.State.READY).limit(0)
            if r.type.releasable
        ]
        for resource in resources:
            resource.ttl = resource.ttl_on_release or ttl
            resource.released = status
            if not resource.backup_task:
                resource.backup_task = True
        return resources

    @internal.common.overridable
    def on_release(self, parameters_):
        """
        Task specific actions. Executed when release has submitted.

        :param parameters_: release specific parameters
        """
        logging.debug("Release parameters: %r", parameters_)
        self._send_release_info_to_email(parameters_)
        self.mark_released_resources(parameters_["release_status"])

    def hint(self, hints):
        """
        Add one or several hints (indexed labels that can be used for fast searching) to task

        :param hints: a value, or list of values to be added as hints
        """
        hints = [str(hint) for hint in common_itertools.chain(hints)]
        self._sdk_server.task[self.id].hints(hints)
        self.Parameters.__explicit_hints__.update(hints)

    @property
    def hints(self):
        # noinspection PyArgumentList
        return self.Parameters.__gethints__()

    def save(self):
        """
        Save task object

        :return: self
        """
        assert self.status == ctt.Status.DRAFT, "Cannot save task in status {!r}".format(self.status)
        # noinspection PyArgumentList
        data = self.Parameters.__getstate__()
        # noinspection PyArgumentList
        data.update(self.Requirements.__getstate__())
        self._sdk_server.task[self.id] = data
        return self

    def enqueue(self):
        """
        Enqueue task to execution

        :return: self
        """
        result = self._sdk_server.batch.tasks.start.update([self.id])[0]
        status, message = map(result.get, ("status", "message"))
        if status != ctm.BatchResultStatus.SUCCESS:
            logging.log(
                logging.WARNING if status == ctm.BatchResultStatus.WARNING else logging.ERROR,
                "Task #%d is %s with %s: %s",
                self.id,
                "enqueued" if status == ctm.BatchResultStatus.WARNING else "not enqueued", status.lower(), message
            )
            if status == ctm.BatchResultStatus.ERROR:
                raise common_errors.TaskNotEnqueued("Task #{} is not enqueued: {}".format(self.id, message))
        return self

    def stop(self):
        """
        Stop task

        :return: self
        """
        if self is self.current:
            raise common_errors.TaskStop("Self stop")
        result = self._sdk_server.batch.tasks.stop.update([self.id])[0]
        if result["status"] != ctm.BatchResultStatus.SUCCESS:
            raise common_errors.TaskError(
                "Error occurred while stopping task #{}: {}".format(self.id, result["message"])
            )
        return self

    def delete(self):
        """
        Delete task

        :return: self
        """
        result = self._sdk_server.batch.tasks["delete"].update([self.id])[0]
        if result["status"] != ctm.BatchResultStatus.SUCCESS:
            raise common_errors.TaskError(
                "Error occurred while deleting task #{}: {}".format(self.id, result["message"])
            )
        return self

    def suspend(self):
        """
        Suspend task

        :return: self
        """
        result = self._sdk_server.batch.tasks.suspend.update([self.id])[0]
        if result["status"] != ctm.BatchResultStatus.SUCCESS:
            raise common_errors.TaskError(
                "Error occurred while suspending task #{}: {}".format(self.id, result["message"])
            )
        if self is self.current:
            try:
                logging.debug("Waking up client via socket at port %s", common_config.Registry().client.port)
                socket.create_connection(("127.0.0.1", common_config.Registry().client.port)).close()
            except Exception as ex:
                logging.warning("Error waking up client instance: %s", ex)

            logging.info("Waiting for suspend")
            common_itertools.progressive_waiter(
                0.1, 30, self.Parameters.kill_timeout,
                lambda: self._sdk_server.task[self.id][:]["status"] not in (ctt.Status.SUSPENDING, ctt.Status.SUSPENDED)
            )
        return self

    def resume(self):
        """
        Resume task execution

        :return: self
        """
        result = self._sdk_server.batch.tasks.resume.update([self.id])[0]
        if result["status"] != ctm.BatchResultStatus.SUCCESS:
            raise common_errors.TaskError(
                "Error occurred while resuming task #{}: {}".format(self.id, result["message"])
            )
        return self

    def path(self, *args):
        return sdk2_path.Path(common_config.Registry().client.tasks.data_dir).joinpath(
            *(ctt.relpath(self.id) + list(map(str, args)))
        )

    @patterns.singleton_property
    def synchrophazotron(self):
        return self.path("synchrophazotron")

    @patterns.singleton_property
    def arcaphazotron(self):
        return self.path("arcaphazotron")

    def set_info(self, info, do_escape=True):
        """
        Adds text into field 'info'

        :param info: text to add
        :param do_escape: if True, escape text before add
        :return: added text
        """
        assert self is self.current, "Method allowed for current task only"
        logging.info("Task #%s info added: %s", self.id, info)
        info = encoding.force_unicode_safe(info)
        info = log.VaultFilter.filter_message_from_logger(logging.getLogger(), info)

        if do_escape:
            info = encoding.escape(info)
        self.__info += u"<div class=\"hr\">{}</div>{}\n".format(
            common_format.dt2str(dt.datetime.now()), info
        )
        self._sdk_server.task.current.execution = dict(description=self.__info)
        return self.__info

    @patterns.singleton_property
    def memoize_stage(self):
        return helpers.MemoizeCreator(self)


class ServiceTask(Task):
    """
    Base class for service tasks
    """


class OldTaskWrapper(Task):
    """
    Base class for SDK1 tasks represented as SDK2 task object.
    """


class ReleaseTemplate(patterns.Abstract):
    """
    An instance of this class is required to describe task's release template data
    (see :py:meth:`release_template`).
    """
    __slots__ = ("cc", "subject", "message", "types")
    # noinspection PyTypeChecker
    __defs__ = (None, None, None, list(iter(ctt.ReleaseStatus)))


class VaultValue(str):
    # noinspection PyInitNewSignature
    def __new__(cls, text, description=""):
        # noinspection PyArgumentList
        self = str.__new__(cls, text)
        self.description = description
        return self


class VaultItem(object):
    def __init__(self, owner_or_name, name=None):
        self.owner, self.name = (None, owner_or_name) if name is None else (owner_or_name, name)

    def data(self):
        """
        Get the vault content
        :return: either `VaultFuture` (in batch mode) or bytestring with vault content
        """
        return Vault.data(self.owner, self.name)

    def __str__(self):
        if self.owner:
            return "{}:{}".format(self.owner, self.name)
        return self.name


@contextlib.contextmanager
def vault_exception_handler():
    try:
        yield
    except rest.Client.HTTPError as ex:
        if ex.response.status_code == httplib.NOT_FOUND:
            raise common_errors.VaultNotFound(ex)
        elif ex.response.status_code == httplib.FORBIDDEN:
            raise common_errors.VaultNotAllowed(ex)
        else:
            raise


class VaultFuture(object):
    __slots__ = ("_future",)

    def __init__(self, fut):
        self._future = fut

    def result(self):
        """
        Return the vault content from the completed future, or raise an exception if the request was unsuccessful.

        :return: bytestring with vault content
        :raises: a subclass of `VaultError` or `HTTPError`
        """

        try:
            with vault_exception_handler():
                data = self._future.result()
        except rest.BatchFuture.NotReadyError:
            raise common_errors.VaultError("Vault future is not completed")

        return Vault._decrypt(data)


class Vault(internal.task.Vault):
    # The variable keeps a batch-enabled REST client that is used in batch-mode
    _local = threading.local()

    # noinspection PyMethodParameters
    @patterns.classproperty
    @contextlib.contextmanager
    def batch(cls):
        """
        This context manager enables batch mode for vault requests.
        Inside the context, `.data()` methods in both `Vault` and `VaultItem` return vault future objects.
        On exit, the vault contents are fetched with a single API request;
        HTTPError is raised if the request is unsuccessful.
        Once the context is exited succesfully, the futures become completed.

        Usage examples:

        .. code-block:: python

            with Vault.batch:
                f1 = Vault.data("owner", "token01")
                f2 = VaultItem("guest", "token02").data()
                f3 = Vault.data("owner", "non-existing-token")

                f1.result()  # raises `VaultError`: the future is not complete yet

            token = f1.result()
            token2 = f2.result()
            f3.result()  # raises `VaultNotFound`
        """

        with rest.Batch(rest.Client()) as batch:
            cls._local.batch = batch
            yield
            cls._local.batch = None

    @classmethod
    def data(cls, owner_or_name, name=None):
        """
        Get the vault content by its name and (optionally) owner.

        :param owner_or_name: owner or name of vault item
        :param name: name of vault item
        :return: either `VaultFuture` (in batch mode) or bytestring with vault content
        """
        owner, name = (None, owner_or_name) if name is None else (owner_or_name, name)

        query = {"name": name}
        if owner is not None:
            query["owner"] = owner

        batch = getattr(cls._local, "batch", None)
        client = batch or rest.Client()  # either ordinary or batch-enabled REST client

        with vault_exception_handler():
            response = client.vault.data.read(**query)

        if batch:
            return VaultFuture(response)

        return cls._decrypt(response)

    @classmethod
    def _decrypt(cls, response, vault_key=None):
        """
        Decrypt raw response from /vault/data endpoint.

        :param response: the response data
        :param vault_key: for internal use, do not set it when calling the method
        :return str: vault content
        """
        assert vault_key, "Vault key is not defined"
        encrypted_data = base64.b64decode(response["data"])
        decrypted_data = crypto.AES(vault_key).decrypt(encrypted_data, False)
        assert decrypted_data is not None, "Vault data cannot be decrypted"
        logging.info(
            "Received vault key (owner=%s, name=%s) of %d bytes length with id %s.",
            response["owner"], response["name"], len(decrypted_data), response["id"]
        )
        if len(decrypted_data) > 5:
            logger = logging.getLogger()
            vault_filter = log.VaultFilter.filter_from_logger(logger)
            if vault_filter:
                vault_filter.add_record(response["id"], decrypted_data)
        return VaultValue(decrypted_data, description=response["description"])


class Wait(six.with_metaclass(abc.ABCMeta, BaseException)):
    @abc.abstractmethod
    def __init__(self, *_, **__):
        pass

    @abc.abstractmethod
    def __call__(self, task):
        pass

    @staticmethod
    def _reset_cache():
        Task.__cache__.clear()
        sdk2_resource.Resource.__cache__.clear()

    @abc.abstractmethod
    def encode_args(self):
        pass

    def encode(self):
        return {
            "type": type(self).__name__,
            "args": self.encode_args()
        }

    @staticmethod
    def decode(data):
        registry = {
            WaitTime.__name__: WaitTime,
            WaitTask.__name__: WaitTask,
            WaitOutput.__name__: WaitOutput,
        }
        klass = registry.get(data["type"], None)
        if klass is None:
            raise ValueError("Unknown Wait type: {}".format(data["type"]))

        return klass(**data["args"])


class WaitTime(Wait):
    def __init__(self, time_to_wait):
        """
        :param time_to_wait: time to wait in seconds
        """
        self.__time_to_wait = int(time_to_wait)

    def __call__(self, task):
        logging.info("Wait %s seconds", self.__time_to_wait)
        if self.__time_to_wait < 1:
            self._reset_cache()
            raise common_errors.NothingToWait
        try:
            # noinspection PyProtectedMember
            task._sdk_server.task.current.trigger.time(period=self.__time_to_wait)
        except rest.Client.HTTPError as ex:
            if ex.status == httplib.CONFLICT:
                raise common_errors.TriggerAlreadyExists
            raise

    @property
    def timeout(self):
        return self.__time_to_wait

    def encode_args(self):
        return {
            "time_to_wait": self.timeout
        }


class WaitTask(Wait):
    def __init__(self, tasks, statuses, wait_all=True, timeout=None):
        """
        :param tasks: task or list of task objects or identifiers the task will wait for
        :param statuses: status or list of statuses or list of groups to wait for
        :param wait_all: bool, wait for all or any of tasks specified
        :param timeout: in seconds, if defined wake up the task after this time in either case
        """
        self.__tasks = list(map(int, common_itertools.chain(tasks)))
        self.__statuses = list(ctt.Status.Group.expand(statuses) - ctt.Status.Group.NONWAITABLE)
        if not self.__statuses:
            raise ValueError("Empty statuses list to wait for")
        self.__wait_all = wait_all
        self.__timeout = timeout and int(timeout)

    def __call__(self, task):
        logging.info(
            "Wait %s tasks %r for %r %s.",
            "all" if self.__wait_all else "any", self.__tasks, self.__statuses,
            "without timeout" if self.__timeout is None else "with timeout {}s.".format(self.__timeout),
        )
        try:
            # noinspection PyProtectedMember
            task._sdk_server.task.current.trigger.task(
                targets=self.__tasks,
                statuses=self.__statuses,
                wait_all=self.__wait_all,
                timeout=self.__timeout
            )
        except rest.Client.HTTPError as ex:
            if ex.status == httplib.NOT_ACCEPTABLE:
                self._reset_cache()
                raise common_errors.NothingToWait
            elif ex.status == httplib.CONFLICT:
                raise common_errors.TriggerAlreadyExists
            raise

    @property
    def timeout(self):
        return self.__timeout

    @property
    def tasks(self):
        return self.__tasks

    @property
    def statuses(self):
        return self.__statuses

    @property
    def wait_all(self):
        return self.__wait_all

    def encode_args(self):
        return {
            "tasks": self.tasks,
            "statuses": self.statuses,
            "wait_all": self.wait_all,
            "timeout": self.timeout,
        }


class WaitOutput(Wait):
    def __init__(self, targets, wait_all, timeout=None):
        """
        :param targets: dictionary {task: [output_fields*]}
        :param wait_all: bool, wait for all or any of task/field pairs specified
        :param timeout: in seconds, if defined wake up the task after this time in either case
        """
        if not targets:
            raise ValueError("No targets specified")
        self.__targets = {}
        for task, fields in six.iteritems(targets):
            self.__targets[int(task)] = list(common_itertools.chain(fields))
        self.__wait_all = wait_all
        self.__timeout = timeout and int(timeout)

    def __call__(self, task):
        logging.info(
            "Wait %s tasks for output fields %r with timeout %r",
            "all" if self.__wait_all else "any", self.__targets, self.__timeout
        )
        try:
            # noinspection PyProtectedMember
            task._sdk_server.task.current.trigger.output(
                targets=self.__targets,
                wait_all=self.__wait_all,
                timeout=self.__timeout,
            )
        except rest.Client.HTTPError as ex:
            if ex.status == httplib.NOT_ACCEPTABLE:
                self._reset_cache()
                raise common_errors.NothingToWait
            elif ex.status == httplib.CONFLICT:
                raise common_errors.TriggerAlreadyExists
            raise

    @property
    def timeout(self):
        return self.__timeout

    @property
    def targets(self):
        return self.__targets

    @property
    def wait_all(self):
        return self.__wait_all

    def encode_args(self):
        return {
            "targets": self.targets,
            "wait_all": self.wait_all,
            "timeout": self.timeout,
        }
