from __future__ import absolute_import

import os
import threading

import six
from six.moves.urllib import parse as urlparse

from sandbox.common import fs
from sandbox.common import os as common_os
from sandbox.common import errors as common_errors
from sandbox.common import patterns
from sandbox.common import itertools as common_itertools
from sandbox.common import threading as common_threading
from sandbox.common.vcs import cache as vcs_cache

from sandbox import sdk2
from sandbox.sdk2.helpers import process


class Git(vcs_cache.CacheableGit):
    """
    Class for working with Git repositories.

    All public methods are protected with file lock to avoid RC during work with repository.
    """

    def __init__(self, url, push_url=None, executable="git", filter_branches=True):
        self._url = url
        self._push_url = push_url
        self.executable = executable
        self._filter_branches = filter_branches
        self._repo_name = self._get_repo_name(url)

    @classmethod
    def raise_if_unavailable(cls):
        """ All public methods are protected with file lock. So do not restrict git usage on multislot clients. """

    @classmethod
    def _get_repo_name(cls, url):
        url_parts = urlparse.urlparse(url)
        return "{}__{}".format(
            url_parts.hostname,
            "_".join(common_itertools.chain(*url_parts.path.split("/"))),
        )

    @patterns.singleton_property
    def cache_repo_dir(self):
        return os.path.abspath(os.path.join(self.base_cache_dir, self._repo_name))

    @patterns.singleton_property
    def cache_repo_lock(self):
        """
        Repository modification lock.
        If current user has root privileges then directory with vcs caches is mounted as read-only filesystem.
        So it is enough to user RLock for safety in that case.
        """
        return threading.RLock() if common_os.User.has_root else common_threading.RFLock(self.cache_repo_dir + ".lock")

    def raw_execute(self, args, cwd=None, shell=None):
        """
        Execute raw command

        :param args: list of arguments to git executable
        :param cwd: working directory
        :param shell: run git process from shell
        :return: process object
        """
        return process.subprocess.Popen(
            [self.executable] + list(args),
            cwd=cwd,
            shell=shell,
            stdout=process.subprocess.PIPE,
            stderr=process.subprocess.PIPE,
        )

    def execute(self, *args, **kwargs):
        logger_name = kwargs.pop("logger_name", self._repo_name)
        command_line = [self.executable] + list(args)
        with process.ProcessLog(task=sdk2.task.Task.current, logger=logger_name) as pl:
            with self.cache_repo_lock:
                rc = process.subprocess.Popen(
                    command_line,
                    stdout=pl.stdout,
                    stderr=pl.stderr,
                    **kwargs
                ).wait()
            if rc:
                raise common_errors.SubprocessError(
                    "Command {} died with exit code {}. Seek details in {}.log".format(
                        command_line, rc, logger_name
                    )
                )

    def touch_cache(self):
        if sdk2.Task.current:
            self.add_cache_metadata(
                self.cache_repo_dir,
                sdk2.Task.current.id,
                url=self._url,
            )

    def _ensure_cache_exists(self):
        if not os.path.exists(self.cache_repo_dir):
            fs.make_folder(self.base_cache_dir)
            self.execute(
                "clone", "--progress", "--mirror", self._url, self.cache_repo_dir,
                logger_name=self._repo_name + "_mirror",
            )

    def update_cache_repo(self, *refs):
        """
        Fetch `refs` to cache repository (if `refs` are not specified, all heads and tags will be fetched).

        :type refs: str
        """
        with self.cache_repo_lock:
            self._ensure_cache_exists()

            refs = refs or ["refs/heads/*", "refs/tags/*"]
            cmd = ["fetch", "--progress", "origin"]
            cmd.extend("+{0}:{0}".format(ref) for ref in refs)
            self.execute(*cmd, cwd=self.cache_repo_dir)

        self.touch_cache()

    def _init_repo(self, target_dir, bare=False, config=None):
        self._ensure_cache_exists()

        cmd = ["init", target_dir]
        if bare:
            cmd.append("--bare")
        self.execute(*cmd)

        self.execute("remote", "add", "origin", self._url, cwd=target_dir)
        if self._push_url:
            self.execute("remote", "set-url", "--push", "origin", self._push_url, cwd=target_dir)

        if bare:
            alternates_file_path = os.path.join(target_dir, "objects", "info", "alternates")
        else:
            alternates_file_path = os.path.join(target_dir, ".git", "objects", "info", "alternates")
        with open(alternates_file_path, "w") as alternates:
            alternates.write(os.path.join(self.cache_repo_dir, "objects"))

        self.setup_git_config(target_dir, config)

    def setup_git_config(self, target_dir, config):
        config = config or {}
        for key, value in sorted(six.iteritems(config)):
            if value is None:
                value = "--unset"
            elif isinstance(value, bool):
                value = str(value).lower()
            else:
                value = str(value)
            self.execute("config", key, value, cwd=target_dir)

    def _checkout(self, target_dir, branch=None, commit=None):
        if branch:
            if self._filter_branches:
                cmd = [self.executable, "ls-remote", "--heads", "--tags", "origin", branch]
            else:
                # need for work with pull requests
                cmd = [self.executable, "ls-remote", "origin", branch]
            ref_lines = process.subprocess.Popen(
                cmd,
                stdout=process.subprocess.PIPE,
                stderr=process.subprocess.PIPE,
                cwd=target_dir,
            ).communicate()[0].splitlines()

            for line in ref_lines:
                remote_commit, full_ref = line.strip().split(None, 1)
                self.execute("update-ref", full_ref, commit or remote_commit, cwd=target_dir)
            self.execute("checkout", "-f", branch, cwd=target_dir)
        elif commit:
            self.execute("checkout", "-f", commit, cwd=target_dir)
        else:
            raise ValueError("A branch or commit should be specified.")

    def checkout(self, target_dir, branch=None, commit=None):
        return self._checkout(target_dir, branch, commit)

    def _clone_from_cache_repo(
        self, target_dir, branch=None, commit=None, sparse_checkout_paths=None, config=None,
    ):
        self._init_repo(target_dir, config=config)

        if sparse_checkout_paths is not None:
            self.execute("config", "core.sparsecheckout", "true", cwd=target_dir)

            with open(os.path.join(target_dir, ".git", "info", "sparse-checkout"), "w") as sparse_checkout:
                sparse_checkout.write("\n".join(sparse_checkout_paths))

        self._checkout(target_dir, branch, commit)

    def clone(self, target_dir, branch=None, commit=None, sparse_checkout_paths=None, config=None):
        """
        Clone repository to `target_dir`.

        A `branch` or `commit` (or both) should be specified.
        If `commit` is specified, it will be checkouted. Otherwise the latest commit of `branch` will be checkouted.
        If `branch` is specified, HEAD will point to it. Otherwise it will be detached.

        To do sparse checkout, specify list of `sparse_checkout_paths`.

        To apply git config values to repository, specify `config` dict.

        :type target_dir: str
        :type branch: str or NoneType
        :type commit: str or NoneType
        :type sparse_checkout_paths: collections.Iterable of str or NoneType
        :type config: dict of (str, str) or NoneType
        """
        refs = [branch] if branch else []
        self.update_cache_repo(*refs)
        with self.cache_repo_lock:
            self._clone_from_cache_repo(
                target_dir, branch, commit=commit, sparse_checkout_paths=sparse_checkout_paths, config=config,
            )
            self.touch_cache()

    def clone_bare(self, target_dir, config=None):
        """
        Clone bare repository to `target_dir`.
        To apply git config values to repository, specify `config` dict.

        :type target_dir: str
        :type config: dict of (str, str) or NoneType
        """
        with self.cache_repo_lock:
            self._init_repo(target_dir, bare=True, config=config)
            self.touch_cache()


class GitLfs(Git):
    """ Class for working with Git Large File Storage """

    def update_cache_repo(self, *refs):
        if not refs:
            raise ValueError("References should be specified")

        super(GitLfs, self).update_cache_repo(*refs)

        self.execute(
            "lfs", "fetch", "origin", *refs,
            cwd=self.cache_repo_dir, logger_name=self._repo_name + "_mirror"
        )

    def _clone_from_cache_repo(self, target_dir, branch=None, *args, **kwargs):
        if branch is None:
            raise ValueError("Branch should be specified")
        super(GitLfs, self)._clone_from_cache_repo(target_dir, branch, *args, **kwargs)
        self.execute("lfs", "checkout", branch, cwd=target_dir)
