from __future__ import absolute_import

import os
import time
import logging
import itertools
import datetime as dt

from sandbox.common import config as common_config
from sandbox.common import context
from sandbox.common import patterns
from sandbox.common import statistics as common_statistics
from sandbox.common import collections as common_collections

import sandbox.common.types.statistics as ctss

from . import svn
from sandbox.sdk2 import task
from sandbox.sdk2 import statistics
from sandbox.sdk2.helpers import process


class Hg(object):
    """ Class for working with mercurial """
    HG_EXECUTABLE = "hg"
    LOG_PREFIX = "hg"

    @patterns.singleton_classproperty
    def _hg_executable(cls):
        if common_config.Registry().client.sdk.hg.use_system_binary:
            return cls.HG_EXECUTABLE
        from sandbox.sdk2 import environments
        return os.path.join(environments.SandboxHgEnvironment().prepare(), "hg", cls.HG_EXECUTABLE)

    @classmethod
    def hg(cls, cmd, opts=None, url=None, path=None, raw=False):
        if not opts:
            opts = ()
        cmd_args = [arg for arg in (url, path) if arg is not None]
        cmd_line = list(itertools.chain((cls._hg_executable,), (cmd,), opts)) + cmd_args
        now = time.time()
        utcnow = dt.datetime.utcnow()
        pl = (
            context.NullContextmanager(
                enter_obj=common_collections.AttrDict(stdout=process.subprocess.PIPE, stderr=process.subprocess.PIPE)
            )
            if raw else
            process.ProcessLog(task=task.Task.current, logger="_".join((cls.LOG_PREFIX, cmd)))
        )
        try:
            with statistics.measure_time("hg"), statistics.measure_time("hg_{}".format(cmd)):
                with pl as pl:
                    proc = process.subprocess.Popen(cmd_line, stdout=pl.stdout, stderr=pl.stderr)
                    if raw:
                        return proc
                    pl.raise_for_status(proc)
        except process.subprocess.CalledProcessError as error:
            raise svn.SvnError("Cannot execute {}. Error: {}".format(cmd_line, error))
        finally:
            if common_statistics.Signaler.instance is not None:
                duration = int((time.time() - now) * 1000)
                common_statistics.Signaler().push(dict(
                    type=ctss.SignalType.TASK_OPERATION,
                    kind=ctss.OperationType.HG,
                    date=utcnow,
                    timestamp=utcnow,
                    method=cmd.lower(),
                    duration=duration,
                ))

    @classmethod
    def id(cls, url):
        """
        Get revision hash for given url
        """
        p = cls.hg("id", ["-i"], url=url, raw=True)
        out, err = p.communicate()
        logging.debug("hg id stdout: %s", out)
        if err:
            logging.debug("hg id stderr: %s", err)
            err = err.splitlines()[0]
        if p.returncode:
            raise svn.SvnError("Cannot get revision for {}: {}".format(url, err))
        return out.strip()
