from __future__ import absolute_import

import os
import re
import sys
import time
import atexit
import shutil
import socket
import hashlib
import logging
import datetime as dt
import tempfile
import itertools
import threading
import subprocess as sp
import collections

import six
from six.moves import urllib
from six.moves.urllib import parse as urlparse

import xml.etree.ElementTree
from xml.parsers import expat

from sandbox.common import fs
from sandbox.common import enum
from sandbox.common import urls as common_urls
from sandbox.common import share as common_share
from sandbox.common import errors as common_errors
from sandbox.common import config as common_config
from sandbox.common import system as common_system
from sandbox.common import patterns
from sandbox.common import itertools as common_itertools
from sandbox.common import statistics as common_statistics

from sandbox.common.vcs import cache as vcs_cache
from sandbox.common.types import misc as ctm
from sandbox.common.types import client as ctc
from sandbox.common.types import statistics as ctst

import sandbox.agentr.errors as agentr_errors

from . import tunnel
from . import zipatch

logger = logging.getLogger("vcs.svn")


class SvnError(common_errors.SandboxException):
    ERROR_CODE_RE = re.compile(r"^svn: (?P<errno>E\d+):", flags=re.MULTILINE)

    @property
    def error_code(self):
        m = self.ERROR_CODE_RE.search(str(self))
        if m:
            return m.group("errno")
        return None


class SvnPermanentError(SvnError, common_errors.TaskError):
    """ Non-retriable svn error """


class SvnTemporaryError(SvnError, common_errors.TemporaryError):
    """ Retryable svn error """


class SvnIncorrectUrl(SvnPermanentError):
    """ SVN url cannot be parsed """


class SvnUserRequired(SvnPermanentError):
    """ SVN command requires that user has been specified """


class SvnPathNotExists(SvnPermanentError):
    """ SVN path does not exist """


class SvnInvalidParameters(SvnPermanentError):
    """ Parameter(s) of SVN command is(are) not valid """


class SvnTimeout(SvnTemporaryError):
    """ Timeout during svn operation """


def _lazy_setting(name):
    @patterns.classproperty
    def prop(cls):
        value = common_config.Registry().root()
        for item in name.split("."):
            value = getattr(value, item)
        return value
    return prop


def _lines_to_text(lines):
    return "".join(line.rstrip() + "\n" for line in lines)


class SvnOutput(object):
    def __init__(self):
        self._messages = []

    def update(self, line):
        self._messages.append(six.ensure_str(line))

    def __str__(self):
        return _lines_to_text(self._messages)


class SvnOutputSummary(object):
    """
    Filter for creating short summary of svn update/export/checkout commands.

    A few words about format (from `svn help update`):
    ```
    For each updated item a line will be printed with characters reporting
    the action taken. These characters have the following meaning:

      A  Added
      D  Deleted
      U  Updated
      C  Conflict
      G  Merged
      E  Existed
      R  Replaced

    Characters in the first column report about the item itself.
    Characters in the second column report about properties of the item.
    A 'B' in the third column signifies that the lock for the file has
    been broken or stolen.
    A 'C' in the fourth column indicates a tree conflict, while a 'C' in
    the first and second columns indicate textual conflicts in files
    and in property values, respectively.
    ```

    """
    ACTION_CODES = {"A", "D", "U", "C", "G", "E", "R", "B", " "}  # " " -- empty action marker

    def __init__(self):
        self._messages = []
        self._actions = collections.Counter()

    def update(self, line):
        # for an updated item first 4 characters are action markers
        line = six.ensure_str(line)
        columns = line[:4]
        # check if all characters are indeed action markers
        if set(columns).issubset(self.ACTION_CODES):
            self._actions[columns] += 1
        else:
            self._messages.append(line)

    def __str__(self):
        lines = list(self._messages) if self._messages else []

        if self._actions:
            for action, count in self._actions.most_common():
                lines.append("{} -- {} items".format(action, count))
            lines.append("Total {} items".format(sum(list(self._actions.values()))))

        return _lines_to_text(lines)


class SvnOutputFilter(logging.Filter):
    def __init__(self, summary=False):
        super(SvnOutputFilter, self).__init__("")
        self._stdout = SvnOutputSummary() if summary else SvnOutput()
        self._stderr = SvnOutput()

    def filter(self, record):
        if record.levelno == logging.DEBUG:
            self._stdout.update(record.msg)
            return False

        if record.levelno == logging.ERROR:
            self._stderr.update(record.msg)

        return True

    @property
    def stdout(self):
        return str(self._stdout)

    @property
    def stderr(self):
        return str(self._stderr)


class SvnCommandResult(object):
    def __init__(self, args, returncode, stdout, stderr):
        self.args = args
        self.returncode = returncode
        self.stdout = stdout
        self.stderr = stderr

    def __str__(self):
        return "Command {} returned code {}\n=== stdout:\n{}\n=== stderr:\n{}\n".format(
            self.args, self.returncode, self.stdout.strip(), self.stderr.strip()
        )


class Svn(object):
    """ Class for working with subversion """
    LOG_PREFIX = "svn"
    SVN_EXECUTABLE = "svn"
    RETRYABLE_ERROR_CODES = ("E210002", "E670005", "E220003", "E170001", "E200030", "E200014")
    SVN_DEFAULT_TIMEOUT = 60
    SVN_STATUS_TIMEOUT = 1800  # svn status can be long for large arguments

    class EntryKind(enum.Enum):
        FILE = "file"
        DIRECTORY = "dir"

    class Depth(enum.Enum):
        EMPTY = "empty"
        FILES = "files"
        IMMEDIATES = "immediates"
        INFINITY = "infinity"
        EXCLUDE = "exclude"

    class SvnOptions(object):
        def __init__(self):
            self.__opts = {
                "--non-interactive": None,
                "--trust-server-cert": None,
                "--config-dir": common_config.Registry().client.sdk.svn.confdir
            }

        def __iter__(self):
            for (k, v) in six.iteritems(self.__opts):
                if v is None:
                    yield k
                elif isinstance(v, six.string_types):
                    yield "=".join((k, v))
                elif isinstance(v, list):
                    for e in ("=".join((k, sv)) for sv in v):
                        yield e

        @staticmethod
        def __bool_prop(name):
            def getter(self):
                return name in self.__opts

            def setter(self, value):
                if value:
                    self.__opts[name] = None
                return value

            return property(getter, setter)
        __bool_prop = __bool_prop.__func__

        @staticmethod
        def __str_prop(name):
            def getter(self):
                return self.__opts.get(name)

            def setter(self, value):
                if value:
                    self.__opts[name] = str(value)
                return value

            return property(getter, setter)
        __str_prop = __str_prop.__func__

        @staticmethod
        def __list_prop(name):
            def getter(self):
                return self.__opts.get(name)

            def setter(self, value):
                if value:
                    self.__opts[name] = list(str(v) for v in value)
                return value

            return property(getter, setter)
        __list_prop = __list_prop.__func__

        @staticmethod
        def __depth_prop(name):
            def check(value):
                if value not in Svn.Depth:
                    raise SvnInvalidParameters("Incorrect {} parameter value '{}'.".format(name, value))
                return value

            def getter(self):
                return self.__opts.get(name)

            def setter(self, value):
                if value:
                    self.__opts[name] = check(value)
                return value

            return property(getter, setter)
        __depth_prop = __depth_prop.__func__

        # bool options
        recursive = __bool_prop("--recursive")
        parents = __bool_prop("--parents")
        no_ignore = __bool_prop("--no-ignore")
        ignore_externals = __bool_prop("--ignore-externals")
        force = __bool_prop("--force")
        verbose = __bool_prop("--verbose")
        xml = __bool_prop("--xml")
        relocate = __bool_prop("--relocate")
        stop_on_copy = __bool_prop("--stop-on-copy")
        patch_compatible = __bool_prop("--patch-compatible")
        summarize = __bool_prop("--summarize")
        ignore_ancestry = __bool_prop("--ignore-ancestry")

        # str options
        changelist = __str_prop("--changelist")
        message = __str_prop("--message")
        revision = __str_prop("--revision")
        limit = __str_prop("--limit")
        change = __str_prop("--change")
        with_revprop = __list_prop("--with-revprop")

        # depth options
        depth = __depth_prop("--depth")
        set_depth = __depth_prop("--set-depth")

    @classmethod
    def is_temporary_error(cls, message):
        return message and any(code in message for code in cls.RETRYABLE_ERROR_CODES)

    @patterns.singleton_classproperty
    def _svn_executable(cls):
        if common_config.Registry().client.sdk.svn.use_system_binary:
            return cls.SVN_EXECUTABLE
        from sandbox.sdk2 import environments
        return environments.SvnEnvironment().prepare()

    @classmethod
    def svn(
        cls, cmd,
        opts=None, url=None, path=None,
        timeout=None, check=True, svn_ssh=None, summary=False
    ):
        """
        Call svn with right command line

        :param cmd: svn subcommand, i.e. "info", "log", etc.
        :param opts: options which will be passed to svn subcommand
        :type opts: Svn.SvnOptions
        :param url: url or path to local repository
        :param path: path to local repository entry
        :param timeout: operation timeout; if None, SVN_DEFAULT_TIMEOUT will be used
        :param check: if True, raise if svn call return code is non-zero
        :param svn_ssh: url to establish ssh multiplexing tunnel to; if None, `url` param will be used
        :param summary: if True, produce additional summary into svn call output

        :return: svn process return code, stdout and stderr
        :rtype: SvnCommandResult
        """
        if not opts:
            opts = cls.SvnOptions()

        url = common_itertools.as_list(url)
        path = common_itertools.as_list(path)
        cmd_line = list(six.moves.map(
            six.ensure_str,
            common_itertools.as_list([cls._svn_executable, cmd], opts, url, path)
        ))

        if svn_ssh is None and url:
            svn_ssh = url[0]  # Try first url to choose a tunnel

        tun = None
        identity_file = os.path.expanduser(os.path.join("~", ".ssh", "id_rsa"))

        if svn_ssh:
            tun = tunnel.ensure_ssh_multiplexing_tunnel(svn_ssh, identity_file)

        env = os.environ.copy()
        env["LC_ALL"] = "en_US.UTF-8"
        env["SVN_SSH"] = (
            env.get("SVN_SSH", "ssh") +
            " -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null" +
            (" -o ControlPath={}".format(tun.control_path) if tun else "") +
            (" -i {}".format(identity_file) if os.path.exists(identity_file) else "")
        )
        # Special environment variable, which can be used by `arcadia.yandex.ru`
        # to optimize load profiles and quotas on remote side.
        cmd_line_str = env["SSH_SANDBOX_SVNCOMMAND"] = " ".join(cmd_line)

        # Windows supports only shell-like execution
        shell = common_config.Registry().this.system.family == ctm.OSFamily.CYGWIN
        if shell:
            cmd_line = cmd_line_str

        svn_output = SvnOutputFilter(summary)
        svn_timeouted = False

        # Logger is used to intercept subprocess output, so create a per-thread unique name,
        # because we don't want several threads to have the same logger instance.
        svn_logger = logger.getChild("{}-{}".format(cls.LOG_PREFIX, threading.current_thread().ident))
        svn_logger.setLevel(logging.DEBUG)
        svn_logger.addFilter(svn_output)

        from sandbox.sdk2 import statistics
        now = time.time()
        utcnow = dt.datetime.utcnow()
        try:
            with statistics.measure_time("svn"), statistics.measure_time("svn_{}".format(cmd)):
                from sandbox.sdk2.helpers import ProcessLog, subprocess
                with ProcessLog(logger=svn_logger, stdout_level=logging.DEBUG) as pl:
                    p = subprocess.Popen(
                        cmd_line,
                        stdout=pl.stdout, stderr=pl.stderr,
                        shell=shell, env=env,
                    )
                    try:
                        # both stdout and stderr are always None
                        p.communicate(timeout=timeout)
                    except subprocess.TimeoutExpired:
                        p.kill()
                        p.communicate()
                        svn_timeouted = True
                    except:
                        p.kill()
                        p.wait()
                        raise

                result = SvnCommandResult(p.args, p.returncode, svn_output.stdout, svn_output.stderr)

                if cls.is_temporary_error(result.stderr):
                    raise SvnTemporaryError("Temporary error:\n{}".format(result))

                if svn_timeouted:
                    raise SvnTimeout("Svn operation was interrupted by timeout {}\n{}".format(timeout, result))

                if check and p.returncode:
                    raise SvnError("Fatal error:\n{}".format(result))

            return result

        finally:
            if common_statistics.Signaler.instance is not None:
                duration = int((time.time() - now) * 1000)
                common_statistics.Signaler().push(dict(
                    type=ctst.SignalType.TASK_OPERATION,
                    kind=ctst.OperationType.SVN,
                    date=utcnow,
                    timestamp=utcnow,
                    method=cmd.lower(),
                    duration=duration,
                ))

            # loggers are cached permanently, remove our "temporary" one
            svn_logger.manager.loggerDict.pop(svn_logger.name)

    @classmethod
    def info(cls, url, timeout=None, look_parent=True):
        """
        Get info about svn url or local repository path

        :param url: svn path or path to local repository
        :param timeout: operation timeout; if None, SVN_DEFAULT_TIMEOUT will be used
        :return: dict with keys:

            - entry_path
            - entry_revision
            - entry_kind (EntryKind.FILE or EntryKind.DIRECTORY)
            - url
            - commit_revision
            - author
            - date
            - depth

        :rtype: dict
        """
        if timeout is None:
            timeout = cls.SVN_DEFAULT_TIMEOUT

        opts = cls.SvnOptions()
        opts.xml = True

        r = cls.svn("info", opts, url=url, timeout=timeout, check=False)
        parsed_scheme = urlparse.urlparse(url).scheme
        if sys.platform == "win32" and parsed_scheme and url.startswith("".join([parsed_scheme, ":", os.sep])):
            parsed_scheme = ""

        try:
            if r.stderr:
                if not parsed_scheme and look_parent:
                    if "E200009" in r.stderr and "W155010" in r.stderr:
                        if not url or url == "/":
                            return
                        dir_path, file_name = os.path.dirname(url), os.path.basename(url)
                        result = cls.info(dir_path)
                        if result:
                            result["entry_path"] = os.path.join(result["entry_path"], file_name)
                            result["url"] = "/".join((result["url"], file_name))
                            data = cls.info(result["url"], look_parent=look_parent)
                            if data:
                                result["entry_revision"] = data["entry_revision"]
                                result["entry_kind"] = data["entry_kind"]
                            else:
                                result.pop("entry_revision", None)
                                result.pop("entry_kind", None)

                        logger.debug("Svn info result: %r", result)
                        return result

                for err_code in ("E200009", "E160006"):
                    if err_code in r.stderr:
                        # url not found
                        return

            if r.returncode:
                raise SvnError("Cannot get info for \"{}\"\n{}".format(url, r))

            entry = xml.etree.ElementTree.fromstring(r.stdout).find("entry")
            result = {
                "entry_path": entry.attrib["path"],
                "entry_revision": entry.attrib["revision"],
                "entry_kind": entry.attrib["kind"],
                "url": entry.find("url").text,
            }
            repository = entry.find("repository")
            if repository is not None:
                result["repository_root"] = repository.find("root").text
            commit = entry.find("commit")
            if commit is not None:
                result["commit_revision"] = commit.attrib["revision"]
                result["author"] = commit.find("author").text
                result["date"] = commit.find("date").text
            wc_info = entry.find("wc-info")
            if wc_info is not None:
                wcroot_abspath = wc_info.find("wcroot-abspath")
                if wcroot_abspath is not None:
                    result["wcroot_abspath"] = wcroot_abspath.text
                depth = wc_info.find("depth")
                if depth is not None:
                    result["depth"] = depth.text

            logger.debug("Svn info result: %r", result)
            return result

        except (xml.etree.ElementTree.ParseError, expat.ExpatError, AttributeError) as error:
            logger.exception("Fail to get svn info")
            raise SvnError("Cannot parse or get xml info for URL {}, error: {}\n{}".format(url, error, r))

    @classmethod
    def checkout(cls, url, path, depth=None, revision=None, force=True):
        """
        Checkout repository to local path

        :param url: arcadia url from which do checkout
        :param path: path to directory on disk
        :param depth: add --depth to the svn command line, possible values:

            - empty
            - files
            - infinity
            - immediates

        :param revision: svn revision
        :param force: add --force to the svn command line
        :return: path to local copy
        """
        opts = cls.SvnOptions()
        opts.depth = depth
        opts.revision = revision
        opts.force = force

        r = cls.svn("checkout", opts, url=url, path=path, summary=True)
        logger.debug("Svn checkout summary:\n%s", r.stdout.strip())

    @classmethod
    def update(cls, path, depth=None, set_depth=None, revision=None, ignore_externals=False, parents=False, force=True):
        """
        Update local repository copy

        :param path: path to the local copy
        :param revision: svn revision to which do update
        :param depth: add --depth to the svn command line, possible values:

            - empty
            - files
            - infinity
            - immediates

        :param set_depth: add --set_depth to the svn command line, possible values:

            - empty
            - files
            - infinity
            - immediates

        :param ignore_externals: add --ignore_externals to the svn command line
        :param parents: make intermediate directories (add --parents to the svn command line)
        :param force: add --force to the svn command line
        :return: path to local copy
        """
        opts = cls.SvnOptions()
        opts.depth = depth
        opts.set_depth = set_depth
        opts.revision = revision
        opts.ignore_externals = ignore_externals
        opts.parents = parents
        opts.force = force

        info = cls.info(path)  # Get svn_ssh url to find out which tunnel to use
        r = cls.svn("update", opts, path=path, svn_ssh=info["repository_root"], summary=True)
        logger.debug("Svn update summary:\n%s", r.stdout.strip())

    @classmethod
    def export(cls, url, path, force=True, depth=None):
        """
        Export subversion url to the path

        :param url: subversion path to item
        :param path: full path (with item name)
        :param force: add --force to the svn command line
        :param depth: add --depth to the svn command line, possible values:

            - empty
            - files
            - infinity
            - immediates
        """
        opts = cls.SvnOptions()
        opts.force = force
        opts.depth = depth

        r = cls.svn("export", opts, url=url, path=path, summary=True)
        logger.debug("Svn export summary:\n%s", r.stdout.strip())

    @classmethod
    def cleanup(cls, path):
        """
        Do svn cleanup in path

        :param path: checkouted svn directory
        :return: svn process return code
        """
        path = os.path.abspath(path)
        cls.svn("cleanup", path=path)

    @classmethod
    def upgrade(cls, path):
        """
        Do svn upgrade in path

        :param path: checkouted svn directory
        :return: svn process return code
        """
        path = os.path.abspath(path)
        cls.svn("upgrade", path=path)

    @classmethod
    def list(cls, url, depth=None, as_list=False):
        """
        Recursively get content of svn directory

        :param url: svn path to directory
        :param depth: add --depth to the svn command line, possible values:

            - empty
            - files
            - infinity
            - immediates

        :param as_list: return as list or as str with /n
        :return: content of directory
        :rtype: str
        """
        opts = cls.SvnOptions()
        opts.depth = depth

        r = cls.svn("list", opts, url=url, check=False)

        if as_list:
            return r.stdout.split()
        return r.stdout

    @classmethod
    def revert(cls, path, recursive=True, depth=None, changelist=None):
        """
        Do svn revert

        :param path: path to the local copy
        :param recursive: do recursively revert, equal to depth='infinity'
        :param depth: add --depth to the svn command line, possible values:

            - empty
            - files
            - infinity
            - immediates

        :param changelist: apply only to the certain items
        :return: path to local copy
        """
        path = os.path.abspath(path)
        logger.info('Revert svn repo %s', path)
        opts = cls.SvnOptions()
        opts.recursive = recursive
        opts.depth = depth
        opts.changelist = changelist
        cls.svn("revert", opts, path=path)
        return path

    @classmethod
    def add(cls, path, parents=False, force=False):
        """
        Do svn add

        :param path: path to add
        :param parents: add --parents to the svn command line
        :param force: add --force to the svn command line
        """
        opts = cls.SvnOptions()
        opts.parents = parents
        opts.force = force

        cls.svn("add", opts=opts, path=path)

    @classmethod
    def delete(cls, url, message=None, user=None):
        """
        Delete item(s) from repository

        :param url: url (or list of urls) of item(s) for deletion
        :type url: str, list, tuple
        :param message: log message
        :type message: str
        :param user: svn user
        :type user: str
        :return: url(s) of deleted item(s)
        :rtype: str, list, tuple
        """
        opts = cls.SvnOptions()
        opts.message = message
        cls.svn("delete", opts, url=url)
        return url

    @classmethod
    def status(cls, path, depth=None, no_ignore=False, changelist=None, timeout=None):
        """
        Get info about status of objects in working copy of repository

        :param path: repository path
        :param depth: add --depth to the svn command line, possible values:

            - empty
            - files
            - infinity
            - immediates

        :param no_ignore: add --no-ignore to the svn command line
        :param changelist: apply only to the certain items
        :param timeout: operation timeout; if None, SVN_STATUS_TIMEOUT will be used
        :return: output of command
        :rtype: str
        """
        if timeout is None:
            timeout = cls.SVN_STATUS_TIMEOUT

        opts = cls.SvnOptions()
        opts.depth = depth
        opts.no_ignore = no_ignore
        opts.changelist = changelist

        r = cls.svn("status", opts, path=path, check=False, timeout=timeout)
        return r.stdout

    @classmethod
    def clean(cls, path):
        """
        Clean repository from unnecessary files

        :param path: repository path
        :return: list of removed items
        :rtype: list
        """
        result = []
        path = os.path.abspath(path)
        repo_status = cls.status(path, no_ignore=True)
        for item in repo_status.split("\n"):
            if item:
                item_info = item[:7]  # the first 7 columns contain SVN status information
                if item_info[0] in ("?", "I"):
                    item_path = item[7:].strip()
                    fs.remove_path(item_path)
                    result.append(item_path)
        return result

    @classmethod
    def commit(cls, path, message, with_revprop=None):
        """
        Commit changes in path(s)

        .. note:: If there is nothing to commit this method does nothing and
            returns an empty string (empty stdout from `svn commit` command)

        :param path: repository path(s)
        :param message: log message
        :param with_revprop: revision property
        """
        path = common_itertools.chain(path)
        path = [os.path.abspath(p) for p in path]
        for p in path:
            if not os.path.exists(p):
                raise SvnPathNotExists("SVN repo {} does not exist".format(p))
        opts = cls.SvnOptions()
        opts.message = message
        if with_revprop:
            opts.with_revprop = with_revprop
        r = cls.svn("commit", opts, path=path)
        return r.stdout

    @classmethod
    def blame(cls, url, verbose=False):
        """
        Get output of `svn blame`
        (lines of the form: revision author code)

        :param url: svn path to item
        :param verbose: add --verbose to the svn command line
        :return: content of the item
        """
        opts = cls.SvnOptions()
        opts.verbose = verbose
        r = cls.svn("blame", opts, url=url, check=False)
        return r.stdout

    @classmethod
    def cat(cls, url, revision=None):
        """
        Get output of file from url

        :param url: svn path to file
        :param revision: svn revision
        :return: content of the file
        """
        opts = cls.SvnOptions()
        opts.revision = revision
        r = cls.svn("cat", opts, url=url, check=False)
        return r.stdout

    @classmethod
    def log(
        cls, url,
        revision_from=None, revision_to=None,
        track_copy=False, limit=None, stop_on_copy=False, fullpath=False, timeout=None
    ):
        """
        Get history of changes

        :param url: svn url
        :param revision_from: beginning revision
        :param revision_to: ending revision
        :param track_copy: track path copies
        :param limit: maximum number of log entries
        :param stop_on_copy: do not cross copies while traversing history
        :param fullpath:
        :param timeout: operation timeout; if None, SVN_DEFAULT_TIMEOUT will be used
        :return: list of dicts with keys:

            - revision
            - author
            - date
            - msg
            - paths - a list of pairs (action, path) or dict depends on fullpath flag

        :rtype: list
        """
        if timeout is None:
            timeout = cls.SVN_DEFAULT_TIMEOUT

        opts = cls.SvnOptions()
        opts.xml = True
        opts.verbose = True

        revision = None
        if revision_from is not None:
            revision = str(revision_from)
        if revision_to is not None:
            if not revision:
                raise ValueError("'revision_to' could be defined only with 'revision_from'")
            revision = ":".join([revision, str(revision_to)])
        opts.revision = revision

        opts.limit = limit
        opts.stop_on_copy = stop_on_copy

        r = cls.svn("log", opts, url=url, timeout=timeout)

        def get_path(path, fullpath):
            if not fullpath:
                return path.attrib["action"], path.text
            else:
                result_dict = path.attrib
                result_dict["text"] = path.text
                return result_dict

        result = []
        try:
            log = xml.etree.ElementTree.fromstring(r.stdout)
            for log_entry in log:
                msg_node = log_entry.find("msg")

                # this is because parsing empty element with xml module produces <None> text node,
                # see RMINCIDENTS-166
                msg_node_text = ""
                if msg_node is not None and msg_node.text is not None:
                    msg_node_text = msg_node.text

                entry = {
                    "revision": int(log_entry.attrib["revision"]),
                    "author": log_entry.find("author").text,
                    "date": dt.datetime.strptime(log_entry.find("date").text, "%Y-%m-%dT%H:%M:%S.%fZ"),
                    "msg": msg_node_text,
                    "paths": [get_path(path, fullpath) for path in log_entry.find("paths")],
                }
                if track_copy:
                    entry["copies"] = []
                    for path in log_entry.find("paths"):
                        if "copyfrom-path" not in path.attrib:
                            continue
                        assert "copyfrom-rev" in path.attrib
                        entry["copies"].append((path.attrib["copyfrom-path"], path.attrib["copyfrom-rev"], path.text))
                result.append(entry)
        except (expat.ExpatError, AttributeError, ValueError) as error:
            logger.exception("Cannot parse xml log output for %s", url)
            raise SvnError("Cannot parse xml log output for {}, error: {}\n{}".format(url, error, r))

        return result

    @classmethod
    def copy(cls, src, dst, message, revision=None, parents=False):
        """
        Copy svn item

        :param src: source svn url or local repository path
        :param dst: destination svn url or local repository path
        :param message: log message
        :param revision: revision on source_path
        :param parents: add --parents to the svn command line
        :return: svn process return code, stdout and stderr
        """
        opts = cls.SvnOptions()
        opts.message = message
        opts.parents = parents
        opts.revision = revision
        return cls.svn("cp", opts, url=[src, dst])

    @classmethod
    def mkdir(cls, url, message=None, parents=False):
        """
        Create svn directory

        :param url: svn url of directory to create
        :param message: log message
        :param parents: add --parents to the svn command line
        """
        opts = cls.SvnOptions()
        opts.message = message
        opts.parents = parents
        cls.svn("mkdir", opts, url=url)

    @classmethod
    def diff(cls, url, change=None, summarize=False):
        """
        Get diff from svn url or local repository

        :param url: svn path or path to local repository
        :type url: str
        :param change: revision number
        :type change: int, None
        :param summarize: add --summarize to svn call
        :type summarize: bool
        :return: diff content
        :rtype: str
        """
        opts = cls.SvnOptions()
        opts.patch_compatible = True

        if change:
            opts.change = change

        if summarize:
            opts.summarize = True

        r = cls.svn("diff", opts, url=url)
        return r.stdout

    @classmethod
    def merge(cls, url, path, change=None, ignore_ancestry=False):
        """
        Merges changes from url to local path. Cherry-pick type.

        :param url: svn path
        :param path: local path to working copy
        :param change: str or list
        :param ignore_ancestry: bool
        """
        if not path:
            raise SvnPathNotExists("Path of local working copy is not specified")
        if change is not None:
            opts = cls.SvnOptions()
            opts.change = ",".join(map(str, common_itertools.chain(change)))
            if ignore_ancestry:
                opts.ignore_ancestry = True
            cls.svn("merge", opts, url=url, path=path)

    @classmethod
    def propget(cls, prop, path, depth=None, recursive=False):
        """
        Do svn propget

        :param prop: prop to get
        :param path: path to get prop from
        :param recursive: do recursively, equal to depth='infinity'
        :param depth: add --depth to the svn command line, possible values:

            - empty
            - files
            - infinity
            - immediates

        :return: svn propget output
        :rtype: list of strings
        """
        opts = cls.SvnOptions()
        opts.depth = depth
        opts.recursive = recursive

        r = cls.svn("propget", opts, path=[prop, path])
        return r.stdout.splitlines()


@six.add_metaclass(patterns.Api)
class Arcadia(vcs_cache.CacheableArcadia, Svn):
    """
    Class for working with Arcadia subversion repository.
    All methods operate with arcadia svn urls in unified format: `arcadia:/some/path[@revision]`.
    Url in format `svn+ssh://[any_user@][any_host]/some/path[@revision]` is also accepted and
    internally is converted to unified format by method normalize_url(), user and host are ignored.

    Some methods (primarily get_arcadia_src_dir) also accept Arcadia HG urls of the following format:

    `arcadia-hg:/[some/path][#rev]`

    `rev` is either commit hash, branch name or any other commit identifier accepted by hg
    (see https://www.mercurial-scm.org/repo/hg/help/revsets)

    All modifying operations (commit, delete, mkdir, create_tag, etc) should be applied with required
    `user` parameter.
    """

    LOG_PREFIX = "arcadia"
    DEFAULT_SCHEME = "svn+ssh"
    ARCADIA_SCHEME = "arcadia"
    ARCADIA_HG_SCHEME = "arcadia-hg"
    ARCADIA_ARC_SCHEME = "arcadia-arc"
    ARCADIA_RW = "arcadia.yandex.ru"
    ARCADIA_RO = "arcadia-ro.yandex.ru"
    ARCADIA_HG = "ssh://zomb-sandbox-rw@arcadia-hg.yandex-team.ru/arcadia.hg"
    RW_USER = _lazy_setting('client.sdk.svn.arcadia.rw.user')
    RO_USER = _lazy_setting('client.sdk.svn.arcadia.ro.user')
    ARCADIA_BASE_URL = "arcadia:/arc"
    ARCADIA_TRUNK_URL = "arcadia:/arc/trunk/arcadia"
    EXCLUDED_DIRECTORIES = {"junk", "frontend", "mobile"}

    ArcadiaURL = collections.namedtuple(
        "ArcadiaURL", ["path", "revision", "branch", "tag", "trunk", "subpath"]
    )

    #: space requirement for Arcadia trunk cache update, MB;
    #: if there's less free space, a `sandbox.common_errors.TemporaryError` will be raised
    #: to avoid cache corruption and removal
    SAFE_TRUNK_UPDATE_SPACE = 1 << 10

    @patterns.singleton_classproperty
    def _strace(cls):
        if not getattr(common_config.Registry().client.sdk.svn.arcadia, "run_strace", False):
            return
        from sandbox import sdk2
        output_path = sdk2.Task.current.log_path("strace.output.gz")
        cls.SVN_EXECUTABLE = "strace -t -s 0 -o |gzip>{} {}".format(output_path, cls.SVN_EXECUTABLE)

    @patterns.singleton_classproperty
    def _tcpdump(cls):
        if not getattr(common_config.Registry().client.sdk.svn.arcadia, "run_tcpdump", False):
            return
        iface = None
        ipv6_addr = socket.getaddrinfo(socket.getfqdn(), 0, socket.AF_INET6)[0][4][0]
        ifconfig_output = sp.check_output("ifconfig")
        last_iface = None
        for line in ifconfig_output.split("\n"):
            if line.startswith(" ") or line.startswith("\t"):
                if ipv6_addr in line and last_iface:
                    iface = last_iface
                    break
            elif line:
                last_iface = line.split()[0].rstrip(":")
        if not iface:
            logger.error("Cannot to get network interface to listen for by tcpdump")
            return
        addrs = [cls.ARCADIA_RW, cls.ARCADIA_RO]
        fd, fifo_path = tempfile.mkstemp()
        os.close(fd)
        os.unlink(fifo_path)
        os.mkfifo(fifo_path)
        from sandbox import sdk2
        from sandbox.sdk2.helpers import process
        output_path = sdk2.Task.current.log_path("tcpdump.pcap.gz")
        pl = process.ProcessLog(logger="tcpdump")
        pl_gzip = process.ProcessLog(logger="tcpdump_gzip")
        process.subprocess.Popen(
            ["cat", fifo_path, "|", "gzip>{}".format(output_path)],
            stdout=pl_gzip.stdout, stderr=pl_gzip.stderr
        )
        p = process.subprocess.Popen(list(itertools.chain(
            [
                "tcpdump",
                "-s", "60",
                "-i", iface,
                "-w", output_path,
                "host"
            ], " or ".join(addrs).split()
        )), stdout=pl.stdout, stderr=pl.stderr)
        atexit.register(lambda: (
            pl.__exit__(None, None, None),
            pl_gzip.__exit__(None, None, None),
            p.kill(),
            os.unlink(fifo_path)
        ))

    @classmethod
    def svn(cls, *args, **kws):
        _ = cls._tcpdump  # noqa
        _ = cls._strace  # noqa
        return super(Arcadia, cls).svn(*args, **kws)

    @classmethod
    def __parse(cls, url):
        """
        Parse url

        :param url: svn url or path
        :return: parsed url or None if it is local path
        """
        parsed_url = urlparse.urlparse(url)
        if parsed_url.scheme:
            return parsed_url

    @classmethod
    def normalize_url(cls, url):
        """
        Convert url to unified arcadia url with format: `arcadia:/some/path[@revision]`

        :param url: svn url
        :return: arcadia url or input url if it is not svn url
        """
        parsed_url = cls.__parse(url)
        if not parsed_url:
            return url
        path = parsed_url.path
        if parsed_url.scheme in (cls.ARCADIA_SCHEME, cls.ARCADIA_HG_SCHEME) and parsed_url.netloc:
            raise common_errors.TaskFailure("Arcadia URL should not contain network location")
        while "//" in path:
            path = path.replace("//", "/")

        if parsed_url.scheme in (cls.ARCADIA_HG_SCHEME, cls.ARCADIA_ARC_SCHEME):
            scheme = parsed_url.scheme
        else:
            scheme = cls.ARCADIA_SCHEME

        return urlparse.urlunparse(parsed_url._replace(scheme=scheme, netloc=None, path=path))

    @classmethod
    def _get_ro_url(cls, url):
        if not cls.__parse(url):
            return url

        # Fix to allow not use tunnel on local Sandbox
        if common_config.Registry().client.sdk.svn.arcadia.force_use_rw:
            return cls._get_rw_url(url)

        url = cls.normalize_url(url)
        netloc = "@".join((cls.RO_USER, cls.ARCADIA_RO))
        parsed_url = urlparse.urlparse(url)._replace(scheme=cls.DEFAULT_SCHEME, netloc=netloc)
        return urlparse.urlunparse(parsed_url)

    @classmethod
    def _get_rw_url(cls, url, user=None):
        if not cls.__parse(url):
            return url
        if user is None:
            user = cls.RO_USER
        url = cls.normalize_url(url)
        if user:
            netloc = "@".join((user, cls.ARCADIA_RW))
        else:
            netloc = cls.ARCADIA_RW
        parsed_url = urlparse.urlparse(url)._replace(scheme=cls.DEFAULT_SCHEME, netloc=netloc)
        return urlparse.urlunparse(parsed_url)

    @classmethod
    def _get_hg_url(cls, url):
        parse = cls.__parse(url)
        if not parse or parse.scheme != cls.ARCADIA_HG_SCHEME:
            return None
        parsed_url = urlparse.urlparse(cls.ARCADIA_HG)._replace(fragment=parse.fragment)
        return urlparse.urlunparse(parsed_url)

    @classmethod
    def _is_ro(cls, url):
        parsed_url = urlparse.urlparse(url)
        netloc = parsed_url.netloc.split("@", 1)[-1]
        return parsed_url.scheme == cls.DEFAULT_SCHEME and netloc == cls.ARCADIA_RO

    @classmethod
    def _is_rw(cls, url):
        parsed_url = urlparse.urlparse(url)
        netloc = parsed_url.netloc.split("@", 1)[-1]
        return parsed_url.scheme == cls.DEFAULT_SCHEME and netloc == cls.ARCADIA_RW

    @classmethod
    def __relocate(cls, url_from, url_to, path):
        opts = cls.SvnOptions()
        opts.relocate = True
        # `svn sw` will talk to `url_to` server, use tunnel for it
        cls.svn("sw", opts, url=[url_from, url_to], path=path, svn_ssh=url_to)

    @classmethod
    def _switch_to_ro(cls, path, info=None):
        info = info or cls.__info(path)
        if not info or cls._is_ro(info["url"]):
            return
        root_path = info["wcroot_abspath"]
        root_url = cls.__info(root_path)["url"]
        ro_root_url = cls._get_ro_url(root_url)
        cls.__relocate(root_url, ro_root_url, root_path)

    @classmethod
    def _switch_to_rw(cls, path, user=None, info=None):
        info = info or cls.__info(path)
        if not info or cls._is_rw(info["url"]) and not user:
            return
        root_path = info["wcroot_abspath"]
        root_url = cls.__info(root_path)["url"]
        rw_root_url = cls._get_rw_url(root_url, user=user)
        cls.__relocate(root_url, rw_root_url, root_path)

    @classmethod
    def svn_url(cls, url):
        """
        Convert arcadia url to svn+ssh
        """
        url = cls.normalize_url(url)
        parsed_url = urlparse.urlparse(url)._replace(scheme=cls.DEFAULT_SCHEME, netloc=cls.ARCADIA_RW)
        return urlparse.urlunparse(parsed_url)

    @classmethod
    def parse_url(cls, url):
        """
        Parse arcadia url

        :param url: arcadia url
        :return: url items
        :rtype: cls.ArcadiaURL
        """
        url_ = cls.normalize_url(url)
        if url != url_:
            logger.debug("Url '%s' is transformed to '%s'", url, url_)
            url = url_
        logger.debug("Parse arcadia url %s", url)
        p = urlparse.urlparse(url)
        # extract revision
        if p.scheme == cls.ARCADIA_HG_SCHEME:
            subpath, revision = p.path, p.fragment or None
            path = "/"
        elif p.scheme == cls.ARCADIA_ARC_SCHEME:
            subpath, revision = p.path, p.fragment or None
            path = "/"
        else:
            path, _, revision = p.path.partition("@")
            if not revision or revision == "HEAD":
                revision = None
            subpath = None

        branch = None
        tag = None
        trunk = False

        while p.scheme not in (cls.ARCADIA_HG_SCHEME, cls.ARCADIA_ARC_SCHEME):
            # extract arcadia branch
            m = re.match(r"/arc/branches/(?P<branch>.*)/arcadia(?P<subpath>/.*)?", path)
            if m:
                branch = m.group("branch")
                subpath = m.group("subpath") or ""
                break
            # extract arcadia tag
            m = re.match(r"/arc/tags/(?P<tag>.*)/arcadia(?P<subpath>/.*)?", path)
            if m:
                tag = m.group("tag")
                subpath = m.group("subpath") or ""
                break
            # check whether url is in arcadia trunk
            m = re.match(r"/arc/trunk/arcadia(?P<subpath>/.*)?", path)
            if m:
                trunk = True
                subpath = m.group("subpath") or ""
                break
            break

        if subpath is not None:
            subpath = subpath.lstrip(os.sep)
        parsed_url = cls.ArcadiaURL(path.lstrip(os.sep), revision, branch, tag, trunk, subpath)
        logger.debug("URL '%s' parsed into %s", url, parsed_url)
        return parsed_url

    @classmethod
    def __replace(cls, url, path=None, revision=None):
        parsed_url = urlparse.urlparse(url)
        if parsed_url.scheme == cls.ARCADIA_HG_SCHEME:
            return cls.__replace_hg(url, path, revision)
        if path is not None:
            parsed_path = parsed_url.path.split("@", 1)
            parsed_new_path = path.split("@", 1)
            if len(parsed_new_path) == 1 and len(parsed_path) > 1:
                path = "@".join((parsed_new_path[0], parsed_path[1]))
            if not path.startswith(os.sep):
                path = os.path.join(os.sep, path)
            parsed_url = parsed_url._replace(path=path)
        if revision is not None:
            new_path = "@".join((parsed_url.path.split("@", 1)[0], str(revision))).rstrip("@")
            parsed_url = parsed_url._replace(path=new_path)
        return urlparse.urlunparse(parsed_url)

    @classmethod
    def __update_trunk_root(cls, path, revision):
        info = cls.info(path)
        opts = cls.__trunk_opts(revision)
        opts.depth = cls.Depth.IMMEDIATES
        r = cls.svn("update", opts, path=path, svn_ssh=info["repository_root"], summary=True)
        logger.debug("Svn update summary immediates:\n%s", r.stdout.strip())

        immediates_directories = set(os.listdir(path))
        directories = list(six.moves.map(
            lambda directory: os.path.join(path, directory), immediates_directories & cls.EXCLUDED_DIRECTORIES
        ))
        if directories:
            opts = cls.__trunk_opts(revision)
            opts.set_depth = cls.Depth.EXCLUDE
            r = cls.svn("update", opts, path=directories, svn_ssh=info["repository_root"], summary=True)
            logger.debug("Svn update excluded directories summary:\n%s", r.stdout.strip())

        directories = list(six.moves.map(
            lambda directory: os.path.join(path, directory), immediates_directories - cls.EXCLUDED_DIRECTORIES
        ))
        if directories:
            opts = cls.__trunk_opts(revision)
            opts.set_depth = cls.Depth.INFINITY
            r = cls.svn("update", opts, path=directories, svn_ssh=info["repository_root"], summary=True)
            logger.debug("Svn update summary:\n%s", r.stdout.strip())

    @classmethod
    def __trunk_opts(cls, revision):
        opts = cls.SvnOptions()
        opts.depth = None
        opts.revision = revision
        opts.ignore_externals = True
        opts.parents = False
        opts.force = True
        return opts

    @classmethod
    def __replace_hg(cls, url, path=None, revision=None):
        parsed_url = urlparse.urlparse(url)
        if path is not None:
            parsed_url = parsed_url._replace(path=path)
        if revision is not None:
            parsed_url = parsed_url._replace(fragment=revision)
        return urlparse.urlunparse(parsed_url)

    @classmethod
    def replace(cls, url, path=None, revision=None):
        """
        Replace component(s) of arcadia url

        :param url: arcadia url
        :param path: modify path, if it is not None
        :param revision: modify revision, if it is not None
        :return: modified url
        :rtype: str
        """
        url = cls.normalize_url(url)
        return cls.__replace(url, path=path, revision=revision)

    @classmethod
    def __info(cls, url, timeout=None, look_parent=True):
        if cls._is_rw(url):
            parsed_url = urlparse.urlparse(url)
            host = parsed_url.netloc.split("@", 1)[-1]
            url = urlparse.urlunparse(
                parsed_url._replace(netloc="@".join((cls.RW_USER, host)))
            )
        return super(Arcadia, cls).info(url, timeout=timeout, look_parent=look_parent)

    @classmethod
    def info(cls, url, timeout=None, look_parent=True, user=None):
        """
        Get info about svn url or local repository path

        :param url: svn path or path to local repository
        :param timeout: operation timeout; if None, SVN_DEFAULT_TIMEOUT will be used
        :param look_parent: if True and url is local path that not exists looking for parent directory
        :return: dict with keys:

            - entry_path
            - entry_revision
            - url
            - commit_revision
            - author
            - date

        :rtype: dict
        """
        info = cls.__info(cls._get_rw_url(url, user), timeout=timeout, look_parent=look_parent)
        if info:
            url = info.get("url")
            if url:
                info["url"] = cls.normalize_url(url)
        return info

    @classmethod
    def __check(cls, url, revision=None, ignore_svn_error=True, ignore_temporary_error=None, look_parent=True):
        if ignore_temporary_error is None:
            ignore_temporary_error = ignore_svn_error
        try:
            return cls.__info(cls.__replace(url, revision=revision), look_parent=look_parent)
        except SvnError as ex:
            if isinstance(ex, SvnTemporaryError) and not ignore_temporary_error or not ignore_svn_error:
                raise
            logger.warning("Ignored SVN exception:", exc_info=True)
            return False

    @classmethod
    def check(cls, url, revision=None, ignore_svn_error=True, look_parent=True, user=None):
        """
        Check existence of arcadia url

        :param url: arcadia url
        :param revision: svn revision
        :param look_parent: if True and url is local path that not exists looking for parent directory
        :return: dict as in `info` method, if url exists, else False
        :rtype: bool|dict
        """
        if not url:
            return False
        return cls.__check(
            cls._get_rw_url(url, user),
            revision=revision,
            ignore_svn_error=ignore_svn_error,
            ignore_temporary_error=False,
            look_parent=look_parent
        )

    @classmethod
    def __get_revision(cls, url):
        info = cls.__info(url)
        if info:
            return info.get("entry_revision")

    @classmethod
    def get_revision(cls, url):
        """
        Get revision (from entry_revision) of arcadia url or local repository path

        :param url: arcadia url or local repository path
        :return: revision or raise exception
        :rtype: str
        """
        url = str(url)
        if url.startswith(cls.ARCADIA_HG_SCHEME):
            from .hg import Hg
            return Hg.id(cls._get_hg_url(url))
        elif url.startswith(cls.ARCADIA_ARC_SCHEME):
            # we cannot parse the excact rev without mounting so let's return it
            # as it is
            return cls.parse_url(url).revision
        else:
            parsed = cls.parse_url(url)
            if parsed.revision and parsed.revision.isdigit():
                return parsed.revision
            return cls.__get_revision(cls._get_rw_url(url))

    @classmethod
    def prepare_checkout(cls, url, path, depth=None, revision=None):
        url = cls.normalize_url(url)
        path = os.path.abspath(str(path))
        parsed_url = cls.parse_url(url)
        if parsed_url.revision and not revision:
            revision = parsed_url.revision

        url_wo_rev = cls.__replace(url, revision="")
        rw_url = cls._get_rw_url(url_wo_rev)
        ro_url = cls._get_ro_url(rw_url)

        logger.info(
            "SVN %s checkout %s %s r%s",
            (depth if depth else ""), url, path, (revision if revision else "HEAD")
        )

        # check RW replica path and revision
        if revision:
            if not cls.__check(rw_url, revision=revision, ignore_svn_error=False):
                raise SvnPathNotExists("Revision {} of {} does not exist".format(revision, rw_url))
        else:
            if not cls.__check(rw_url, ignore_svn_error=False):
                raise SvnPathNotExists("Path {} does not exist".format(rw_url))

        # wait for path in RO
        ro_path_exists, _ = common_itertools.progressive_waiter(
            0, 10, 180, lambda: cls.__check(ro_url, ignore_svn_error=False)
        )
        if not ro_path_exists:
            raise common_errors.TemporaryError(
                "Path {}{} has not been replicated to RO replica for 3 minutes".format(
                    url_wo_rev, " in revision {}".format(revision) if revision else ""
                )
            )
        return ro_url, rw_url, revision, path

    @classmethod
    def ro_has_revision(cls, ro_url, revision):
        # 5s is a p99.99 replication latency (https://st.yandex-team.ru/SANDBOX-4484)
        # try six times with 1 second interval
        for _ in range(5):
            if cls.__check(ro_url, revision=revision, ignore_svn_error=False):
                return True
            time.sleep(1)
        return cls.__check(ro_url, revision=revision, ignore_svn_error=False)

    @patterns.Api.register
    @classmethod
    def checkout(cls, url, path, depth=None, revision=None, force=True):
        """
        Checkout repository to local path

        :param url: arcadia url from which do checkout
        :param path: path to directory on disk
        :param depth: add --depth to the svn command line, possible values:

            - empty
            - files
            - infinity
            - immediates

        :param revision: svn revision
        :param force: add --force to the svn command line
        :return: path to local copy
        """

        ro_url, rw_url, revision, path = cls.prepare_checkout(url, path, depth, revision)

        if revision and cls.ro_has_revision(ro_url, revision):
            # revision exists in RO replica - co from RO to @REV and sw
            super(Arcadia, cls).checkout(cls.__replace(ro_url, revision=revision), path, depth=depth)
            cls.__relocate(ro_url, rw_url, path)
        else:
            # HEAD requested (or there is no such revision in RO replica)  - co from RO to @HEAD, sw and up
            super(Arcadia, cls).checkout(ro_url, path, depth=depth, revision=None, force=force)
            cls.__relocate(ro_url, rw_url, path)
            super(Arcadia, cls).update(path, depth=depth, revision=revision, force=force)

        return path

    @patterns.Api.register
    @classmethod
    def update(cls, path, depth=None, set_depth=None, revision=None, ignore_externals=False, parents=False, force=True):
        """
        Update local repository copy

        :param path: path to the local copy
        :param revision: svn revision to which do update
        :param depth: add --depth to the svn command line, possible values:

            - empty
            - files
            - infinity
            - immediates

        :param set_depth: add --set_depth to the svn command line, possible values:

            - empty
            - files
            - infinity
            - immediates

        :param ignore_externals: add --ignore_externals to the svn command line
        :param parents: add --parents to the svn command line
        :param force: add --force to the svn command line
        :return: path to local copy
        """

        return cls.__update(
            path, depth=depth, set_depth=set_depth, revision=revision,
            ignore_externals=ignore_externals, parents=parents, force=force
        )

    @classmethod
    def __export(cls, url, path, revision=None, force=True, depth=None):
        url = cls.__replace(url, revision=revision)
        super(Arcadia, cls).export(url, path, force=force, depth=depth)

    @classmethod
    def __remove_svn_meta(cls, path):
        shutil.rmtree(os.path.join(path, ".svn"), ignore_errors=True)

    @patterns.Api.register
    @classmethod
    def export(cls, url, path, revision=None, force=True, depth=None):
        """
        Export arcadia url to the path.

        :param url: arcadia path to item
        :param path: full path (with item name)
        :param revision: svn revision
        :param force: add --force to the svn command line
        :param depth: add --depth to the svn command line, possible values:

            - empty
            - files
            - infinity
            - immediates

        :return: path to local copy
        """
        url = cls.normalize_url(url)
        path = os.path.abspath(str(path))
        # some svn commands not working with @REV so we split it
        parsed_url = cls.parse_url(url)
        if parsed_url.revision and not revision:
            revision = parsed_url.revision

        logger.info("SVN Export %s %s %s", url, path, revision if revision else "HEAD")

        url_with_rev = cls.__replace(url, revision=revision)
        info = cls.info(url_with_rev)
        if not info:
            raise SvnPathNotExists("{} does not exist".format(url_with_rev))
        if info["entry_kind"] == Svn.EntryKind.FILE:
            # export svn file from rw
            rw_url = cls._get_rw_url(cls.__replace(url, revision=""))
            cls.__export(rw_url, path, revision=revision, force=force, depth=depth)
        else:
            # do checkout of svn directory and remove metadata
            cls.checkout(url, path, depth=depth, revision=revision, force=force)
            cls.__remove_svn_meta(path)

        return path

    @classmethod
    def list(cls, url, depth=None, as_list=False):
        """
        Recursively get content of svn directory

        :param url: svn path to directory
        :param depth: add --depth to the svn command line, possible values:

            - empty
            - files
            - infinity
            - immediates

        :param as_list: return as list or as str with /n
        :return: content of directory
        :rtype: str
        """
        return super(Arcadia, cls).list(
            cls._get_rw_url(url, user=cls.RW_USER),
            depth=depth,
            as_list=as_list
        )

    @classmethod
    def delete(cls, url, message=None, user=None):
        """
        Delete item(s) from repository

        :param url: url (or list of urls) of item(s) for deletion
        :type url: str, list, tuple
        :param message: log message
        :type message: str
        :param user: svn user
        :type user: str
        :return: url(s) of deleted item(s)
        :rtype: str, list, tuple
        """
        if isinstance(url, six.string_types):
            url = [url]
        is_url = any(cls.__parse(u) for u in url)
        if is_url and not user:
            raise SvnUserRequired("User is not specified")
        url = [cls._get_rw_url(u, user=user) for u in url]
        return super(Arcadia, cls).delete(url, message=message)

    @classmethod
    def commit(cls, path, message, user=None, with_revprop=None):
        """
        Commit changes in path(s)

        .. note:: If there is nothing to commit this method does nothing and
            returns an empty string (empty stdout from `svn commit` command)

        :param path: repository path(s)
        :param message: log message
        :param user: svn user
        :param with_revprop: revision property
        """

        from sandbox import sdk2

        if not user:
            raise SvnUserRequired("User is not specified")

        path = list(common_itertools.chain(path))
        for target in path:
            cls._switch_to_rw(target, user=user)

        if user == cls.RW_USER and sdk2.Task.current:
            message = "{} (committed from Sandbox task #{})".format(message, sdk2.Task.current.id)

        return super(Arcadia, cls).commit(path, message, with_revprop)

    @classmethod
    def merge(cls, url, path, change=None, ignore_ancestry=False):
        super(Arcadia, cls).merge(cls._get_rw_url(url), path, change=change, ignore_ancestry=ignore_ancestry)

    @classmethod
    def blame(cls, url, verbose=False):
        """
        Get output of `svn blame`
        (lines of the form: revision author code)

        :param url: svn path to item
        :param verbose: add --verbose to the svn command line
        :return: content of the item
        """
        return super(Arcadia, cls).blame(cls._get_rw_url(url), verbose=verbose)

    @classmethod
    def cat(cls, url, revision=None):
        """
        Get output of file from url

        :param url: svn path to file
        :param revision: svn revision
        :return: content of the file
        """
        return super(Arcadia, cls).cat(cls._get_rw_url(url), revision=revision)

    @classmethod
    def append(cls, url, path):
        """
        Append path to url

        :param url: url to which do append
        :param path: path to append
        :return: url
        :rtype: str
        """
        parsed_url = cls.parse_url(url)
        return cls.replace(url, path=os.path.join(parsed_url.path, path.strip(os.path.sep)))

    @classmethod
    def parent_dir(cls, url, count=1):
        """
        Get parent path of url of upper level of `count`, but not above '/'.

        :param url: svn url
        :param count: relative level of parent path
        :return: svn url
        :rtype: str
        """
        parsed_url = cls.parse_url(url)
        path = os.path.normpath(parsed_url.path)
        for i in range(count):
            path = os.path.dirname(path)
        return cls.replace(url, path=path)

    @classmethod
    def log(
        cls, url,
        revision_from=None, revision_to=None,
        track_copy=False, limit=None, stop_on_copy=False, fullpath=False, timeout=None, user=None,
    ):
        """
        Get history of changes.

        :param url: svn url
        :param revision_from: beginning revision
        :param revision_to: ending revision
        :param track_copy: track path copies
        :param limit: maximum number of log entries
        :param stop_on_copy: do not cross copies while traversing history
        :param fullpath:
        :param timeout: operation timeout; if None, SVN_DEFAULT_TIMEOUT will be used
        :return: list of dicts with keys:

            - revision
            - author
            - date
            - msg
            - paths - a list of pairs (action, path)

        :rtype: list
        """
        return super(Arcadia, cls).log(
            cls._get_rw_url(url, user), revision_from=revision_from, revision_to=revision_to,
            track_copy=track_copy, limit=limit, stop_on_copy=stop_on_copy, fullpath=fullpath, timeout=timeout
        )

    @classmethod
    def copy(cls, src, dst, message, user=None, revision=None, parents=False):
        """
        Copy svn item

        :param src: source svn url or local repository path
        :param dst: destination svn url or local repository path
        :param message: log message
        :param user: svn user
        :param revision: revision on source_path
        :param parents: add --parents to the svn command line
        :return: svn process return code, stdout and stderr
        """
        if (cls.__parse(src) or cls.__parse(dst)) and not user:
            raise SvnUserRequired("User is not specified")
        src = cls._get_rw_url(src, user=user)
        dst = cls._get_rw_url(dst, user=user)
        return super(Arcadia, cls).copy(src, dst, message, revision=revision, parents=parents)

    @classmethod
    def freeze_url_revision(cls, url):
        """
        Get svn url with fixed revision

        :param url: svn url
        :return: svn url

        Returns the same path is url is of Arc type
        """
        # TODO: refactor it outside of this class
        if url.startswith(cls.ARCADIA_ARC_SCHEME):
            # We cannot fix the revision in Arc without mounting the repo
            # So short circuit and return the same revision
            return url
        url = cls.normalize_url(url)
        revision = cls.get_revision(url)
        parsed_url = cls.parse_url(url)
        if parsed_url.revision != revision:
            return cls.__replace(url, revision=revision)
        else:
            return url

    @classmethod
    def create_tag(cls, source_url, tag_url, user, message="", revision=None, parents=False):
        """
        Create tag from svn url

        :param source_url: source svn url
        :param tag_url: svn url of tag
        :param message: log message
        :param revision: svn revision of source_url
        :param parents: add --parents to the svn command line
        :return: svn process return code, stdout and stderr
        """
        from sandbox import sdk2
        if not message:
            message = "Create tag {} from svn url {}".format(tag_url, source_url)
        if sdk2.Task.current:
            message = "{}. Created from Sandbox task {}".format(
                message, common_urls.get_task_link(sdk2.Task.current.id)
            )
        return cls.copy(source_url, tag_url, message, user=user, revision=revision, parents=parents)

    @classmethod
    def mkdir(cls, url, user=None, message=None, parents=False):
        """
        Create svn directory

        :param url: svn url of directory to create
        :param user: svn user
        :param message: log message
        :param parents: add --parents to the svn command line
        """
        if cls.__parse(url) and not user:
            raise SvnUserRequired("User is not specified")
        super(Arcadia, cls).mkdir(cls._get_rw_url(url, user=user), message=message, parents=parents)

    @staticmethod
    def fetch_patch(arcadia_patch, dest_dir):
        """
        Fetch patch and prepare it to be applied. Understands the following schemas/inputs
        (format is `<schema>:<patch contents>`):

        - `zipatch`: patch stored in a .`zip` archive with patch operations (see `sdk2.vcs.zipatch` module)
        - `rbtorrent`: raw patch shared with `sky share` call
        - `resource`: identifier of a Sandbox resource with raw patch inside
        - `arc` or `rb`: Arcanum review number
        - `http(s)://link-to-raw.patch.file`
        - raw patch (`unicode` or `str` (array of bytes))

        :param arcadia_patch: patch or a source of it (with described limitations)
        :param dest_dir: directory to save the patch into
        :return: (path to patch file, is the file a zipatch)
        :rtype: tuple
        """

        if not arcadia_patch or arcadia_patch == "None":
            return None, False

        from sandbox import sdk2

        is_zipatch = False
        if arcadia_patch.startswith("zipatch:"):
            arcadia_patch = arcadia_patch.strip()
            logging.info("Zipatch supplied")
            arcadia_patch = arcadia_patch[len("zipatch:"):]
            is_zipatch = True

        def retry_urlopen(url, exceptions=urllib.error.URLError, tries=3, delay=1, backoff=3, *args, **kwargs):
            _delay = delay
            for _ in range(tries):
                try:
                    return urllib.request.urlopen(url, *args, **kwargs)
                except exceptions as e:
                    logger.warning("%s, retrying in %s seconds...", e, _delay)
                    time.sleep(_delay)
                    _delay *= backoff
            else:
                t, v, tb = sys.exc_info()
                six.reraise(t, v, tb)

        if arcadia_patch.startswith("rbtorrent:"):
            import api.copier.errors
            arcadia_patch = arcadia_patch.strip()
            logger.info("Check patch file from torrent %s", arcadia_patch)
            copier_options = {"network": api.copier.Network.Backbone}
            try:
                patch_file_info = common_share.files_torrent(arcadia_patch, **copier_options)
            except api.copier.errors.ResourceNotAvailable as error:
                raise common_errors.TaskError(
                    "Cannot download patch file {0}. Error:\n{1}".format(arcadia_patch, error))

            if len(patch_file_info) != 1:
                raise common_errors.TaskError("Incorrect patch file in torrent: you should share only one file.")

            logger.info("Get patch file from torrent %s", arcadia_patch)
            common_share.skynet_get(arcadia_patch, dest_dir, **copier_options)
            patch_filename = patch_file_info[0]["name"]
            arcadia_patch_path = "{0}/{1}".format(dest_dir, patch_filename)

        elif arcadia_patch.startswith("resource:"):
            arcadia_patch = arcadia_patch.strip()
            # download patch as resource
            patch_resource_id = int(arcadia_patch[len("resource:"):])
            arcadia_patch_path = sdk2.Task.current.agentr.resource_sync(patch_resource_id)

        elif arcadia_patch.startswith("arc:") or arcadia_patch.startswith("rb:"):
            arcadia_patch = arcadia_patch.strip()
            review_id = int(arcadia_patch[arcadia_patch.find(":") + 1:])

            # get last patch nevermind published or not
            patch_info = sdk2.Task.current.agentr.patch_info(review_id, published=None)
            if not patch_info:
                raise common_errors.TaskError("Pull request {} not found".format(review_id))

            logger.info("Patch info for pull request %s found: %s", review_id, patch_info)

            if not patch_info.get("zipatch"):
                raise common_errors.TaskError(
                    "Pull request (iteration {}) cannot be checked out, because it's either not for Arcadia "
                    "or too big or too old (zipatch not found)".format(patch_info["revision"])
                )

            arcadia_patch_path = os.path.join(dest_dir, "arcadia.patch")
            with open(arcadia_patch_path, "wb") as patch:
                patch.write(retry_urlopen(patch_info["zipatch"]).read())
            is_zipatch = True

        else:
            if arcadia_patch.startswith("http"):   # SEARCH-500
                arcadia_patch = arcadia_patch.strip()
                url = arcadia_patch
                u = retry_urlopen(url)
                arcadia_patch = u.read()
                md5 = hashlib.md5(arcadia_patch)
                logger.debug("downloaded {length} bytes (md5={md5}) from url {u}".format(
                    length=len(arcadia_patch),
                    u=url,
                    md5=md5.hexdigest()
                ))

            arcadia_patch = six.ensure_binary(arcadia_patch)

            arcadia_patch_path = dest_dir + "/arcadia.patch"
            logger.info("Write arcadia patch to a file %s", arcadia_patch_path)
            open(arcadia_patch_path, "wb").write(arcadia_patch)

        return arcadia_patch_path, is_zipatch

    @staticmethod
    def apply_patch_file(arcadia_path, arcadia_patch_path, is_zipatch=False, strip_depth=0):
        """
        Apply patch prepared via fetch_patch
        """
        logger.info("Apply arcadia patch")
        arcadia_path = str(arcadia_path)
        if is_zipatch:
            z = zipatch.Zipatch(arcadia_patch_path)
            z.apply(arcadia_path)
        else:
            from sandbox.sdk2.helpers import process
            with process.ProcessLog(logger="patch_arcadia") as pl:
                process.subprocess.check_call(
                    [
                        "patch",
                        "-d", arcadia_path,
                        "--ignore-whitespace",
                        "-p" + str(strip_depth),
                        "-i", arcadia_patch_path,
                    ],
                    stdout=pl.stdout, stderr=pl.stderr
                )

        return arcadia_patch_path  # i wonder if this file is gonna be ever deleted

    @classmethod
    def apply_patch(cls, arcadia_path, arcadia_patch, dest_dir):
        """
        Apply patch from torrent or as text diff
        (diff should be taken from arcadia root)
        """
        arcadia_patch_path, is_zipatch = cls.fetch_patch(arcadia_patch, str(dest_dir))
        if arcadia_patch_path is None:
            return None
        else:
            return cls.apply_patch_file(arcadia_path, arcadia_patch_path, is_zipatch)

    @classmethod
    def __is_good_folder_for_svn_url(cls, folder, url):
        """
        Check whether folder is the repository for url, user and revision in url is ignored

        :param folder: local path
        :param url: svn url
        :rtype: bool
        """
        is_trunk = False
        try:
            try:
                folder_info = super(Arcadia, cls).info(folder)
                if folder_info is None:
                    return False, is_trunk
            except SvnError as exc:
                if "E155037" in exc.message:
                    cls.cleanup(folder)
                    folder_info = super(Arcadia, cls).info(folder)
                elif "E155036" in exc.message:
                    cls.upgrade(folder)
                    folder_info = super(Arcadia, cls).info(folder)
                else:
                    raise
            folder_url = cls.normalize_url(folder_info["url"])
            is_trunk = folder_url == cls.ARCADIA_TRUNK_URL
            logger.debug("Compare svn urls %s and %s", folder_url, url)
            url_info = cls.info(url)
            if not url_info:
                return False, is_trunk
            local_url = cls.parse_url(folder_url)
            remote_url = cls.parse_url(url_info["url"])
            return local_url.path == remote_url.path, is_trunk
        except SvnError:
            return False, is_trunk

    @classmethod
    def _update_cache(cls, path, revision, update_root=False):
        try:
            logger.info("Updating %r", path)
            cls.cleanup(path)
            cls.clean(path)
            cls.revert(path=path, recursive=True)
            if update_root:
                cls.__update(path=path, revision=revision, update_root=True)
            else:
                cls.update(path=path, revision=revision, ignore_externals=True, set_depth=cls.Depth.INFINITY)
            return True
        except SvnTemporaryError:
            raise
        except SvnError as err:
            logger.error("Cannot update %r. Error: %s", path, err)
            fs.remove_path(path)
            return False

    @classmethod
    def update_arcadia_hg_cache(cls, revision):
        from sandbox import sdk2
        if not sdk2.Task.current:
            raise SvnError("Arcadia Hg cache can only be updated during task execution")
        agentr = sdk2.Task.current.agentr
        if not agentr:
            raise SvnError("Unable to update Arcadia Hg cache")
        try:
            agentr.arcadia_hg_clone(revision)
        except agentr_errors.NoSpaceLeft:
            raise SvnTemporaryError("There is no space available for hg cache")
        except agentr_errors.WorkerFailed:
            raise SvnError("Subprocess failed when updating hg cache: see agentr.log for details")
        except agentr_errors.ARException as exc:
            raise SvnError("Error when updating hg cache: {}".format(exc))

    @classmethod
    def get_arcadia_hg_dir(cls, url, revision, skip_update=False):
        parsed_url = cls.__parse(url)
        subpath = parsed_url.path.strip("/")
        cache_dir = os.path.join(cls.base_cache_dir, "arcadia.hg")
        target_dir = os.path.join(cache_dir, subpath)

        if not skip_update:
            cls.raise_if_unavailable()
            cls.update_arcadia_hg_cache(revision)

        if not os.path.exists(target_dir):
            raise SvnError("Target directory {} does not exist in Arcadia Hg cache".format(subpath))

        return target_dir

    @classmethod
    def __checkout_trunk_root(cls, url, path, revision=None):
        ro_url, rw_url, revision, path = cls.prepare_checkout(url, path, cls.Depth.FILES, revision)

        if revision and cls.ro_has_revision(ro_url, revision):
            # revision exists in RO replica - co from RO to @REV and sw
            super(Arcadia, cls).checkout(
                cls.__replace(ro_url, revision=revision), path, depth=cls.Depth.IMMEDIATES, force=True
            )
            cls.__update_trunk_root(path, revision)
            cls.__relocate(ro_url, rw_url, path)
        else:
            # HEAD requested (or there is no such revision in RO replica)  - co from RO to @HEAD, sw and up
            super(Arcadia, cls).checkout(ro_url, path, depth=cls.Depth.IMMEDIATES, revision=None, force=True)
            cls.__relocate(ro_url, rw_url, path)
            cls.__update_trunk_root(path, revision)
        return path

    @classmethod
    def __update(
        cls, path, depth=None, set_depth=None, revision=None, ignore_externals=False, parents=False, force=True,
        update_root=False
    ):
        path = os.path.abspath(path)
        logger.info("Update svn repo %s to revision %s", path, revision or "HEAD")
        cls._switch_to_rw(path)
        # check out from RO
        info = cls.__info(path)
        path_rev = None
        if info:
            ro_url = cls._get_ro_url(info["url"])
            ro_info = cls.__check(ro_url)
            path_rev = info.get("entry_revision")
            if ro_info:
                ro_rev = ro_info.get("entry_revision")
                if ro_rev and int(ro_rev) >= int(info["entry_revision"]):
                    ro_rev = min(int(revision), int(ro_rev)) if revision and revision != "HEAD" else ro_rev
                    cls._switch_to_ro(path, info=info)
                    if update_root:
                        cls.__update_trunk_root(path=path, revision=revision)
                    else:
                        super(Arcadia, cls).update(
                            path, depth, set_depth, ro_rev, ignore_externals, parents, force=force
                        )
                    cls._switch_to_rw(path)
                    info = cls.__info(path)
                    path_rev = info.get("entry_revision")
        update_depth = set_depth and set_depth != info.get("depth")
        if path_rev is None or revision is None or int(revision) != int(path_rev) or update_depth:
            if update_root:
                cls.__update_trunk_root(path=path, revision=revision)
            else:
                super(Arcadia, cls).update(path, depth, set_depth, revision, ignore_externals, parents, force=force)
        return path

    @classmethod
    def get_arcadia_src_dir(cls, url, copy_trunk=False):
        """
        Get checkouted arcadia directory with respect to cache.
        If repository with url already exists in cache then revert it to revision from url, else checkout it.
        By default revision is HEAD.

        :param url: arcadia svn url
        :param copy_trunk: in case of getting of branch, copy local trunk and switch it to the branch
        :return: local repository path
        :rtype: str
        """

        url = cls.normalize_url(url)
        logging.debug("Getting source directory for url %s", url)
        revision = cls.get_revision(url)

        if not revision:
            return

        if url.startswith(cls.ARCADIA_HG_SCHEME):
            cls.raise_if_unavailable()
            fs.make_folder(cls.base_cache_dir)
            return cls.get_arcadia_hg_dir(url, revision)

        from sandbox import sdk2

        task_id = sdk2.Task.current.id if sdk2.Task.current else None
        free_space = common_system.get_disk_space(cls.base_cache_dir).free >> 20
        logging.info("There's %d MB free space left on cache partition", free_space)

        # find a suitable branch, update it to the revision and revert local changes
        parsed_url = cls.parse_url(url)
        if not parsed_url.trunk or parsed_url.subpath or ctc.Tag.SSD in common_config.Registry().client.tags:
            svn_dir = tempfile.mkdtemp()
            logger.info("Checkout %r to %r", url, svn_dir)
            try:
                cls.checkout(url=url, path=svn_dir)
            except SvnError as ex:
                logger.exception("Failed to checkout Arcadia")
                fs.remove_path(svn_dir)
                raise common_errors.TaskError("Unable to checkout {}: {}".format(url, ex))
            return svn_dir

        cls.raise_if_unavailable()
        fs.make_folder(cls.base_cache_dir)

        trunk_dir = None

        for cache_dir in cls.get_cache_dirs():
            is_good, is_trunk = cls.__is_good_folder_for_svn_url(cache_dir, url)
            if is_trunk and not trunk_dir:
                trunk_dir = cache_dir
            if is_good:
                if is_trunk and free_space < cls.SAFE_TRUNK_UPDATE_SPACE:
                    raise common_errors.TemporaryError(
                        "There isn't enough free space on a disk to safely update Arcadia trunk cache without "
                        "making it unusable (%d Mb free; %d Mb required)",
                        free_space, cls.SAFE_TRUNK_UPDATE_SPACE
                    )
                update_root = parsed_url.trunk and not parsed_url.subpath
                successful_update = cls._update_cache(cache_dir, revision, update_root=update_root)
                if successful_update:
                    cls.add_cache_metadata(cache_dir, task_id, is_permanent=is_trunk, url=url)
                    return cache_dir

        # not found in the cache, do checkout
        new_cache_dir = fs.get_unique_file_name(cls.base_cache_dir, "arcadia_cache")
        if copy_trunk and trunk_dir:
            logger.info("Copy trunk cache %r to %r", trunk_dir, new_cache_dir)
            shutil.copytree(trunk_dir, new_cache_dir, True)
            cls.svn("sw", url=[cls._get_rw_url(url)], path=new_cache_dir)
            cls.add_cache_metadata(new_cache_dir, task_id, is_permanent=False, url=url)
            if not cls._update_cache(new_cache_dir, revision):
                raise common_errors.TaskError("Unable to update {!r}".format(new_cache_dir))
        else:
            fs.make_folder(new_cache_dir, delete_content=True)
            logger.info("Checkout %r to %r", url, new_cache_dir)
            try:
                if parsed_url.trunk and not parsed_url.subpath:
                    cls.__checkout_trunk_root(url, new_cache_dir)
                else:
                    cls.checkout(url=url, path=new_cache_dir)
                cache_info = super(Arcadia, cls).info(new_cache_dir)
                is_trunk = cache_info and cls.normalize_url(cache_info["url"]) == cls.ARCADIA_TRUNK_URL
                cls.add_cache_metadata(new_cache_dir, task_id, is_permanent=is_trunk, url=url)
            except SvnError as ex:
                logger.exception("Failed to checkout Arcadia")
                fs.remove_path(new_cache_dir)
                raise common_errors.TaskError("Unable to checkout {}: {}".format(url, ex))
        return new_cache_dir

    @classmethod
    def trunk_url(cls, path="", revision=None):
        """
        Get url with arcadia trunk

        :param path: additional path in trunk
        :param revision: svn revision
        :return: arcadia url
        """
        parsed_url = urlparse.urlparse(cls.ARCADIA_TRUNK_URL)
        path = os.path.normpath(os.path.join(
            parsed_url.path,
            path.strip(os.path.sep)
        ))
        url = urlparse.urlunparse(parsed_url._replace(path=path))
        if revision:
            url = cls.replace(url, revision=str(revision))
        return cls.normalize_url(url)

    @classmethod
    def branch_url(cls, branch, path="", revision=None):
        """
        Get url with arcadia branch

        :param branch: name of branch
        :param path: additional path in branch
        :return: arcadia url
        """
        url = os.path.normpath(os.path.join(
            cls.ARCADIA_BASE_URL,
            "branches",
            branch.strip(os.path.sep),
            "arcadia",
            path.strip(os.path.sep)
        ))
        if revision:
            url = cls.replace(url, revision=str(revision))
        return url

    @classmethod
    def tag_url(cls, tag, path="", revision=None):
        """
        Get url with arcadia tag

        :param tag: name of tag
        :param path: additional path in tag
        :return: arcadia url
        """
        url = os.path.normpath(os.path.join(
            cls.ARCADIA_BASE_URL,
            "tags",
            tag.strip(os.path.sep),
            "arcadia",
            path.strip(os.path.sep)
        ))
        if revision:
            url = cls.replace(url, revision=str(revision))
        return url

    @classmethod
    def diff(cls, url, change=None):
        return super(Arcadia, cls).diff(cls._get_rw_url(url), change=change)


class ArcadiaTestData(vcs_cache.CacheableArcadiaTestData, Svn):
    """
    Class for working with arcadia test data
    """

    SUBSYS = ["arcadia_tests_data", "rearrange_data", "data"]

    @classmethod
    def test_data_location(cls, url):
        """
        Determines local path for the given Subversion URL.

        :param url: Test data Subversion URL location.
        :return: a `tuple` of test data root path, requested sub-directory path and
                 Subversion root URL of the test data.
        """
        logger.info("Checking arcadia test data url %s", url)
        kind = next((f for f in cls.SUBSYS if "/{}/".format(f) in url), None)
        if not kind:
            raise common_errors.TaskError("Incorrect arcadia (test) data url '{}'".format(url))
        parsed_url = Arcadia.parse_url(url)
        path = parsed_url.path
        revision = parsed_url.revision

        # get path to data
        cache_dir = None
        if not cls.base_cache_dir:
            cls.base_cache_dir = tempfile.mkdtemp()
        if "arc/trunk" in path:
            cache_dir = os.path.join(cls.base_cache_dir, "trunk" if kind == "arcadia_tests_data" else kind)
        elif "arc/tags" in path or "arc/branches" in path:
            branch_name = re.search(r"arc/(tags|branches)/(.*/{})".format(kind), path).group(2).replace("/", "-")
            cache_dir = os.path.join(cls.base_cache_dir, branch_name)

        if not cache_dir:
            raise common_errors.TaskError(
                "Incorrect arcadia test data path '{}'. Cannot get test data for this URL".format(path)
            )

        rel_dir = re.search(r"/{}/(.*)$".format(kind), path).group(1)
        if rel_dir:
            url = url.rsplit(rel_dir, 1)[0].rstrip("/")

        if os.path.exists(cache_dir):
            try:
                try:
                    test_data_dir_info = Arcadia.info(cache_dir)["url"]
                except SvnError as exc:
                    if "E155037" in exc.message:
                        cls.cleanup(cache_dir)
                        test_data_dir_info = Arcadia.info(cache_dir)["url"]
                    elif "E155036" in exc.message:
                        cls.upgrade(cache_dir)
                        test_data_dir_info = Arcadia.info(cache_dir)["url"]
                    else:
                        raise
                logger.debug("Compare URLs - local '%s' vs remote '%s'.", test_data_dir_info, url)
                test_data_dir_info_path = Arcadia.parse_url(test_data_dir_info).path
                url_path = Arcadia.parse_url(url).path
                if not test_data_dir_info_path.rstrip("/") == url_path.rstrip("/"):
                    logger.warning("Test data local copy at '%s' and requested URLs differs.", cache_dir)
                    fs.remove_path(cache_dir)
            except SvnError as ex:
                logger.warning("Unable to get subversion info about '%s': %s", cache_dir, ex)
                fs.remove_path(cache_dir)
        return cache_dir, rel_dir, url, revision

    @classmethod
    def __remove_extra_files(cls, path):
        rm = []
        out = cls.status(path)
        regex = re.compile("^[ C?!+~*]{7}\s*(.+?)\s*$")
        for line in out.splitlines():
            m = regex.match(line)
            if m:
                rm.append(m.group(1))
        if rm:
            logger.info("Extra files detected: %r. Full status output is:\n" + out, rm)
            from sandbox.sdk2.helpers import process
            with process.ProcessLog(logger="svn_remove_extra_files") as pl:
                process.subprocess.Popen(
                    ["rm", "-rf"] + [os.path.join(path, fn) for fn in rm if fn],
                    stdout=pl.stdout, stderr=pl.stderr
                ).wait()

    @classmethod
    def cleanup_arcadia_test_data_folder(cls, task, path, remove_extra_files=True):
        """
        Cleanup and update test data in path

        :param task: Task object
        :param path: path to local cache
        :return: True if path is existing directory
        """
        from sandbox.sdk2.helpers import process
        if not os.path.isdir(path):
            return False
        logger.info("Trying to cleanup and update the test data in folder %s", path)
        try:
            cls.cleanup(path)
        except SvnError:
            with process.ProcessLog(logger="svn_cleanup_failed_rm_folder") as pl:
                process.subprocess.Popen(
                    ["rm", "-rf", path],
                    stdout=pl.stdout, stderr=pl.stderr
                ).wait()
            return False

        try:
            cls.revert(path, recursive=True)
        except SvnError:
            with process.ProcessLog(logger="svn_revert_failed_rm_folder") as pl:
                process.subprocess.Popen(
                    ["rm", "-rf", path],
                    stdout=pl.stdout, stderr=pl.stderr
                ).wait()
            return False

        if remove_extra_files:
            cls.__remove_extra_files(path)

        return os.path.isdir(path)

    @classmethod
    def get_arcadia_test_data(cls, task, url, copy_path=None):
        """
        Checkout or update test data from trunk or branch and return path to local cache.

        :param task: Task object
        :param url: arcadia url of test data
        :param copy_path: if it is not None then copy data to it
        :return: path to test data
        """

        cls.raise_if_unavailable()

        logger.info("Getting arcadia test data from svn path: %s", url)
        rootpath, relpath, rootsvn, revision = cls.test_data_location(url)
        fullpath = os.path.join(rootpath, relpath) if relpath else rootpath
        logger.debug(
            "Test data root path: %r, relative: %r, full: %r, SVN root: %r", rootpath, relpath, fullpath, rootsvn
        )

        if not cls.cleanup_arcadia_test_data_folder(task, rootpath):
            if task:
                cls.add_cache_metadata(rootpath, task.id, url=url)
            logger.info("Checkout arcadia test data to folder '%s'", rootpath)
            Arcadia.checkout(rootsvn, rootpath, depth=Svn.Depth.EMPTY if relpath else Svn.Depth.INFINITY)

        if relpath:
            dirs = relpath.split("/")
            subdir = rootpath
            Arcadia.update(subdir, depth=Svn.Depth.EMPTY)
            for i, d in enumerate(dirs):
                subdir = os.path.join(subdir, d)
                if not os.path.isdir(subdir):
                    if os.path.exists(subdir):
                        os.unlink(subdir)
                    if i + 1 < len(dirs):
                        Arcadia.update(subdir, depth=Svn.Depth.EMPTY)
        Arcadia.update(fullpath, set_depth=Svn.Depth.INFINITY, revision=revision)

        if copy_path:
            logger.info("Trying to copy arcadia test data '%s' to folder '%s'", fullpath, copy_path)
            copy_path = os.path.abspath(copy_path)
            if os.path.exists(copy_path):
                fs.remove_path(copy_path)
            try:
                shutil.copytree(fullpath, copy_path, ignore=shutil.ignore_patterns('.svn'))
            except (IOError, OSError) as e:
                raise common_errors.TaskError("Cannot copy '{}' to '{}'. Error: {}".format(
                    fullpath, copy_path, e
                ))
            fullpath = copy_path

        return fullpath
