from __future__ import absolute_import

import os
import sys
import json
import stat
import shutil
import zipfile
import filecmp
import logging

import six

_EXECUTABLE_MODE = stat.S_IXUSR | stat.S_IXGRP | stat.S_IXOTH


def _remove_path(path):
    if os.path.exists(path) or os.path.lexists(path):
        if os.path.islink(path) or os.path.isfile(path):
            os.remove(path)
        else:
            shutil.rmtree(path)
    else:
        logging.debug("path {p} does not exist, nothing to remove".format(p=repr(path)))


def remove_file_handler(path, data, action=None):
    logging.debug("removing file {p}".format(p=repr(path)))
    _remove_path(path)


def remove_tree_handler(path, data, action=None):
    logging.debug("removing directory {p}".format(p=repr(path)))
    _remove_path(path)


def store_file_handler(path, data, action=None):
    dir = os.path.dirname(path)
    mkdir_handler(dir, data)
    logging.debug("writing to file {p} {l} bytes".format(p=repr(path), l=len(data)))
    open(path, "wb").write(data)
    if action:
        original_mode = stat.S_IMODE(os.stat(path).st_mode)
        new_mode = (
            original_mode | _EXECUTABLE_MODE
            if action.get("executable", False)
            else original_mode & ~_EXECUTABLE_MODE
        )
        os.chmod(path, new_mode)
        logging.debug("chmod file %s from %s to %s", repr(path), oct(original_mode), oct(new_mode))


def svn_copy_handler(path, data, action=None):
    pass


def mkdir_handler(path, data, action=None):
    if not os.path.exists(path):
        try:
            os.makedirs(path)
            logging.debug("created directory {d}".format(d=path))
        except Exception as e:
            logging.debug("failed to create directory {d}: {e}".format(d=path, e=e))


FILES_DIR = "files"
ACTIONS_FILE = "actions.json"
ACTION_HANDLERS = {
    "remove_file": remove_file_handler,
    "remove_tree": remove_tree_handler,
    "store_file": store_file_handler,
    "svn_copy": svn_copy_handler,
    "mkdir": mkdir_handler,
}


class ZipatchWriter(object):
    def __init__(self):
        self.actions = []

    def init_from_svn_status_output(self, svn_status_output, root=None):
        for line in svn_status_output.splitlines():
            (type, fullpath) = line.split(None, 1)

            relpath = fullpath if root is None else os.path.relpath(fullpath, root)

            if type in ["A", "M"]:
                if os.path.isfile(fullpath):
                    self.add_action("store_file", relpath, file=fullpath)
            if type in ["D"]:
                self.add_action("remove_file", relpath)

    def init_from_zipatch(self, zipatch_path):
        z = zipfile.ZipFile(zipatch_path, "r")
        assert z.testzip() is None

        actions_json = z.read(ACTIONS_FILE)
        actions = json.loads(actions_json)
        for a in actions:
            data = z.read(a["file"]) if a.get("file") else None
            self.add_action(a["type"], a["path"], data=data)

        z.close()

    def add_action(self, action_type, path, file=None, data=None):
        assert action_type in ACTION_HANDLERS

        self.actions.append({
            "type": action_type,
            "path": path,
            "file": file,
            "data": data,
        })

        logging.debug("added action {t} path={p}, file={f}, data={d}".format(
            t=action_type,
            p=path,
            f=file,
            d=len(data) if data is not None else None,
        ))

    def save(self, file):
        logging.debug("saving zipatch to file {f}".format(f=file))

        z = zipfile.ZipFile(file, "w", zipfile.ZIP_DEFLATED)

        zip_actions = []
        for a in self.actions:
            if a["type"] == "store_file":
                zip_path = os.path.join(FILES_DIR, a["path"])

                zip_actions.append({
                    "type": a["type"],
                    "path": a["path"],
                    "file": zip_path,
                })

                if a["file"] is not None:
                    logging.debug("adding file {f} to zipatch as {p}".format(f=a["file"], p=zip_path))
                    z.write(a["file"], zip_path)
                elif a["data"] is not None:
                    logging.debug("adding data {d} to zipatch as {p}".format(d=len(a["data"]), p=zip_path))
                    z.writestr(zip_path, a["data"])
                else:
                    raise Exception("action {t} path={p}: neither file nor data were specified".format(
                        t=a["type"],
                        p=a["path"],
                    ))

            elif a["type"] in ["remove_file", "remove_tree"]:
                zip_actions.append({
                    "type": a["type"],
                    "path": a["path"],
                })

        z.writestr(ACTIONS_FILE, json.dumps(zip_actions, indent=4))

        assert z.testzip() is None
        z.close()


class Zipatch(object):
    def __init__(self, path):
        self.path = path

    def apply(self, dirpath):
        logging.debug("applying zipatch {f} to directory {d}".format(f=self.path, d=dirpath))
        z = zipfile.ZipFile(self.path, "r")
        assert z.testzip() is None

        actions_json = z.read(ACTIONS_FILE)
        actions = json.loads(actions_json)
        logging.debug("read {l} actions".format(l=len(actions)))
        for a in actions:
            logging.debug("applying action {a}".format(a=a))
            handler = ACTION_HANDLERS[a["type"]]
            real_path = os.path.join(dirpath, a["path"])
            data = z.read(a["file"]) if a.get("file") else None
            handler(real_path, data, a)

        z.close()


def main(args):
    w = ZipatchWriter()
    for arg in args:
        w.add_action("store_file", arg, file=arg)
    w.save("patch.zipack")


def diff(dir1, dir2, patch):
    """
    this is very simple proof of concept differ, it is buggy and should not be used as a replacement for diff
    """
    z = ZipatchWriter()

    def fill_patch(dcmp, dir=""):
        for file in dcmp.diff_files + dcmp.right_only:
            z.add_action("store_file", os.path.join(dir, file), file=os.path.join(dcmp.right, file))

        for file in dcmp.left_only:
            z.add_action("remove_file", os.path.join(dir, file))

        for (subdir, sub_dcmp) in six.iteritems(dcmp.subdirs):
            fill_patch(sub_dcmp, os.path.join(dir, subdir))

    dcmp = filecmp.dircmp(dir1, dir2)
    fill_patch(dcmp)
    z.save(patch)


def patch(patch, dir):
    z = Zipatch(patch)
    z.apply(dir)


if __name__ == "__main__":
    if sys.argv[1] == "diff":
        diff(sys.argv[2], sys.argv[3], sys.argv[4])
    elif sys.argv[1] == "patch":
        try:
            cwd = sys.argv[3]
        except IndexError:
            cwd = os.getcwd()
        patch(sys.argv[2], cwd)
