# coding=utf-8
from __future__ import absolute_import, unicode_literals

import base64
import logging
import collections
import six

from sandbox.common import log
from sandbox.common import yav
from sandbox.common import rest
from sandbox.common import errors as common_errors


class Data(dict):
    version = None


class SecretValue(object):
    """ Single key-value pair from a secret with additional info about encoding """
    key = ""
    value = ""
    encoding = ""

    def __init__(self, key, value, encoding):
        self.key = key
        self.value = value
        self.encoding = encoding

    def get_decoded(self):
        need_decoding = self.encoding == "base64"
        return base64.b64decode(self.value) if need_decoding else self.value


class SecretsStorage(object):
    """ Internal storage for key-values of one secret """

    def __init__(self, secret_data):
        """
        :param secret_data: secret info received from API
        :type secret_data: str (serialized web.api.v1.schemas.yav.SecretData)
        """
        self.storage = {}  # type: dict(str, SecretValue)
        for item in secret_data["values"]:
            key = item["key"]
            self.storage[key] = SecretValue(key, item["value"], item["encoding"])
        self.version = secret_data["version"]

    def get_decoded(self, key):
        """ Returns automatically decoded (if needed) value """
        secret_value = self.storage[key]  # type: SecretValue
        return secret_value.get_decoded()

    def to_data(self, decoded):
        """
        Returns secret as a dictionary.
        :param decoded: Whether values are decoded automatically in case of 'file' value.
        """
        as_dict = {
            secret_value.key: (secret_value.get_decoded() if decoded else secret_value.value)
            for secret_value in six.itervalues(self.storage)
        }
        data = Data(as_dict)
        data.version = self.version
        return data


class Secret(object):
    """ Holder for Yav secret uuid and optional version. When omitted, latest version is assumed. """

    def __init__(self, secret, version=None, default_key=None):
        # noinspection PyArgumentList
        self.secret = secret if isinstance(secret, yav.Secret) else yav.Secret(secret, version, default_key)

    @classmethod
    def __decode__(cls, value):
        return cls(yav.Secret.__decode__(value))

    def data(self, auto_decode=False):
        """
        Fetch a single secret value.
        Whether decode data inside or not, depends on secret parameter `auto_decode`

        :return: the secret value
        :rtype: `Data`
        """
        return Yav(foo_alias=self).get_decoded("foo_alias") if auto_decode else Yav(foo_alias=self).foo_alias

    def value(self, auto_decode=False):
        """
        Fetch secret value for given default_key

        :return: value from secret data for default_key if preset
        """
        if self.default_key is not None:
            return self.data(auto_decode)[self.default_key]
        else:
            return self.data(auto_decode)

    @property
    def default_key(self):
        return self.secret.default_key

    def __str__(self):
        return str(self.secret)


class Yav(object):

    class Error(common_errors.TaskFailure):
        pass

    def __init__(self, **aliases):
        """
        :param aliases: Mapping `short name to be used later in task code` -> sdk2.yav.Secret
        :type aliases: dict(str, yav.Secret)
        """

        self.__cache = {}  # type: dict(str, SecretsStorage)
        aliases = collections.OrderedDict(six.iteritems(aliases))

        secrets = []
        for alias, item in six.iteritems(aliases):
            if not isinstance(item, Secret):
                raise ValueError("{}: `sdk2.yav.Secret` expected, got {}".format(alias, type(item).__name__))
            secrets.append({"id": item.secret.secret_uuid, "version": item.secret.version_uuid})

        server = rest.Client()
        try:
            response = server.yav.secrets.data(secrets=secrets)  # serialized web.api.v1.schemas.yav.SecretDataList
            secret_data_list = response["items"]
        except server.HTTPError as exc:
            try:
                resp = exc.response.json()
                msg = resp["reason"]
            except (ValueError, KeyError):
                msg = exc.response.response.text
            raise self.Error("Cannot fetch secret values: {}".format(msg))

        for alias, secret_data in zip(aliases, secret_data_list):
            storage = SecretsStorage(secret_data)
            self.__cache[alias] = storage

        self.__setup_log_filter()

    @classmethod
    def setup(cls, key):
        cls.__key = key

    def __setup_log_filter(self):
        logger = logging.getLogger()
        vault_filter = log.VaultFilter.filter_from_logger(logger)

        if not vault_filter:
            return

        for alias, secret_storage in six.iteritems(self.__cache):
            for key, value in six.iteritems(secret_storage.to_data(True)):
                if len(value) < 6:
                    continue

                non_escaped = six.ensure_binary(value)  # u"абв" -> b"\xd0\xb0\xd0\xb1\xd0\xb2"
                escaped = six.ensure_binary(value, encoding="unicode_escape")  # u"абв" -> b"\\u0430\\u0431\\u0432"

                vault_filter.add_record(key, non_escaped)
                if non_escaped != escaped:
                    vault_filter.add_record(key + "__escaped__", escaped)

    def __getattr__(self, alias):
        """
        Returns dictionary-like object of secret key-values for a given secret alias.
        Does nothing about encoding for backward-compatibility.

        :rtype: Data
        """
        try:
            secret_storage = self.__cache[alias]
            return secret_storage.to_data(False)
        except Exception:
            raise AttributeError(alias)

    def get_decoded(self, alias):
        """
        Returns dictionary-like object of secret key-values for a given secret alias.
        Automatically decode file secrets if needed.

        :rtype: Data
        """
        try:
            secret_storage = self.__cache[alias]
            return secret_storage.to_data(True)
        except Exception:
            raise AttributeError(alias)
