import os
import sys
import argparse

import yaml
import pyuwsgi

from sandbox.common import oauth
from sandbox.common import config
from sandbox.common.types import misc as ctm


class IndentDumper(yaml.Dumper):
    def increase_indent(self, flow=False, indentless=False):
        return super(IndentDumper, self).increase_indent(flow, False)


def indent_text(text, indent, indent_char=" "):
    return "".join(indent * indent_char + line + "\n" for line in text.splitlines())


def dump_config(config, indent=2):
    # Stored as strings, append them to the dumped config in the end
    caches = config["uwsgi"].pop("caches")
    sharedareas = config["uwsgi"].pop("sharedareas")
    mules = config["uwsgi"].pop("mules")
    metrics = config["uwsgi"].pop("metrics")
    routes = config["uwsgi"].pop("routes")

    config = yaml.dump(config, Dumper=IndentDumper, default_flow_style=False, indent=indent)
    config += indent_text(caches, indent)
    config += indent_text(sharedareas, indent)
    config += indent_text(mules, indent)
    config += indent_text(metrics, indent)
    config += indent_text(routes, indent)

    return config


def get_uwsgi_yaml_config(args, registry):
    from library.python import resource
    config = yaml.load(resource.find("serviceapi/uwsgi.yaml"), Loader=yaml.CSafeLoader)
    uwsgi = config["uwsgi"]

    # Override defaults
    port = args.port or registry.server.api.port
    uwsgi["http-socket"] = "[::]:{}".format(port)
    uwsgi["workers"] = (args.workers or registry.server.api.workers) + 1  # One worker for dropping requests

    if registry.server.api.enable_stats:
        uwsgi["stats"] = "[::]:{}".format(port + 1)
    else:
        uwsgi.pop("stats")

    # Use adaptive process spawning if max number of workers is too big
    if registry.server.api.enable_cheaper and uwsgi["workers"] >= 32:
        uwsgi["cheaper-algo"] = "spare"
        uwsgi["cheaper"] = 16
        uwsgi["cheaper-step"] = 4
        uwsgi["cheaper-initial"] = 32

    if args.stderr_log:
        uwsgi["req-logger"] = "stdio"
        uwsgi["logger"] = "stdio"
    else:
        def log_for(name):
            log_path = os.path.join(registry.server.log.root, registry.server.api.log[name])
            log_dir = os.path.dirname(log_path)
            if not os.path.exists(log_dir):
                os.makedirs(log_dir)
            return "file:" + log_path

        uwsgi["req-logger"] = log_for("access_log")
        uwsgi["logger"] = log_for("server_log")

    return dump_config(config)


def get_uwsgi_args(args):
    r, w = os.pipe()
    os.write(w, get_uwsgi_yaml_config(args, config.Registry()))
    os.close(w)

    return ["--yaml", "fd://" + str(r)]


if __name__ == "__main__":

    # Check if path to local settings is defined
    config.ensure_local_settings_defined()

    if config.Registry().common.installation == ctm.Installation.LOCAL and ctm.OAUTH_TOKEN_ENV_NAME not in os.environ:
        # * fetch OAuth token for Yav on local installations;
        # * exec into itself with the token added to the environment variables.
        #
        # The exec part is required: simply updating `os.environ` somehow messes up with uwsgi
        # so that it stops updating the process title and even segfaults in some cases.

        env = os.environ.copy()
        env[ctm.OAUTH_TOKEN_ENV_NAME] = oauth.token() or ""
        os.execve(sys.executable, sys.argv, env)

    parser = argparse.ArgumentParser()
    parser.add_argument("-v", "--version", action="store_true", help="Print version")
    parser.add_argument("-p", "--port", type=int, help="Http port")
    parser.add_argument("-w", "--workers", type=int, help="Number of workers")
    parser.add_argument("--stderr-log", help="Write logs to stderr", action="store_true")
    args = parser.parse_args()

    if args.version:
        import library.python.svn_version as sv
        print(sv.svn_version())
        sys.exit(0)

    pyuwsgi.run(get_uwsgi_args(args))
