import re
import logging
import posixpath

import flask
import flask_cors
import werkzeug

from sandbox import common
import sandbox.common.types.misc as ctm

from sandbox.serviceapi import constants

from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping


__all__ = ("create_app",)


def rule_wrapper(rule, **kwargs):
    if kwargs.pop("accept_all_methods", False):
        kwargs["methods"] = None
    return werkzeug.routing.Rule(rule, **kwargs)


def setup_logging():
    handler = logging.StreamHandler()
    handler.setFormatter(logging.Formatter("%(asctime)s %(levelname)-7s (%(pathname)s:%(lineno)s) %(message)s"))

    logging.root.handlers = []  # Somebody could have already initialized logging
    logging.root.addHandler(handler)
    logging.root.setLevel(logging.DEBUG)

    # Make noisy libraries less noisy
    logging.getLogger("requests.packages.urllib3").setLevel(logging.ERROR)
    logging.getLogger("urllib3.connectionpool").setLevel(logging.WARNING)


def setup_database():
    mapping.ensure_connection(
        uri=common.config.Registry().server.mongodb.connection_url,
        max_pool_size=100,
    )


def setup_yasandbox():
    controller.initialize()


class SandboxFlask(flask.Flask):
    VERSION_RE = re.compile("\A(v1.0|v\d+)\Z")

    def create_url_adapter(self, request):
        """ Redefine to change api version """
        if request is not None:
            # Try to get adapter by request path
            path = posixpath.normpath(request.environ[constants.Path.PATH_INFO])
            if common.config.Registry().common.installation == ctm.Installation.LOCAL:
                striped_path = path.lstrip("/")
                if striped_path.startswith("proxy/api"):
                    path = striped_path[5:]
            request.environ[constants.Path.PATH_INFO] = path
            try:
                default_adapter = self.url_map.bind_to_environ(request.environ, server_name=self.config["SERVER_NAME"])
                rule, rw = default_adapter.match(return_rule=True)
            except:
                return super(SandboxFlask, self).create_url_adapter(request)
            if rule.rule != constants.Path.NOT_IMPLEMENTED_URL:
                return default_adapter

            # Try to downgrade api version, setup maximum version to constants.MAX_VERSION to avoid dos by high version
            splited_path = path.lstrip("/").split("/", 2)

            if (
                len(splited_path) > 2 and
                splited_path[0] in ("api", "quotaManagement") and
                self.VERSION_RE.match(splited_path[1])
            ):
                api, version_part, suffix = splited_path
                version = float(version_part[1:])

                for new_version in range(min(int(version) - 1, constants.Path.MAX_VERSION), 0, -1):
                    new_path = "/".join((
                        "", api, "v{}{}".format(new_version, ".0" if new_version == 1 else ""), suffix
                    ))
                    request.environ[constants.Path.PATH_INFO] = new_path

                    try:
                        adapter = self.url_map.bind_to_environ(
                            request.environ, server_name=self.config["SERVER_NAME"]
                        )
                        rule, rw = adapter.match(return_rule=True)
                        if rule.rule != constants.Path.NOT_IMPLEMENTED_URL:
                            return adapter
                    except:
                        pass

            request.environ[constants.Path.PATH_INFO] = path
            return default_adapter

        # If there is not request return adapter by parent method
        return super(SandboxFlask, self).create_url_adapter(request)


def create_app():
    # Perform some global initialization
    setup_logging()
    setup_database()
    setup_yasandbox()
    common.statistics.Signaler(
        common.statistics.SignalHandlerInternalServiceApi(),
        component=ctm.Component.SERVICEAPI,
        all_signals=True
    )

    app = SandboxFlask(__name__, static_folder=None)
    app.url_rule_class = rule_wrapper
    flask_cors.CORS(app, supports_credentials=True)

    from . import plugins
    plugins.batch.init_plugin(app)
    plugins.request_metrics.init_plugin(app)
    plugins.context.init_plugin(app)
    plugins.legacy.init_plugin(app)
    plugins.solomon.init_plugin(app)
    plugins.health_check.init_plugin(app)
    plugins.http_check.init_plugin(app)
    plugins.openapi.init_plugin(app)
    plugins.static.init_plugin(app)
    plugins.mds.init_plugin(app)
    plugins.versions.init_plugin(app)

    from . import handlers
    handlers.init_handlers(app)

    from .handlers import quota_management
    quota_management.init_handlers(app)

    from . import web
    web.init_api(app)

    web.aggregator.Aggregator().url_map = app.url_map

    return app
