import time
import uuid
import base64
import struct
import hashlib
import urlparse
import binascii

import requests

import paramiko.rsakey
import paramiko.dsskey
import paramiko.message

import httplib
import collections

import flask

from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.user as ctu

from sandbox.web.api import v1
from sandbox.yasandbox import context, controller
from sandbox.yasandbox.database import mapping

from sandbox.serviceapi.web import RouteV1, exceptions


class SSHKey(RouteV1(v1.authenticate.SSHKey)):
    @classmethod
    def _get_user_keys(cls, login):
        persons = controller.User.staff_api.persons.read(login=login, _fields="keys").get("result")
        return persons[0].get("keys", []) if persons else []

    @classmethod
    def post(cls, login, fingerprint, query, signature):
        login = login.strip().lower()
        try:
            fingerprint = binascii.unhexlify(fingerprint.strip().lower())
        except (ValueError, TypeError) as ex:
            raise exceptions.BadRequest("Invalid fingerprint provided: {}".format(ex))

        try:
            cnonce = binascii.unhexlify(query.get("cnonce").strip().lower())
            cnow = struct.unpack("L", cnonce)[0] >> 32
            now = int(time.time())
            # Check the salt has been created recently.
            if cnow < now - 43200 or cnow > now + 43200:
                raise ValueError("Client time does not match current date")
        except (ValueError, TypeError, AttributeError, struct.error) as ex:
            raise exceptions.BadRequest("Invalid cnonce provided: {}".format(ex))

        keys = cls._get_user_keys(login)
        if not keys:
            raise exceptions.NotFound("No SSH keys found for user '{}'".format(login))

        pubkey = None
        for k in keys:
            try:
                # get the second field from the public key file.
                keydata = base64.b64decode(k["key"].split(None)[1])
                msg = paramiko.message.Message(keydata)
                ktype = msg.get_string()
                msg.rewind()

                if ktype == "ssh-rsa":
                    pubkey = paramiko.rsakey.RSAKey(msg=msg)
                elif ktype == "ssh-dss":
                    pubkey = paramiko.dsskey.DSSKey(msg=msg)
                else:
                    raise ValueError("Unknown key format {}".format(ktype))

                if fingerprint == pubkey.get_fingerprint():
                    break
            except Exception as ex:
                context.current.logger.error(
                    "Error loading user's '%s' key #%s (%s): %s", login, k["id"], k["description"], ex
                )
                continue

        if not pubkey:
            raise exceptions.NotFound(
                "No user's '{}' public key '{}' found.".format(login, binascii.hexlify(fingerprint))
            )

        h = hashlib.sha1()
        h.update(login)
        h.update(fingerprint)
        h.update(cnonce)

        if not pubkey.verify_ssh_sig(h.digest(), paramiko.message.Message(signature)):
            raise exceptions.BadRequest(
                "Provided signature of {} bytes length cannot verify user's '{}' "
                "digest '{}' with key '{}' and cnonce '{}'.".format(
                    len(signature), login,
                    h.hexdigest(), binascii.hexlify(fingerprint), binascii.hexlify(cnonce)
                )
            )
        token = controller.OAuthCache.refresh(login, source=ctu.TokenSource.SSH_KEY).token
        return flask.current_app.response_class(
            response=token,
            status=httplib.CREATED,
            content_type="text/plain",
        )


class OauthBase(object):
    _client_token_cache = None

    @common.utils.classproperty
    def _client_token(cls):
        if cls._client_token_cache is None:
            settings = common.config.Registry()
            cls._client_token_cache = (
                common.utils.read_settings_value_from_file(settings.server.auth.oauth.client_config)
                if settings.server.auth.enabled else
                ':'
            )
        return collections.namedtuple('ClientToken', ('id', 'secret'))(*cls._client_token_cache.split(':'))


class OAuthClient(OauthBase, RouteV1(v1.authenticate.OAuthClient)):
    @classmethod
    def get(cls):
        return v1.schemas.authenticate.OAuthClientInfo.create(
            client=cls._client_token.id, url=common.config.Registry().client.auth.oauth_url
        )


class OAuthToken(OauthBase, RouteV1(v1.authenticate.OAuthToken)):
    @classmethod
    def _request_token(cls, url, data):
        url += '/token'
        data['client_id'], data['client_secret'] = cls._client_token
        headers = {
            'Host': urlparse.urlparse(url).hostname,
            'Content-type': 'application/x-www-form-urlencoded',
        }
        resp = requests.post(url, data=data, headers=headers).json()
        token = resp.get('access_token')
        if not token:
            raise Exception('OAuth server returned no token: {!r}'.format(resp))
        return token

    @classmethod
    def post(cls, body):
        settings = common.config.Registry()

        if not settings.server.auth.enabled:
            token = "d41d8cd98f00b204e9800998ecf8427e"  # empty MD5 checksum
        elif body.login and body.password:
            context.current.logger.debug("User '%s' requested OAuth token via password authentication.", body.login)
            try:
                token = cls._request_token(
                    settings.client.auth.oauth_url,
                    {'grant_type': 'password', 'username': body.login, 'password': body.password},
                )
            except Exception as ex:
                raise exceptions.BadRequest("OAuth server returned no token: " + str(ex))
        else:
            try:
                code = body.code
                context.current.logger.debug("User '%s' grant OAuth token via code '%s'.", code)
                token = cls._request_token(
                    settings.client.auth.oauth_url,
                    {'grant_type': 'authorization_code', 'code': code},
                )
            except Exception as ex:
                raise exceptions.BadRequest("OAuth server returned no token: " + str(ex))

            context.current.logger.info("%r OAuth token: %r", context.current.user.login, token)
        return flask.current_app.response_class(
            response=token,
            status=httplib.CREATED,
            content_type="text/plain",
        )


class ExternalOAuthToken(RouteV1(v1.authenticate.ExternalOAuthToken)):
    @classmethod
    def post(cls, body):
        try:
            user = context.current.request.check_validated_user(
                body.login, controller.User.validate(body.login), is_request_author=False
            )
            session = mapping.OAuthCache(
                token=uuid.uuid4().hex,
                login=user.login,
                ttl=body.ttl,
                source=ctu.TokenSource.EXTERNAL_SESSION,
                app_id=body.job_id,
                state=str(ctt.SessionState.ACTIVE),
                properties=mapping.OAuthCache.Properties(
                    session_maker=context.current.user.login,
                    service=body.service,
                    sandbox_task_id=body.sandbox_task_id,
                ),
            )
            session.save(force_insert=True)
        except ValueError as ex:
            raise exceptions.BadRequest(str(ex))

        return v1.schemas.authenticate.ExternalOAuth.create(
            token=session.token,
            login=session.login,
            ttl=session.ttl,
            service=session.properties.service,
            job_id=session.app_id,
            sandbox_task_id=session.properties.sandbox_task_id
        )

    @classmethod
    def delete(cls, body):
        session = mapping.OAuthCache.objects.lite().with_id(body.token)
        if session is None:
            raise exceptions.NotFound("Session not found")

        if not session.properties or session.properties.service != body.service:
            raise exceptions.Forbidden("Session doesn't belong to service {}".format(body.service))

        if session.properties is None or session.properties.session_maker != context.current.user.login:
            raise exceptions.Forbidden("Forbidden to delete session created by other session maker")

        session.delete()
        return "", httplib.NO_CONTENT


class ExternalOauthSessionSearch(RouteV1(v1.authenticate.ExternalOauthSessionSearch)):
    @classmethod
    def post(cls, body):
        query = {}

        if body.token:
            query["token"] = body.token
        if body.job_id:
            query["app_id"] = body.job_id

        if not query:
            raise exceptions.BadRequest("Token or job_id required")
        session = mapping.OAuthCache.objects(**query).first()
        if session is None:
            raise exceptions.NotFound("External session not found")
        if session.properties.service != body.service:
            raise exceptions.Forbidden(
                "Can't get external session with service '{}' from service '{}'".format(
                    session.properties.service, body.service
                )
            )
        if session.properties.session_maker != context.current.user.login:
            raise exceptions.Forbidden("Can't get external session created by other session maker")

        return v1.schemas.authenticate.ExternalOAuth.create(
            token=session.token,
            login=session.login,
            ttl=session.ttl,
            service=session.properties.service,
            job_id=session.app_id,
            sandbox_task_id=session.properties.sandbox_task_id
        )


class TaskSessionSearch(RouteV1(v1.authenticate.TaskSessionSearch)):
    @classmethod
    def post(cls, body):
        session = mapping.OAuthCache.objects(token=body.token).first()
        if session is None:
            raise exceptions.NotFound("Task session not found")

        return v1.schemas.authenticate.TaskSession.create(
            token=session.token,
            login=session.login,
            ttl=session.ttl,
            task_id=session.task_id
        )
