# coding: utf-8

import json
import time
import uuid
import base64
import httplib
import calendar
import datetime as dt
import itertools as it

import flask

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.user as ctu
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
import sandbox.common.types.statistics as cts

from sandbox import common
from sandbox.web.api import v1
from sandbox.common import config
import sandbox.common.joint.errors as jerrors
from sandbox.yasandbox.database import mapping
from sandbox.yasandbox import context, controller

import sandbox.serviceq.errors as qerrors

from sandbox.serviceapi import mappers
from sandbox.serviceapi.web import RouteV1
from sandbox.serviceapi.web import exceptions


registry = config.Registry()


class ClientRouteBase(object):
    STALE_SESSION_AGE = 5


class ClientQueryMapper(object):
    @classmethod
    def prepare_update_data(cls, data):
        os = data.os
        disk = data.disk
        slots = data.slots
        update_data = {
            "system": {
                "arch": os.name,
                "os_version": os.version,
                "physmem": data.ram,
                "cpu_model": data.cpu,
                "ncpu": data.ncpu,
                "total_space": disk.total_space >> 20,
                "free_space": disk.free_space >> 20,
                "used_space": 100 - float(disk.free_space) / disk.total_space * 100,
                "used_space_value": (disk.total_space - disk.free_space) >> 20,
                "reserved_space": (disk.locked_space or 0) >> 20,
                "disk_status": disk.status,
                "platform": data.platform,
                "lxc": bool(data.lxc),
                "porto": bool(data.porto),
                "root": data.root,
                "fqdn": data.fqdn,
                "dc": data.dc,
                "fileserver": data.fileserver,
                "tasks_dir": data.tasks_dir,
                "total_slots": slots.total if slots else None,
                "used_slots": slots.used if slots else None,
            },
            "revisions": {
                "sdk": data.sdk,
                "client": data.client,
                "tasks": data.tasks,
                "venv": data.venv,
                "server": data.server,
            }
        }
        age = data.age
        return update_data, age


class Client(ClientRouteBase, RouteV1(v1.client.Client)):
    @classmethod
    def get(cls, id_):
        client = controller.Client.get(id_)
        if client is None:
            raise exceptions.NotFound("Client #{} not found.".format(id_))

        return mappers.client.SingleClientMapper().dump(client)

    @classmethod
    def put(cls, id_, body):
        client = controller.Client.get(id_)

        if not client:
            raise exceptions.NotFound("Client with id '{}' not found".format(id_))

        update, protocol_version = ClientQueryMapper.prepare_update_data(body)
        context.current.logger.debug("Updating client %r v%s.", id_, protocol_version)
        client.info["revisions"] = update.pop("revisions")
        update["system"]["inetfs"] = [ctm.Network.Family.V6] + (
            [ctm.Network.Family.V4] if ctc.Tag.IPV4 in client.tags else []
        )

        jobs = {job.id: job.state for job in body.jobs or []}
        controller.Client.update(client, update)

        # Reasons for client to validate their jobs after this request
        reset_reasons = []

        for session in mapping.OAuthCache.client_sessions(id_):
            # Calculate session expiration markers
            ttl_expired = session.created + dt.timedelta(seconds=session.ttl) < dt.datetime.utcnow()
            ping_expired = session.validated + dt.timedelta(minutes=cls.STALE_SESSION_AGE) < dt.datetime.utcnow()

            if ttl_expired:
                # Total session TTL expired, client should have dropped it.
                # Maybe there is a bug in client code?
                context.current.logger.warning("Session %s ttl expired, client should kill the job!", session.token)

            if session.token in jobs:
                # Client has reported a job, touch it to make sure is stays alive
                mapping.OAuthCache.touch(session.token)
                # Now check for state change
                reported_state = jobs.pop(session.token, None)
                actual_state = session.state

                if reported_state != actual_state:
                    reset_reasons.append(
                        "Job {!r} is in {!r} state, while it should be {!r}".format(
                            session.token, reported_state, actual_state
                        )
                    )
            else:
                # Client hasn't reported this job for a while, invalidated it
                if ping_expired:
                    controller.OAuthCache.expire(session)
                    reset_reasons.append(
                        "Session {!r} expired (validated {})".format(session.token, session.validated)
                    )

        if jobs:
            reset_reasons.append(
                "There are no active sessions for jobs: {!r}".format(sorted(jobs))
            )

        if reset_reasons:
            context.current.logger.info("Request client %s to perform validation, reasons: %s", id_, reset_reasons)
            data = {"reason": "\n".join(reset_reasons)}
            return flask.current_app.response_class(
                response=data,
                status=httplib.RESET_CONTENT,
            )

        return "", httplib.NO_CONTENT


class ClientList(ClientRouteBase, RouteV1(v1.client.ClientList)):
    # Mapping of request_param -> (query_builder_param, ordering_param)
    # If `ordering_param` is None -- ordering by this field is not supported
    LIST_QUERY_MAP = {
        "id": ("hostname", None),
        "platform": ("platform", None),
        "alive": ("alive", None),
        "alive_offset": ("alive_offset", None),
        "tags": ("tags", None),
        "search_query": ("search_query", None),
        "busy": ("busy", None),
        "limit": ("limit", None),
        "offset": ("offset", None),
        "fields": ("fields", None)
    }

    @classmethod
    def get_filtered(cls, clients, platform):
        if platform is not None:
            clients = [
                cl
                for cl in clients
                if (
                    platform == common.platform.get_platform_alias(cl.info.get("system", {}).get("platform")) or
                    (cl.lxc and platform in common.platform.LXC_PLATFORMS)
                )
            ]

        return clients

    @classmethod
    def get(cls, query):
        query, offset, limit = cls.remap_query(query)

        fields = query.pop("fields", None)
        busy = query.pop("busy", None)
        alive = query.pop("alive", None)
        alive_offset = query.pop("alive_offset")

        if fields is not None:
            fields = set(fields) | {"id"}
            extra_fields = fields - set(mappers.client.ClientListMapper.ALL_FIELDS)
            if extra_fields:
                raise exceptions.BadRequest("Fields '{}' do not exist.".format(", ".join(extra_fields)))

        if alive is not None:
            alive_after = dt.datetime.utcnow() - dt.timedelta(
                seconds=registry.server.web.mark_client_as_dead_after + alive_offset
            )
            query["update_ts"] = calendar.timegm(alive_after.timetuple()) * (1 if alive else -1)

        platform = query.pop("platform", None)

        # Get all potential matching clients from the database
        clients = controller.Client.list_query(**query).lite().order_by("hostname")

        client_mapper = mappers.client.ClientListMapper(fields=fields)

        running_tasks = None
        # Filter out clients based on platform
        clients = cls.get_filtered(clients, platform)
        if busy is not None:
            running_tasks = mappers.client.ClientListMapper.get_running_tasks(clients)
            clients = [cl for cl in clients if not ((cl.hostname in running_tasks) ^ busy)]

        return v1.schemas.client.ClientList.create(
            offset=offset,
            limit=limit,
            total=len(clients),
            items=client_mapper.dump(clients, offset, limit, running_tasks=running_tasks)
        )


class ClientServiceResources(RouteV1(v1.client.ClientServiceResources)):
    @classmethod
    def get(cls, id_, query):
        limit = query["limit"]
        kind = query["kind"]
        if kind == ctc.RemovableResources.DELETED:
            ret = [res.id for res in controller.Resource.resources_to_remove(id_, limit=limit, replicated=False)]
        elif kind == ctc.RemovableResources.EXTRA:
            limit = limit if limit is not None else 50
            ret = [res.id for res in controller.Resource.extra_resources_to_drop(id_, limit=limit)]
        elif kind == ctc.RemovableResources.REPLICATED:
            ret = [res.id for res in controller.Resource.resources_to_remove(id_, limit=limit, replicated=True)]
        else:
            # Assume small amount of resources on single host. Sort locally since order_by("id") works very slowly
            ret = sorted(mapping.Resource.objects(hosts_states__host=id_).fast_scalar("id"))
        return ret


class ClientServiceResourcesDrop(RouteV1(v1.client.ClientServiceResourcesDrop)):
    @classmethod
    def post(cls, id_, body):
        controller.Resource.drop_host_resources(id_, body)
        return "", httplib.NO_CONTENT


class ClientJob(RouteV1(v1.client.ClientJob)):
    MAX_GET_A_JOB_DURATION = 15  # in seconds

    @common.utils.classproperty
    def __local_mode(self):
        return common.config.Registry().common.installation == ctm.Installation.LOCAL

    @classmethod
    def _tasks_executor_type(cls, resource):
        if resource.type in ctr.SandboxTasksResources:
            image = resource.type == ctr.SandboxTasksResources.SANDBOX_TASKS_IMAGE
            if (
                resource.owner == common.config.Registry().common.service_group and
                resource.attributes_dict().get("auto_deploy")
            ):
                return ctt.ImageType.REGULAR_IMAGE if image else ctt.ImageType.REGULAR_ARCHIVE
            else:
                return ctt.ImageType.CUSTOM_IMAGE if image else ctt.ImageType.CUSTOM_ARCHIVE
        return ctt.ImageType.BINARY

    @classmethod
    def get(cls, id_):
        client = controller.Client.get(id_)
        if not client:
            raise exceptions.NotFound("Client with id '{}' not found".format(id_))

        if ctc.ReloadCommand.RESET in controller.Client.pending_service_commands(client):
            data = {"reason": "Client '{}' is resetting".format(id_)}
            return flask.current_app.response_class(
                response=data,
                status=httplib.RESET_CONTENT,
            )

        return [
            v1.schemas.client.ClientJob.create(id=job.token, state=job.state, options=job.get_update_options())
            for job in mapping.OAuthCache.client_sessions(client.hostname)
        ]

    @classmethod
    def _restore_tasks_from_sessions(cls, client, sessions):
        task_tokens = {s.task_id: s.token for s in sessions}
        context.current.logger.info(
            "Reassigning tasks #%s to client %r",
            ",".join(it.imap(str, task_tokens)) if task_tokens else task_tokens,
            client.hostname
        )
        for task in mapping.Task.objects(id__in=task_tokens.keys()).lite():
            try:
                arch = controller.TaskQueue.get_task_platform(client, task)
            except controller.TaskQueue.NoPlatform:
                arch = None
            yield task, arch, task_tokens.get(task.id)

    @classmethod
    def _make_execute_args(cls, client, task, token, session, arch, features, restored):
        if arch is None:
            arch = task.requirements.platform or "any"

        execute_args = {
            "task_id": task.id,
            "id": token,
            "dns": task.requirements.dns,
            "kill_timeout": controller.Task.kill_timeout(task),
            "arch": "" if arch == "any" else arch,
            "vault_key": base64.b64encode(session.vault),
            "iteration": controller.Task.iteration(task),
            "reserved_space": task.requirements.disk_space << 20,
            "ram": task.requirements.ram,
            "cores": task.requirements.cores or 0,
            "ramdrive": (
                {"type": task.requirements.ramdrive.type, "size": task.requirements.ramdrive.size}
                if task.requirements.ramdrive and task.execution.status in ctt.Status.Group.ASSIGNABLE else
                None
            ),
            "features": features,
        }

        if client.lxc and task.execution.status in ctt.Status.Group.ASSIGNABLE:
            # TODO: SANDBOX-2869: Execute non-container tasks via plain "execute" command.
            # TODO: But container tasks via brand-new "execute_lxc" one.
            cnt = controller.Task.needs_container(task)
            if cnt and cnt is not True:
                try:
                    res_id, platform, venv_res = controller.Task.get_container_platform(task)
                except common.errors.ResourceNotFound:
                    # send empty platform/venv data so that client can reject it
                    res_id, platform, venv_res = cnt, None, None
                execute_args["container"] = {
                    "id": res_id,
                    "alias": platform,
                    "venv_rid": venv_res and venv_res["id"]
                }
            else:
                platform, res = controller.Client.container(client, arch)
                if res:
                    venv_res = controller.Resource.venv_resource(platform)
                    execute_args["container"] = {
                        "id": res["id"],
                        "alias": platform,
                        "venv_rid": venv_res and venv_res["id"]
                    }

        if client.porto and task.execution.status in ctt.Status.Group.ASSIGNABLE:
            if bool(task.requirements.porto_layers):
                layers, platform = controller.Task.get_porto_container_properties(task)
            else:
                lxc_container = controller.Task.needs_container(task)
                # TODO: Support LXC container on singleslot PORTOD (https://st.yandex-team.ru/SANDBOX-9561)
                if client.multislot and lxc_container and lxc_container is not True:
                    res_id, platform, _ = controller.Task.get_container_platform(task)
                    if platform:
                        platform = common.platform.get_platform_alias(platform)
                    layers = [res_id]
                else:
                    # Use default porto runtime
                    platform, res = controller.Client.container(client, arch)
                    layers = [res["id"]]
            venv_res = controller.Resource.venv_resource(platform)
            execute_args["container"] = {
                "layers": layers,
                "alias": platform,
                # TODO: This is a hack.
                # Server should perform containers checks before creating session
                # and move task to EXCEPTION if container is invalid.
                "venv_rid": venv_res.get("id") if not cls.__local_mode else None,
            }

        if controller.Task.tasks_archive_resource(task):
            tasks_rid = controller.Task.tasks_archive_resource(task)
            tasks_resource = controller.Resource.get(tasks_rid)
            exec_type = cls._tasks_executor_type(tasks_resource) if tasks_resource else ctt.ImageType.INVALID
        else:
            # FIXME: SANDBOX-5927
            if str(ctc.Tag.Group.LINUX) in client.tags and not task.requirements.privileged:
                tasks_rid = controller.Resource.tasks_image.get("id")
                exec_type = ctt.ImageType.REGULAR_IMAGE
            else:
                tasks_rid = controller.Resource.tasks_resource.get("id")
                exec_type = ctt.ImageType.REGULAR_ARCHIVE

        execute_args["tasks_rid"] = tasks_rid
        execute_args["exec_type"] = exec_type

        if task.execution.status == ctt.Status.RELEASING:
            execute_args.update({
                "command": str(ctc.Command.RELEASE),
                "kill_timeout": common.config.Registry().common.task.execution.terminate_timeout,
                "release_params": mapping.to_dict(task.execution.release_params),
            })
        elif task.execution.status == ctt.Status.STOPPING:
            execute_args.update({
                "command": str(ctc.Command.STOP),
                "status": ctt.Status.STOPPED,
                "kill_timeout": common.config.Registry().common.task.execution.terminate_timeout,
            })
        elif task.execution.status in ctt.Status.Group.ASSIGNABLE:
            execute_args["command"] = str(ctc.Command.EXECUTE)
            if task.requirements.privileged:
                execute_args["command"] = str(ctc.Command.EXECUTE_PRIVILEGED)
        else:
            if task.execution.status in (ctt.Status.RELEASED, ctt.Status.DELETED, ctt.Status.STOPPED):
                context.current.logger.warning(
                    "Another client already processed task #%s/%s", task.id, task.execution.status
                )
            else:
                context.current.logger.error("Task #%s has unexpected status %s", task.id, task.execution.status)

            if restored:
                # Don't drop session if it was restored, this request is a stale retry from client
                context.current.logger.warning(
                    "Don't drop session %s, this request is a stale retry from client", session.token
                )
            else:
                # Drop session, because client will retry with the same token and session will be restored
                session.delete()

            return None

        assigned_msg = "Assigned to {} as job {}".format(client.hostname, common.utils.obfuscate_token(session.token))
        if task.execution.status == ctt.Status.ENQUEUED:
            try:
                controller.Task.set_status(
                    task, ctt.Status.ASSIGNED, host=client.hostname, event=assigned_msg, logger=context.current.logger
                )
            except common.errors.UpdateConflict as ex:
                session.delete()
                context.current.logger.warning("Cannot assign task to %s: %s", client.hostname, ex)
                return None
        else:
            controller.Task.audit(mapping.Audit(
                task_id=task.id,
                content=assigned_msg,
                source=ctt.RequestSource.SYS
            ))

        return execute_args

    @classmethod
    def post(cls, id_, body):
        client = controller.Client.get(id_)
        if not client:
            raise exceptions.NotFound("Client with id '{}' not found".format(id_))

        age = body.age
        if age < 1:
            raise exceptions.BadRequest("Protocol version {!r} should be positive number".format(age))

        # Always update client ping timestamp on getajob
        update_data = {
            "system": {
                "free_space": body.disk_free >> 20
            }
        }
        controller.Client.update(client, data=update_data, merge=True)

        tokens = body.tokens
        if not tokens:
            return "", httplib.NO_CONTENT
        tokens = list(common.utils.chain(tokens))

        for token in tokens:
            try:
                assert uuid.UUID(token).hex == token
            except (ValueError, AssertionError):
                raise exceptions.BadRequest("Invalid token: {}".format(token))

        features = [str(tag) for tag in client.tags if tag in ctc.Tag.Group.FEATURES]

        active = [_.task_id for _ in mapping.OAuthCache.client_sessions(client.hostname)]
        context.current.logger.info("Host %r active task sessions: %r", client.hostname, active)

        # Check if sessions with this tokens are already registered
        sessions = {o.token: o for o in mapping.OAuthCache.objects(token__in=tokens)}

        response = []
        if sessions:
            jobs_info = list(cls._restore_tasks_from_sessions(client, sessions.values()))
            restored = True

            for task, arch, token in jobs_info:
                context.current.logger.info(
                    "Restored task #%s from session (status %r, client %r)",
                    task.id, task.execution.status, client.hostname
                )
                session = sessions[token]
                if (
                    task.execution.status in (ctt.Status.STOPPING, ctt.Status.RELEASING) and
                    ctc.Tag.POSTEXECUTE not in client.tags
                ):
                    context.current.logger.warning(
                        "Task #%s want to stop or release on host %s without POSTEXECUTE tag.",
                        task.id, client.hostname
                    )
                    if session.aborted:
                        context.current.logger.warning("Remove aborted session for task %s.", task.id)
                        session.delete()
                    continue
                execute_args = cls._make_execute_args(client, task, token, session, arch, features, restored)
                if execute_args is not None:
                    response.append(execute_args)
        elif controller.Client.pending_service_commands(client):
            if active:
                context.current.logger.info(
                    "Client %r scheduled for reloading, but still has %d active jobs",
                    client.hostname, len(active)
                )
                return "", httplib.NO_CONTENT

            if str(ctc.Tag.Group.LINUX) in client.tags:
                tasks_res = controller.Resource.tasks_image
            else:
                tasks_res = controller.Resource.tasks_resource
            pending_command = controller.Client.next_service_command(client, reset=True)
            options = dict()
            if pending_command.author:
                options["author"] = pending_command.author
            if pending_command.comment:
                options["comment"] = pending_command.comment
            resp = [{
                "arch": "", "command": pending_command.command, "tasks_rid": tasks_res.get("id"), "options": options
            }]
            return flask.current_app.response_class(
                response=json.dumps(resp),
                status=httplib.CREATED,
            )
        else:
            start_time = time.time()
            try:
                jobs_lock = controller.TaskQueue.qclient.lock_jobs(tokens)
                jobs_info = controller.TaskQueue.task_to_execute(
                    client, tokens, body.free, logger=context.current.logger
                )
            except (qerrors.QTimeout, qerrors.QRetry, jerrors.Reconnect):
                raise exceptions.RequestTimeout("Request timeout.")
            except qerrors.QSemaphoresTemporaryUnavailable:
                raise exceptions.ServiceUnavailable("Semaphores temporary unavailable.")
            restored = False

            for job_item in jobs_info:
                if job_item is None:
                    return "", httplib.RESET_CONTENT
                task, arch, token = job_item
                context.current.logger.info(
                    "Assigning task #%s in status %r to client %r with tags %r",
                    task.id, task.execution.status, client.hostname, sorted(client.tags)
                )
                session = mapping.OAuthCache(
                    token=token,
                    login=task.author,
                    owner=task.owner,
                    # TTL of session must be more than ttl of task to prevent accidental session expiration.
                    # Also, add 5 additional minutes to terminate it.
                    ttl=int(controller.Task.kill_timeout(task) * 1.05) + 360,
                    source="{}:{}".format(ctu.TokenSource.CLIENT, client.hostname),
                    app_id=str(task.id),
                    task_id=task.id,
                    state=str(ctt.SessionState.ACTIVE),
                    # generate random key for vault data encryption
                    vault=common.crypto.AES.generate_key(),
                )

                try:
                    session.save(force_insert=True)
                except mapping.NotUniqueError as ex:
                    if "token" in ex.message:
                        context.current.logger.warning("Session for token %s already exists", token)
                    if "task_id" in ex.message:
                        context.current.logger.warning("Session for task %s already exists", task.id)
                    # Conflict has occurred. It's ok, client should just retry
                    raise exceptions.Conflict("Conflict error, retry request.")

                jobs_lock.send(token)  # release job
                sessions[token] = session

                if task.execution.status in (ctt.Status.RELEASING, ctt.Status.STOPPING):
                    controller.TaskStatusNotifierTrigger.create_from_task(task.id)

                context.current.logger.info(
                    "Registered new session %r with TTL '%s' for task #%s bound to client %r",
                    session.token, common.utils.td2str(session.ttl), task.id, client.hostname
                )

                execute_args = cls._make_execute_args(client, task, token, session, arch, features, restored)
                if execute_args is not None:
                    response.append(execute_args)

                if time.time() - start_time > cls.MAX_GET_A_JOB_DURATION:
                    break

            jobs_lock.send(None)  # release all jobs

        client.info.pop("idle", None)
        controller.Client.update(client)

        if not response:
            return "", httplib.NO_CONTENT

        if restored:
            http_code = httplib.OK
        else:
            http_code = httplib.CREATED

        return flask.current_app.response_class(
            response=json.dumps(response),
            status=http_code,
        )

    @classmethod
    def _handle_aborted_session(cls, task, session):
        if (
            task.execution.status == session.abort_reason or
            (task.execution.status == ctt.Status.STOPPED and session.abort_reason == ctt.Status.STOPPING)
        ):
            # Everything is ok: task status is as it should be or task is already stopped on the client
            return

        context.current.logger.warning(
            "Task #%s current status is %s, but should be %s.",
            task.id, task.execution.status, session.abort_reason
        )
        messages = {
            ctt.Status.STOPPING: "Task stop",
            ctt.Status.DELETED: "Task delete",
            ctt.Status.EXPIRED: "Task expired"
        }
        controller.Task.set_status(task, session.abort_reason, event=messages[session.abort_reason], force=True)

    @classmethod
    def _task_is_non_rejectable(cls, task):
        # Don't reject task if it's already in "final" status.
        # This means user code has been executed, all resources are finalized, etc.
        terminal_states = (
            ctt.Status.SUCCESS, ctt.Status.RELEASED, ctt.Status.FAILURE,
            ctt.Status.EXCEPTION, ctt.Status.TIMEOUT, ctt.Status.EXPIRED,
            ctt.Status.STOPPED, ctt.Status.DELETED, ctt.Status.NOT_RELEASED,
        )
        if task.execution.status in terminal_states:
            return True

        # WAIT_* are also considered "final" for one execution step
        if task.execution.status in ctt.Status.Group.WAIT:
            return True

        # task might return from WAIT state while its session is still active
        # which means it can potentially be rejected by client
        if task.execution.status == ctt.Status.ENQUEUING:
            return True

        return False

    @classmethod
    def _reject_task(cls, task, session, reason=None, restart=None, reject_type=None):
        # Do nothing if task is already in "final" state
        if cls._task_is_non_rejectable(task):
            context.current.logger.warning(
                "Host %r rejected task #%s, but it's already in status %s, don't reset it",
                session.client_id, task.id, task.execution.status
            )
            return

        restarts_limit = common.config.Registry().server.services.tasks_enqueuer.maximum_allowed_restarts
        restart_count = mapping.Audit.objects.filter(task_id=task.id, status=ctt.Status.ENQUEUED).count()

        restarts_limit_exceeded = restart_count > restarts_limit
        # Only tasks with default executor can be retried/re-enqueued
        if restarts_limit_exceeded:
            retry_allowed = False
        elif controller.Task.tasks_archive_resource(task) and not restart:
            # FIXME: SANDBOX-6163: Temporary use `lower` here
            executor_type = cls._tasks_executor_type(
                controller.Resource.get(controller.Task.tasks_archive_resource(task))
            )
            retry_allowed = str(executor_type).lower() in map(str.lower, ctt.ImageType.Group.REGULAR)
        else:
            retry_allowed = True

        if session.expired:
            event = "Session expired, revoke task from host '{}'".format(session.client_id)
            rejection_reason = ctc.RejectionReason.SESSION_EXPIRED
        else:
            if reason:
                if reject_type == ctc.RejectionReason.ASSIGNED_TIMEOUT:
                    rejection_reason = reject_type
                else:
                    rejection_reason = ctc.RejectionReason.HOST_REJECTED
                event = "Host '{}' rejected task: {}".format(session.client_id, reason)
            else:
                # TODO: Report this fact somewhere
                context.current.logger.warning(
                    "Host %r rejected task #%s without a good reason", session.client_id, task.id
                )
                event = "Host '{}' rejected task without a reason".format(session.client_id)
                rejection_reason = ctc.RejectionReason.HOST_REJECTED_NO_REASON

        if restarts_limit_exceeded:
            event = "The task exceeded maximum amount of queue entrances ({}); last error: {}".format(
                restarts_limit, event
            )
            rejection_reason = ctc.RejectionReason.RESTART_LIMIT_EXCEEDED

        common.statistics.Signaler().push(dict(
            type=cts.SignalType.TASK_REJECTION,
            task_id=task.id,
            owner=task.owner,
            client_id=session.client_id,
            reason=rejection_reason
        ))

        # Check if task hasn't started executing, so we can safely put it back to Q
        if retry_allowed and task.execution.status == ctt.Status.ASSIGNED:
            context.current.logger.info(
                "Re-enqueue task #%s (%s), host: %s, token: %s",
                task.id, task.execution.status, session.client_id, session.token
            )
            controller.Task.set_status(task, ctt.Status.ENQUEUING, event=event, force=True)
            controller.TaskQueue.qclient.execution_completed(session.token)
            controller.TaskQueue.qclient.prequeue_push(task.id)
            return

        # RELEASING/RELEASED tasks should become NOT_RELEASED.
        if task.execution.status in (ctt.Status.RELEASING, ctt.Status.RELEASED):
            status = ctt.Status.NOT_RELEASED
        else:
            # TEMPORARY is only allowed for currently executing tasks (PREPARING and EXECUTING).
            if retry_allowed and task.execution.status in (ctt.Status.PREPARING, ctt.Status.EXECUTING, ctt.Status.FINISHING):
                status = ctt.Status.TEMPORARY
            elif restarts_limit_exceeded:
                status = ctt.Status.STOPPED
            else:
                status = ctt.Status.EXCEPTION

        context.current.logger.info("Host %r rejected task #%s/%s", session.client_id, task.id, task.execution.status)
        controller.Task.set_status(task, status, event=event, force=True, reset_resources_on_failure=True)

    @classmethod
    def delete(cls, id_, body):
        if not mapping.Client.objects(hostname=id_).count():
            raise exceptions.NotFound("Client {} not found".format(id_))

        token = body.token
        session = mapping.OAuthCache.objects.with_id(token)
        if not session:
            raise exceptions.NotFound("No task session {} registered.".format(token))

        task = mapping.Task.objects.with_id(session.task_id)

        reject = body.reject
        reason = body.reason
        target_status = body.target_status
        wait_targets = body.wait_targets

        context.current.logger.info(
            "Job %s for task #%s in status %s%s (host %s) reported by %s as completed",
            token, task.id, task.execution.status,
            " ({}{})".format(
                target_status,
                ", wait_targets={}".format(wait_targets) if wait_targets else ""
            ) if target_status else "",
            task.execution.host, id_
        )

        if target_status:
            try:
                controller.Task.set_status(task, target_status, event=reason, wait_targets=wait_targets)
            except common.errors.IncorrectStatus as er:
                if task.execution.status not in ctt.Status.Group.TRANSIENT:
                    context.current.logger.warning("Impossible to set status for already finished task: %s", er)
                else:
                    raise exceptions.BadRequest(str(er))

        if session.aborted:
            context.current.logger.info("Task #%s session %s has been aborted", task.id, session.token)
            cls._handle_aborted_session(task, session)
        elif reject:
            # Client "rejected" the job, this can happen in the following cases:
            # * executor failed (e.g. unhandled exception)
            # * job session has been expired
            cls._reject_task(task, session, reason=reason, restart=body.restart, reject_type=body.reject_type)

        # Client is done with this session, can safely remove it
        try:
            completed = controller.TaskQueue.qclient.execution_completed(session.token)
        except (qerrors.QTimeout, qerrors.QRetry, jerrors.Reconnect) as ex:
            # no need to additional retry, TaskQueueValidator gathers not completed jobs
            context.current.logger.error("Error while completing job for task #%s: %r", task.id, ex)
        else:
            if completed:
                context.current.logger.info(
                    "Completed task #%s at %s (qp=%s) with pool %s",
                    completed[0].id, completed[0].finished, completed[0].consumption, completed[0].pool
                )
                controller.Task.close_all_intervals(
                    task, update=True, consumption=completed[0].consumption
                )
                now = dt.datetime.utcnow()
                if task.execution.intervals.execute:
                    execution_interval = task.execution.intervals.execute[-1]
                    execution_host = task.execution.host
                    if execution_host:
                        execution_host_tags = mapping.Client.objects.fast_scalar("tags").with_id(execution_host) or []
                    else:
                        execution_host_tags = []

                    ram = task.requirements.ram or 0
                    if task.requirements.ramdrive:
                        ram += (task.requirements.ramdrive.size or 0)

                    common.statistics.Signaler().push(dict(
                        type=cts.SignalType.TASK_SESSION_COMPLETION,
                        start=execution_interval.start,
                        finish=execution_interval.finish,
                        task_id=task.id,
                        host=task.execution.host
                    ))
                    common.statistics.Signaler().push(dict(
                        type=cts.SignalType.TASK_INTERVALS,
                        date=now,
                        timestamp=now,
                        task_id=task.id,
                        task_owner=task.owner,
                        task_type=task.type,
                        consumption=execution_interval.consumption,
                        duration=execution_interval.duration,
                        start=execution_interval.start,
                        pool=str(completed[0].pool),
                        client_id=execution_host,
                        client_tags=execution_host_tags,
                        privileged=int(task.requirements.privileged or 0),
                        cores=task.requirements.cores,
                        ram=ram,
                        caches=int(task.requirements.caches != []),
                        disk=int(task.requirements.disk_space or 0)
                    ))
            else:
                context.current.logger.error("Empty 'completed' is returned for task #%s", task.id)

        # Activate triggers before deleting session if task is going to WAIT_TASK or WAIT_TIME
        if task.execution.status in (ctt.Status.WAIT_TASK, ctt.Status.WAIT_TIME):
            mapping.TimeTrigger.objects(source=task.id).update(set__activated=True)
            if task.execution.status == ctt.Status.WAIT_TASK:
                mapping.TaskStatusTrigger.objects(source=task.id).update(set__activated=True)
        session.delete()
        return "", httplib.NO_CONTENT


class ClientUserTagBase(object):
    @classmethod
    def __validate(cls, tag):
        if tag not in ctc.Tag.Group.USER:
            raise exceptions.BadRequest("User tag '{}' must start with 'USER_'".format(tag))

        group = mapping.Group.objects(user_tags__name=tag).first()
        if group is None:
            raise exceptions.NotFound("Group with tag '{}' not found.".format(tag))

        if (
            group.name not in controller.Group.get_user_groups(context.current.user) and
            not context.current.user.super_user
        ):
            raise exceptions.Forbidden("User doesn't belong to group {} and can't use tag {}.".format(group.name, tag))

    @classmethod
    def _process_operation(cls, id_, tag, operation):
        cls.__validate(tag)
        client = mapping.Client.objects.with_id(id_)
        if client is None:
            raise exceptions.NotFound("Client '{}' not found.".format(id_))
        try:
            client_tags = controller.Client.update_tags(client, [tag], operation)
        except Exception as ex:
            raise exceptions.BadRequest("Exception in tags updating: {}".format(ex))
        if client_tags is None:
            raise exceptions.ServiceUnavailable("Too many requests for updating tags for this client")
        return "", httplib.NO_CONTENT


class ClientUserTagList(RouteV1(v1.client.ClientUserTagList), ClientUserTagBase):
    @classmethod
    def post(cls, id_, tag):
        return cls._process_operation(id_, tag, controller.Client.TagsOp.ADD)


class ClientUserTag(RouteV1(v1.client.ClientUserTag), ClientUserTagBase):
    @classmethod
    def delete(cls, id_, tag):
        return cls._process_operation(id_, tag, controller.Client.TagsOp.REMOVE)
