"""
Commonly used definitions
"""

from sandbox.web.api import v1

import sandbox.yasandbox.controller

import sandbox.serviceq.types as qtypes


def quota_info(owner, quotas=None):
    """
    Create schema object for quota info of the user or group

    :param owner: task owner, group or user login
    :param quotas: structure returned by TaskQueue.owners_rating(), for caching purpose may be None
    :rtype: sandbox.web.api.v1.schemas.group.GroupQuota
    """
    if quotas is None:
        quotas = sandbox.yasandbox.controller.TaskQueue.owners_rating(owner=owner)
    default_pool_quotas = quotas.get(None)
    if default_pool_quotas:
        rating_total = default_pool_quotas[0]
        rating_index, info = default_pool_quotas[1].get(owner, (None, qtypes.OwnersRatingItem()))
        ret = v1.schemas.user.QuotaInfo.create(
            consumption=v1.schemas.user.QuotaConsumption.create(
                real=info.real_consumption,
                future=info.future_consumption
            ),
            limit=info.quota,
            rating=v1.schemas.user.OwnerRating.create(
                index=rating_index,
                total=rating_total,
                queue_size=info.queue_size,
                executing_jobs=info.executing_jobs
            )
        )
    else:
        ret = v1.schemas.user.QuotaInfo.create(
            consumption=v1.schemas.user.QuotaConsumption.create(
                real=0,
                future=0
            ),
            limit=0,
            rating=v1.schemas.user.OwnerRating.create(
                index=0,
                total=0,
                queue_size=0,
                executing_jobs=0
            )
        )
    pools = []
    for pool, pool_quotas in quotas.items():
        if pool is None:
            continue
        rating_total = pool_quotas[0]
        rating_index, info = pool_quotas[1].get(owner, (None, None))
        if info is None or info.is_default_quota and info.real_consumption == 0:
            continue
        pools.append(v1.schemas.user.QuotaInfoInPool.create(
            pool=pool,
            consumption=v1.schemas.user.QuotaConsumption.create(
                real=info.real_consumption,
                future=info.future_consumption
            ),
            limit=info.quota,
            rating=v1.schemas.user.OwnerRating.create(
                index=rating_index,
                total=rating_total,
                queue_size=info.queue_size,
                executing_jobs=info.executing_jobs
            )
        ))
    ret.pools = pools
    return ret
