import re
import httplib
import mongoengine as me

import six
import flask

from sandbox.web.api import v1

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.user as ctu
import sandbox.common.types.notification as ctn

from sandbox.yasandbox import context, controller
from sandbox.yasandbox.database import mapping

from sandbox.serviceapi import mappers
from sandbox.serviceapi.web import RouteV1, exceptions


class GroupBase(object):
    MAX_USER_TAG_COUNT = 5  # Max number of client tags for group
    USER_TAG_RE = re.compile("\AUSER_[A-Z0-9_]+\Z")  # Regular expression to check user tags

    @classmethod
    def _check_members(cls, members):
        try:
            controller.user.validate_credentials(members)
        except Exception as ex:
            raise exceptions.BadRequest("User validation error: {}".format(ex))

    @classmethod
    def _validate_user_tags(cls, user_tags):
        if user_tags is None:
            return
        if len(user_tags) > cls.MAX_USER_TAG_COUNT:
            raise exceptions.BadRequest("Maximum {} tags allowed for group.".format(cls.MAX_USER_TAG_COUNT))

        unknown_tags = [tag for tag in user_tags if cls.USER_TAG_RE.match(tag) is None]

        if unknown_tags:
            raise exceptions.BadRequest(
                "Tags {} must have prefix 'USER_' and consist of upper case letters, digits and '_'.".format(
                    ", ".join(unknown_tags)
                )
            )

    @staticmethod
    def _check_user_permission(user, group):
        if user.login not in group.get_users() and not user.super_user:
            raise exceptions.Forbidden(
                "User {} is not permitted to view or edit group {}".format(user.login, group.name)
            )


class GroupList(GroupBase, RouteV1(v1.group.GroupList)):
    LIST_QUERY_MAP = {
        "name": ("name", None),
        "parent": ("parent", None),
        "user": ("users", None),
        "user_tag": ("user_tags__name", None),
        "mds_strong_mode": ("mds_strong_mode", None),
        "mds_transfer_resources": ("mds_transfer_resources", None),
        "limit": ("limit", None),
        "offset": ("offset", None),
        "order": ("order_by", None),
        "fields": ("fields", None),
        "abc": ("abc", None),
    }

    @classmethod
    def get(cls, query):
        query, offset, limit = cls.remap_query(query)

        fields = query.pop("fields", None)
        order_by = query.pop("order_by", None)

        if fields is not None:
            fields = set(fields) | {"name"}
            extra_fields = fields - set(mappers.group.GroupMapperBase.ALL_FIELDS)
            if extra_fields:
                raise exceptions.BadRequest("Fields '{}' do not exist.".format(", ".join(extra_fields)))

        db_query = mapping.Group.objects(**query)
        total = db_query.count()
        if order_by:
            db_query = db_query.order_by(*order_by)
        if offset:
            db_query = db_query.skip(offset)
        db_query = db_query.limit(limit)

        quotas = controller.TaskQueue.owners_rating()
        group_mapper = mappers.group.GroupListMapper(fields=fields, quotas=quotas)

        return v1.schemas.group.GroupList.create(
            limit=limit,
            offset=offset,
            total=total,
            items=[group_mapper.dump(group) for group in db_query]
        )

    @classmethod
    def post(cls, body):
        if not common.config.Registry().common.installation != ctm.Installation.LOCAL and not body.abc:
            raise exceptions.BadRequest("ABC service should be set")
        name, abc = body.name, body.abc
        if not controller.Group.check_abc_setting_permissions(context.current.user, body.abc):
            raise exceptions.Forbidden(
                "You must be hardware_resources_manager or product_head to set abc service to {}".format(body.abc)
            )
        sources = [mapping.Group.SyncSource(source=source.source, group=source.group) for source in body.sources or []]
        controller.Group.normalize_sync_sources(sources)
        users = set()
        for source in sources:
            source.content = controller.Group.get_source_content(source)
            users.update(source.get_users())

        if context.current.user.login not in users:
            user_source = mapping.Group.SyncSource(source=ctu.GroupSource.USER, group=context.current.user.login)
            user_source.content = controller.Group.get_source_content(user_source)
            sources = [user_source] + sources
            users.add(context.current.user.login)

        cls._check_members(users)
        cls._validate_user_tags(body.user_tags)

        try:
            group = controller.Group.create(
                mapping.Group(
                    abc=abc,
                    name=name,
                    users=list(users),
                    email=body.email,
                    messenger_chat_id=body.messenger_chat_id,
                    telegram_chat_id=body.telegram_chat_id,
                    juggler_settings=mapping.Group.JugglerSettings(
                        default_host=body.juggler_settings.default_host,
                        default_service=body.juggler_settings.default_service,
                        checks={
                            name: mapping.Group.JugglerSettings.JugglerCheck(
                                host=check.host, service=check.service
                            ) for name, check in six.iteritems(body.juggler_settings.checks) if name in ctn.JugglerCheck
                        }
                    ) if body.juggler_settings else None,
                    sources=sources,
                    user_tags=[mapping.Group.UserTag(name=tag) for tag in set(body.user_tags or [])],
                    mds_strong_mode=body.mds_strong_mode,
                    mds_transfer_resources=body.mds_transfer_resources is ctm.NotExists or body.mds_transfer_resources,
                )
            )
        except (ValueError, me.errors.ValidationError, me.errors.NotUniqueError) as ex:
            raise exceptions.BadRequest(str(ex))

        return mappers.group.GroupMapper().dump(group)


class Group(GroupBase, RouteV1(v1.group.Group)):
    @classmethod
    def get(cls, name):
        group = mapping.Group.objects.with_id(name)
        if group is None:
            raise exceptions.NotFound("Group {} not found.".format(name))
        return mappers.group.GroupMapper().dump(group)

    @classmethod
    def put(cls, name, body):
        group = mapping.Group.objects.with_id(name)
        if not group:
            raise exceptions.NotFound("Document {} not found".format(name))
        cls._check_user_permission(context.current.user, group)
        syncs = body.sources
        if syncs:
            controller.Group.normalize_sync_sources(syncs)
            syncs = [
                mapping.Group.SyncSource(source=sync.source, group=sync.group) for sync in syncs if sync.group
            ]
            users = set()
            for sync in syncs:
                if not cls._validate_source_group(sync.group):
                    raise exceptions.BadRequest("Sync {} contains non-ascii element".format(sync.source))
                sync.content = controller.Group.get_source_content(sync)
                users.update(sync.get_users())
            if not context.current.user.super_user and context.current.user.login not in users:
                raise exceptions.BadRequest(
                    "User {} cannot remove themself from group".format(context.current.user.login)
                )
            group.sources = syncs

        current_abc_group = group.abc
        for field in (
            "email", "abc", "messenger_chat_id", "telegram_chat_id", "mds_strong_mode", "mds_transfer_resources"
        ):
            value = getattr(body, field)
            if value is not ctm.NotExists:
                setattr(group, field, value)

        if body.juggler_settings is not ctm.NotExists:
            group.juggler_settings = mapping.Group.JugglerSettings(
                default_host=body.juggler_settings.default_host,
                default_service=body.juggler_settings.default_service,
                checks={
                    name: mapping.Group.JugglerSettings.JugglerCheck(
                        host=check.host, service=check.service
                    ) for name, check in six.iteritems(body.juggler_settings.checks) if name in ctn.JugglerCheck
                }
            ) if body.juggler_settings else None

        if current_abc_group and not group.abc:
            raise exceptions.Forbidden("It's forbidden to remove ABC group")

        if group.abc != current_abc_group:
            if not controller.Group.check_abc_setting_permissions(context.current.user, body.abc):
                raise exceptions.Forbidden(
                    "You must be hardware_resources_manager or product_head to set abc group to {}".format(body.abc)
                )

            quotas = controller.TaskQueue.qclient.multiple_owners_quota_by_pools([name], return_defaults=False)
            if any(quota.limit is not None for pool, ((_, quota),) in quotas.items() if pool is not None):
                raise exceptions.BadRequest("Cannot change ABC service for group with non default quotas")

            if group.abcd_account:
                config = common.config.Registry().common.abcd
                d_tvm_service_id = config.d_tvm_service_id
                tvm_ticket = common.tvm.TVM().get_service_ticket([d_tvm_service_id])[d_tvm_service_id]
                tvm_auth = common.auth.TVMSession(tvm_ticket)
                d_api = common.rest.Client(config.d_api_url, auth=tvm_auth)
                abc_id = common.abc.abc_service_id(group.abc)
                folder = next(
                    (f for f in d_api.services[abc_id].folders.read()["items"] if f["displayName"] == "default"),
                    None
                )
                if folder:
                    group.abcd_account.folder_id = folder["id"]

        if body.user_tags is not None:
            cls._validate_user_tags(body.user_tags)
            group.user_tags = [mapping.Group.UserTag(name=tag) for tag in set(body.user_tags)]

        if group._get_changed_fields():
            try:
                if not group.get_sources():
                    controller.Group.edit(group)
                else:
                    controller.Group.sync(group)
            except (ValueError, me.errors.ValidationError, me.errors.NotUniqueError) as ex:
                raise exceptions.BadRequest(str(ex))

        return (
            mappers.group.GroupMapper().dump(group)
            if ctm.HTTPHeader.WANT_UPDATED_DATA in flask.request.headers else
            ("", httplib.NO_CONTENT)
        )

    @classmethod
    def delete(cls, name):
        doc = mapping.Group.objects.with_id(name)
        if not doc:
            raise exceptions.NotFound("Document {} not found".format(name))
        cls._check_user_permission(context.current.user, doc)
        doc.delete()
        return "", httplib.NO_CONTENT

    @classmethod
    def _validate_source_group(cls, group):
        try:
            group.decode("ascii")
        except UnicodeDecodeError:
            return False
        return True


class GroupParent(RouteV1(v1.group.GroupParent)):
    @classmethod
    def put(cls, name, body):
        if not context.current.user.super_user:
            raise exceptions.Forbidden("Only administrator can change group parent.")
        parent_group = body.parent
        qc = controller.TaskQueue.qclient
        if qc.quota(name) is None:
            qc.set_quota(name, None)
        if qc.quota(parent_group) is None:
            qc.set_quota(parent_group, None)
        qc.set_parent_owner(name, parent_group)
        return "", httplib.NO_CONTENT
