import mongoengine.errors

import sandbox.common.types.notification as ctn

from sandbox.web.api import v1
from sandbox.yasandbox import context, controller
from sandbox.yasandbox.database import mapping

from sandbox.serviceapi import mappers
from sandbox.serviceapi.web import RouteV1, exceptions


class NotificationList(RouteV1(v1.notification.NotificationList)):
    LIST_QUERY_MAP = {
        "author": ("author", "author"),
        "recipient": ("recipient", "send_to"),
        "transport": ("transport", "transport"),
        "sent": ("sent", "sent"),
        "task_id": ("task_id", "task_id"),
        "created": ("created", "date"),
        "limit": ("limit", None),
        "offset": ("offset", None),
        "order": ("order_by", None),
    }

    @classmethod
    def get(cls, query):
        query, offset, limit = cls.remap_query(query)
        order_by = query.pop("order_by")  # Empty list if not specified
        if not order_by:
            order_by = ["-date"]

        mongo_query = mapping.Notification.objects(**controller.Notification.list_query(**query))

        if order_by:
            mongo_query = mongo_query.order_by(*order_by)
        if offset:
            mongo_query = mongo_query.skip(offset)

        # `limit` is required and thus always present
        mongo_query = mongo_query.limit(limit)
        notification_mapper = mappers.notification.NotificationListItemMapper()

        return v1.schemas.notification.NotificationList.create(
            offset=offset,
            limit=limit,
            total=mongo_query.count(),
            items=[notification_mapper.dump(doc) for doc in mongo_query]
        )

    @classmethod
    def post(cls, body):
        data = {
            "author": context.current.user.login,
            "send_to": body.recipients,
            "subject": body.subject or "Default subject",
            "body": body.body,
            "transport": body.transport,
            "type": body.type or ctn.Type.TEXT,
            "headers": body.headers,
            "charset": body.charset or ctn.Charset.UTF,
            "view": body.view or ctn.View.DEFAULT,
            "task_id": body.task_id,
            "host": body.host,
            "urgent": body.urgent or False,
            "check_status": body.check_status,
            "juggler_tags": body.juggler_tags
        }

        if context.current.request.is_task_session:
            data["task_id"] = data["task_id"] or context.current.request.session.task_id

        if data["send_to"]:
            recipients = data["send_to"]
            try:
                send_to = controller.TaskStatusNotifierTrigger.notification_resolver[body.transport](recipients)
                if recipients and not send_to:
                    raise exceptions.BadRequest(
                        "There are no recipients after filtering {}. Impossible to send message.".format(recipients)
                    )
                data["send_to"] = send_to
            except ValueError as error:
                raise exceptions.BadRequest(str(error))

        try:
            doc = mapping.Notification(**data).save(force_insert=True)
            # cut off microseconds so as mongodb saves datetime accurate to milliseconds
            doc.date = doc.date.replace(microsecond=doc.date.microsecond // 1000 * 1000)
        except mongoengine.errors.ValidationError as ex:
            raise exceptions.BadRequest(str(ex))
        return mappers.notification.NotificationMapper().dump(doc)


class Notification(RouteV1(v1.notification.Notification)):
    @classmethod
    def get(cls, id_):
        doc = mapping.Notification.objects.with_id(id_)
        if doc is None:
            raise exceptions.NotFound("Notification {} not found.".format(id_))
        return mappers.notification.NotificationMapper().dump(doc)
