import abc
import json
import time
import uuid
import logging
import datetime as dt
import functools as ft

import six
import requests
import botocore.exceptions
import library.python.oauth

from .blueprints import v1

from sandbox.common import abc as common_abc
from sandbox.common import mds as common_mds
from sandbox.common import tvm as common_tvm
from sandbox.common import auth as common_auth
from sandbox.common import rest as common_rest
from sandbox.common import config as common_config
from sandbox.common import patterns as common_patterns

from sandbox.yasandbox import context
from sandbox.yasandbox import controller
from sandbox.serviceapi import web
from sandbox.serviceapi.web import exceptions
from sandbox.yasandbox.database import mapping

ALLOWED_TVM_SERVICES = [
    2023015,  # testing
    2023017,  # production
]

PROVIDER = "sandbox"
ACCOUNT_SPACE_SEGMENTATION = "ACCOUNT_TYPE"

COMPUTING_QUOTA = "computing_quota"
COMPUTE_SEGMENT = "COMPUTE"
COMPUTE_CORES = "cores"

STORAGE_QUOTA= "storage_quota"
STORAGE_SEGMENT = "STORAGE"
COMPUTE_MILLICORES = "millicores"
STORAGE_GIBIBYTES = "gibibytes"
STORAGE_TEBIBYTES = "tebibytes"
STORAGE_PEBIBYTES = "pebibytes"

S3_MDS_CLIENT_ID = "7f68d0b447494976b3639dfa486b3637"
S3_MDS_CLIENT_SECRET = "5f7f5c622150459bb9f9b4a0f3c1b813"
SSH_KEY_PATH = "/home/zomb-sandbox/.ssh/id_rsa_server"


def check_auth(method):
    @ft.wraps(method)
    def wrapper(*args, **kws):
        tvm_service = context.current.request.tvm_service
        if tvm_service not in ALLOWED_TVM_SERVICES:
            raise exceptions.Forbidden("Access for TVM service id {} is not allowed".format(tvm_service))
        return method(*args, **kws)
    return wrapper


def log_request(method):
    param_names = method.__code__.co_varnames[1:]
    @ft.wraps(method)
    def wrapper(*args):
        params = ", ".join(
            "=".join((
                name,
                json.dumps(dict(value), cls=common_rest.Client.CustomEncoder) if name == "body" else value
            ))
            for name, value in zip(param_names, args[1:])
            if name != "_"
        )
        logging.debug("%s: %s", args[0].__name__, params)
        ret = method(*args)
        logging.debug(
            "%s: response=%s", args[0].__name__, json.dumps(dict(ret), cls=common_rest.Client.CustomEncoder)
        )
        return ret
    return wrapper


@common_patterns.singleton
def abcd_resources():
    config = common_config.Registry().common.abcd
    d_tvm_service_id = config.d_tvm_service_id
    tvm_ticket = common_tvm.TVM().get_service_ticket([d_tvm_service_id])[d_tvm_service_id]
    tvm_auth = common_auth.TVMSession(tvm_ticket)
    d_api = common_rest.Client(config.d_api_url, auth=tvm_auth)
    provider_id = next(iter(item["id"] for item in d_api.providers[:]["items"] if item["key"] == PROVIDER), None)
    assert provider_id is not None
    resource_types = {
        item["id"]: item["key"]
        for item in d_api.providers[provider_id].resourcesDirectory.resourceTypes[:]["items"]
    }
    segmentations = {
        ss["id"]: {
            "key": ss["key"],
            "segments": {
                s["id"]: s["key"]
                for s in d_api.providers[provider_id].resourcesDirectory.segmentations[ss["id"]].segments[:]["items"]
            }
        }
        for ss in d_api.providers[provider_id].resourcesDirectory.segmentations[:]["items"]
    }
    resources = {}
    for item in d_api.providers[provider_id].resourcesDirectory.resources[:]["items"]:
        resource_type_key = resource_types[item["segmentations"]["resourceTypeId"]]
        resources_by_type = resources.setdefault(resource_type_key, {})
        resources_by_type[item["key"]] = [
            [
                segmentations[seg["segmentationId"]]["key"],
                segmentations[seg["segmentationId"]]["segments"][seg["segmentId"]]
            ]
            for seg in item["segmentations"]["segmentations"]
            if segmentations[seg["segmentationId"]]["key"] != ACCOUNT_SPACE_SEGMENTATION
        ]
    return resources


def match_resource_key(resource_key):
    """
    :type resource_key: v1.schemas.resource.ResourceKey
    :return: resource key, e.g. GENERIC_LINUX_SSD
    :rtype: str
    """
    resources = abcd_resources().get(resource_key.resourceTypeKey)
    if resources is None:
        return
    match_segments = set((seg.segmentationKey, seg.segmentKey) for seg in resource_key.segmentation)
    for key, segments in resources.items():
        if set(map(tuple, segments)) == match_segments:
            return key


@common_patterns.ttl_cache(3600)
def s3_mds_oauth_token():
    return library.python.oauth.get_token(
        client_id=S3_MDS_CLIENT_ID, client_secret=S3_MDS_CLIENT_SECRET, keys=[SSH_KEY_PATH]
    )


class ProvisionsMeta(abc.ABCMeta):
    __subclasses__ = {}

    def __new__(mcs, name, bases, namespace):
        cls = mcs.__base__.__new__(mcs, name, bases, namespace)
        segment = namespace.get("segment")
        if segment is not None:
            mcs.__subclasses__[segment] = cls
        return cls


class Provisions(six.with_metaclass(ProvisionsMeta, object)):
    @classmethod
    def by_segment(cls, segment):
        # type: (str) -> type(Provisions)
        """ Get class by segment """
        sub_cls = cls.__subclasses__.get(segment)
        if sub_cls is None:
            raise exceptions.NotAcceptable("Segment is not supported")
        return sub_cls

    @staticmethod
    @abc.abstractmethod
    def update(name, provisions, last_update, quota_version):
        pass

    @staticmethod
    @abc.abstractmethod
    def fill(accounts):
        pass


class ComputeProvisions(Provisions):
    segment = COMPUTE_SEGMENT

    @staticmethod
    def update(name, provisions, last_update, quota_version):
        updated_provisions = []
        for provision in provisions or ():
            resource_key = match_resource_key(provision.resourceKey)
            if resource_key is None:
                continue
            provided_amount = provision.providedAmount
            if provision.providedAmountUnitKey == COMPUTE_CORES:
                provided_amount *= 1000
            try:
                controller.TaskQueue.qclient.set_quota(
                    name, provided_amount or None, pool=resource_key, use_cores=True
                )
                updated_provisions.append(v1.schemas.provision.Provision.create(
                    resourceKey=provision.resourceKey,
                    providedAmount=provision.providedAmount,
                    providedAmountUnitKey=provision.providedAmountUnitKey,
                    allocatedAmount=provision.providedAmount,
                    allocatedAmountUnitKey=provision.providedAmountUnitKey,
                    lastUpdate=last_update,
                    quotaVersion=quota_version
                ))
            except Exception as ex:
                logging.exception(
                    "Error setting quota for %s in pool %s to %s %s",
                    name, resource_key, provision.providedAmount, provision.providedAmountUnitKey
                )
                raise exceptions.BadRequest(str(ex))
        return updated_provisions

    @staticmethod
    def fill(accounts):
        quotas = controller.TaskQueue.qclient.multiple_owners_quota_by_pools(
            owners=[account.displayName for account in accounts],
            use_cores=True,
            return_defaults=False,
            secondary=True
        )
        for account in accounts:
            provisions = []
            for pool, quotas_by_pool in quotas.items():
                if pool is None:
                    continue
                quotas_by_pool = dict(quotas_by_pool)
                quota = quotas_by_pool.get(account.displayName)
                if quota is None or quota.limit is None:
                    continue
                provisions.append(
                    v1.schemas.provision.Provision.create(
                        resourceKey=v1.schemas.resource.ResourceKey.create(
                            resourceTypeKey=COMPUTING_QUOTA,
                            segmentation=[
                                v1.schemas.resource.ResourceSegmentKey.create(
                                    segmentationKey=segmentation,
                                    segmentKey=segment,
                                )
                                for segmentation, segment in abcd_resources()[COMPUTING_QUOTA][pool]
                            ],
                        ),
                        providedAmount=quota.limit,
                        providedAmountUnitKey=COMPUTE_MILLICORES,
                        allocatedAmount=quota.limit,
                        allocatedAmountUnitKey=COMPUTE_MILLICORES,
                        quotaVersion=account.accountVersion,
                        lastUpdate=account.lastUpdate,
                    )
                )
            account.provisions = provisions


class StorageProvisions(Provisions):
    segment = STORAGE_SEGMENT

    @staticmethod
    def update(name, provisions, last_update, quota_version):
        updated_provisions = []
        for provision in provisions or ():
            resource_key = match_resource_key(provision.resourceKey)
            if resource_key is None:
                continue
            provided_amount = provision.providedAmount
            if provision.providedAmountUnitKey == STORAGE_GIBIBYTES:
                provided_amount <<= 30
            elif provision.providedAmountUnitKey == STORAGE_TEBIBYTES:
                provided_amount <<= 40
            elif provision.providedAmountUnitKey == STORAGE_PEBIBYTES:
                provided_amount <<= 50
            try:
                s3_idm_settings = common_config.Registry().common.mds.s3.idm
                token = s3_mds_oauth_token()
                data = {"max_size": provided_amount, "is_anonymous_read_enabled": True}
                r = requests.patch(
                    "/".join((s3_idm_settings.url, "management/bucket", name)),
                    headers=dict(common_auth.OAuth(token)),
                    data=json.dumps(data)
                )
                if r.status_code >= 400:
                    raise Exception(r.text)
                bucket_stats = common_mds.S3.bucket_stats(name)
                allocated = 0
                if bucket_stats:
                    allocated = bucket_stats["used_space"] >> 30
                updated_provisions.append(v1.schemas.provision.Provision.create(
                    resourceKey=provision.resourceKey,
                    providedAmount=provision.providedAmount,
                    providedAmountUnitKey=provision.providedAmountUnitKey,
                    allocatedAmount=allocated,
                    allocatedAmountUnitKey=STORAGE_GIBIBYTES,
                    lastUpdate=last_update,
                    quotaVersion=quota_version
                ))
            except Exception as ex:
                logging.exception(
                    "Error setting quota for %s to %s %s",
                    name, provision.providedAmount, provision.providedAmountUnitKey
                )
                raise exceptions.BadRequest(str(ex))
        return updated_provisions

    @staticmethod
    def fill(accounts):
        for account in accounts:
            bucket_stats = common_mds.S3.bucket_stats(account.displayName)
            if bucket_stats:
                quota = bucket_stats["max_size"] >> 30
            else:
                quota = 0
            account.provisions = [
                v1.schemas.provision.Provision.create(
                    resourceKey=v1.schemas.resource.ResourceKey.create(
                        resourceTypeKey=STORAGE_QUOTA,
                        segmentation=[
                            v1.schemas.resource.ResourceSegmentKey.create(
                                segmentationKey=segmentation,
                                segmentKey=segment,
                            )
                            for segmentation, segment in abcd_resources()[STORAGE_QUOTA][STORAGE_SEGMENT]
                        ],
                    ),
                    providedAmount=quota,
                    providedAmountUnitKey=STORAGE_GIBIBYTES,
                    allocatedAmount=quota,
                    allocatedAmountUnitKey=STORAGE_GIBIBYTES,
                    quotaVersion=account.accountVersion,
                    lastUpdate=account.lastUpdate,
                )
            ]


class GetAccount(web.QuotaManagementRouteV1(v1.account.GetAccount)):
    @classmethod
    def _get_account(cls, account_id, body, model):
        abc_name = common_abc.abc_service_name(body.abcServiceId)
        obj = model.objects(
            abcd_account__id=account_id, abcd_account__folder_id=body.folderId, abc=abc_name
        ).first()
        if obj is None:
            raise exceptions.NotFound("Account not found")
        last_update = v1.schemas.common.LastUpdate.create(
            timestamp=int(time.mktime(obj.abcd_account.last_update.timestamp.timetuple())),
            author=v1.schemas.common.UserId.create(
                staffLogin=obj.abcd_account.last_update.author,
            ),
            operationId=obj.abcd_account.last_update.operation_id,
        )
        account = v1.schemas.account.Account.create(
            accountId=account_id,
            displayName=obj.name,
            folderId=body.folderId,
            deleted=False,
            accountVersion=obj.abcd_account.version or 0,
            accountsSpaceKey=body.accountsSpaceKey,
            freeTier=False,
            lastUpdate=last_update,
        )
        return account

    @classmethod
    def _get_compute_account(cls, account_id, body):
        account = cls._get_account(account_id, body, mapping.Group)
        if body.withProvisions:
            ComputeProvisions.fill([account])
        return account

    @classmethod
    def _get_storage_account(cls, account_id, body):
        account = cls._get_account(account_id, body, mapping.Bucket)
        bucket = account.displayName
        if body.withProvisions:
            StorageProvisions.fill([account])
        account.displayName = mapping.Bucket.abcd_account_name(bucket)
        return account

    @classmethod
    @check_auth
    @log_request
    def post(cls, _, account_id, body):
        segment = body.accountsSpaceKey.segmentation[0].segmentKey
        if segment == COMPUTE_SEGMENT:
            return cls._get_compute_account(account_id, body)
        elif segment == STORAGE_SEGMENT:
            return cls._get_storage_account(account_id, body)
        raise exceptions.NotAcceptable("Segment is not supported")


class CreateAccount(web.QuotaManagementRouteV1(v1.account.CreateAccount)):
    @classmethod
    def _create_group(cls, body):
        abc_name = common_abc.abc_service_name(body.abcServiceId)
        account_id = str(uuid.uuid4())
        group = mapping.Group(
            name=body.displayName,
            users=[body.author.staffLogin],
            abc=abc_name,
            abcd_account=mapping.ABCDAccount(
                id=account_id,
                folder_id=body.folderId,
                last_update=mapping.ABCDAccount.LastUpdate(
                    author=body.author.staffLogin,
                    operation_id=body.operationId
                )
            )
        )
        created = False
        try:
            controller.user.Group.create(group)
            created = True
        except controller.user.Group.AlreadyExists as ex:
            group = mapping.Group.objects.with_id(body.displayName)
            if group is None or group.abcd_account and body.operationId != group.abcd_account.last_update.operation_id:
                logging.exception("Error while creating group")
                raise exceptions.BadRequest(str(ex))
            if group.abc.lower() != abc_name.lower():
                err_msg = "Group {} belongs to foreign ABC service".format(group.name)
                logging.error(err_msg)
                raise exceptions.BadRequest(err_msg)
            if not group.abcd_account:
                group.abcd_account = mapping.ABCDAccount(
                    id=account_id,
                    folder_id=body.folderId,
                    last_update=mapping.ABCDAccount.LastUpdate(
                        author=body.author.staffLogin,
                        operation_id=body.operationId
                    )
                )
                group.save()
                created = True
        except ValueError as ex:
            raise exceptions.BadRequest(str(ex))
        return created, group

    @classmethod
    def _create_bucket(cls, body):
        abc_name = common_abc.abc_service_name(body.abcServiceId)
        account_id = str(uuid.uuid4())
        bucket_name = "sandbox-{}-{}".format(body.abcServiceId, body.displayName)
        if mapping.Bucket.abcd_account_name(bucket_name) is None:
            raise exceptions.BadRequest("invalid account name: {}".format(body.displayName))
        bucket = mapping.Bucket(
            name=bucket_name,
            abc=abc_name,
            abcd_account=mapping.ABCDAccount(
                id=account_id,
                folder_id=body.folderId,
                last_update=mapping.ABCDAccount.LastUpdate(
                    author=body.author.staffLogin,
                    operation_id=body.operationId
                )
            )
        )
        created = False
        try:
            s3_client = common_mds.S3().s3_client()
            s3_client.create_bucket(Bucket=bucket_name)
            s3_client.put_bucket_lifecycle_configuration(
                Bucket=bucket_name,
                LifecycleConfiguration={
                    "Rules": [
                        {
                            "ID": "Abort3DaysMultiparts",
                            "Filter": {
                                "Prefix": ""
                            },
                            "Status": "Enabled",
                            "AbortIncompleteMultipartUpload": {
                                "DaysAfterInitiation": 3
                            }
                        }
                    ]
                }
            )
            created = True
            bucket.save()
        except botocore.exceptions.ClientError as ex:
            if ex.response["Error"]["Code"] != "BucketAlreadyOwnedByYou":
                raise exceptions.BadRequest(str(ex))
            bucket = mapping.Bucket.objects.with_id(bucket_name)
            if bucket is None or body.operationId != bucket.abcd_account.last_update.operation_id:
                logging.exception("Error while creating bucket")
                raise exceptions.BadRequest(str(ex))
        return created, bucket

    @classmethod
    def _create_account(cls, body, create_object, segment):
        created, obj = create_object(body)
        last_update = v1.schemas.common.LastUpdate.create(
            timestamp=int(time.mktime(obj.abcd_account.last_update.timestamp.timetuple())),
            author=v1.schemas.common.UserId.create(
                staffLogin=obj.abcd_account.last_update.author,
            ),
            operationId=obj.abcd_account.last_update.operation_id,
        )
        if created:
            provisions = Provisions.by_segment(segment).update(
                obj.name, body.provisions, last_update, obj.abcd_account.version
            )
        else:
            account = v1.schemas.account.Account.create(
                displayName=obj.name,
                accountVersion=obj.abcd_account.version or 0,
                lastUpdate=last_update,
            )
            Provisions.by_segment(segment).fill([account])
            provisions = account.provisions
        account = v1.schemas.account.Account.create(
            accountId=obj.abcd_account.id,
            displayName=body.displayName,
            folderId=body.folderId,
            provisions=provisions,
            accountVersion=obj.abcd_account.version,
            accountsSpaceKey=body.accountsSpaceKey,
            lastUpdate=last_update,
            deleted=False,
            freeTier=False
        )
        return account

    @classmethod
    @check_auth
    @log_request
    def post(cls, _, body):
        segment = body.accountsSpaceKey.segmentation[0].segmentKey
        if segment == COMPUTE_SEGMENT:
            create_func = cls._create_group
        elif segment == STORAGE_SEGMENT:
            create_func = cls._create_bucket
        else:
            raise exceptions.NotAcceptable("Segment is not supported")
        return cls._create_account(body, create_func, segment)


class DeleteAccount(web.QuotaManagementRouteV1(v1.account.DeleteAccount)):
    @classmethod
    @check_auth
    @log_request
    def delete(cls, _, account_id, body):
        # TODO
        raise exceptions.NotAcceptable("Not supported")


class RenameAccount(web.QuotaManagementRouteV1(v1.account.RenameAccount)):
    @classmethod
    @check_auth
    @log_request
    def post(cls, _, account_id, body):
        # TODO
        raise exceptions.NotAcceptable("Not supported")


class MoveAccount(web.QuotaManagementRouteV1(v1.account.MoveAccount)):
    @classmethod
    @check_auth
    @log_request
    def post(cls, _, account_id, body):
        # TODO
        raise exceptions.NotAcceptable("Not supported")


class ListAccounts(web.QuotaManagementRouteV1(v1.account.ListAccounts)):
    MAX_LIMIT = 100

    @classmethod
    def _list_accounts(cls, body, offset, model):
        accounts = []
        abc_name = common_abc.abc_service_name(body.abcServiceId)
        kws = {"abcd_account__id__gt": offset}
        if abc_name is not None:
            kws["abc"] = abc_name
        if body.folderId:
            kws["abcd_account__folder_id"] = body.folderId
        for obj in model.objects(**kws).order_by("abcd_account__id").limit(min(body.limit, cls.MAX_LIMIT)):
            if not obj.abcd_account:
                continue
            account_id = obj.abcd_account.id
            folder_id = obj.abcd_account.folder_id
            if not account_id or not folder_id:
                continue
            last_update = v1.schemas.common.LastUpdate.create(
                timestamp=int(time.mktime(obj.abcd_account.last_update.timestamp.timetuple())),
                author=v1.schemas.common.UserId.create(
                    staffLogin=obj.abcd_account.last_update.author,
                ),
                operationId=obj.abcd_account.last_update.operation_id,
            )
            accounts.append(v1.schemas.account.Account.create(
                accountId=account_id,
                displayName=obj.name,
                folderId=folder_id,
                deleted=False,
                accountVersion=obj.abcd_account.version or 0,
                accountsSpaceKey=body.accountsSpaceKey,
                freeTier=False,
                lastUpdate=last_update,
            ))
        return accounts

    @classmethod
    def _list_compute_accounts(cls, body, offset):
        accounts = cls._list_accounts(body, offset, mapping.Group)
        if body.withProvisions:
            ComputeProvisions.fill(accounts)
        return accounts

    @classmethod
    def _list_storage_accounts(cls, body, offset):
        accounts = cls._list_accounts(body, offset, mapping.Bucket)
        if body.withProvisions:
            StorageProvisions.fill(accounts)
        for account in accounts:
            account.displayName = mapping.Bucket.abcd_account_name(account.displayName)
        return accounts

    @classmethod
    @check_auth
    @log_request
    def post(cls, _, body):
        offset = body.pageToken or ""
        accounts = []
        segment = body.accountsSpaceKey.segmentation[0].segmentKey
        if segment == COMPUTE_SEGMENT:
            accounts = cls._list_compute_accounts(body, offset)
        elif segment == STORAGE_SEGMENT:
            accounts = cls._list_storage_accounts(body, offset)
        return v1.schemas.account.ListAccountsResponse.create(
            accounts=accounts,
            nextPageToken=accounts[-1].accountId if accounts else None
        )


class RevokeFreeTier(web.QuotaManagementRouteV1(v1.account.RevokeFreeTier)):
    @classmethod
    @check_auth
    @log_request
    def post(cls, _, account_id, body):
        # TODO
        raise exceptions.NotAcceptable("Not supported")


def update_provision(obj, provisions, author, operation_id, segment):
    if obj.abcd_account.last_update.operation_id == operation_id:
        last_update = v1.schemas.common.LastUpdate.create(
            timestamp=int(time.mktime(obj.abcd_account.last_update.timestamp.timetuple())),
            author=v1.schemas.common.UserId.create(
                staffLogin=obj.abcd_account.last_update.author,
            ),
            operationId=obj.abcd_account.last_update.operation_id,
        )
        account = v1.schemas.account.Account.create(
            displayName=obj.name,
            accountVersion=obj.abcd_account.version or 0,
            lastUpdate=last_update,
        )
        Provisions.by_segment(segment).fill([account])
        provisions = account.provisions
    else:
        # TODO: compare body.knownProvisions with existing quotas and raise BadRequest in case of a mismatch
        last_update = v1.schemas.common.LastUpdate.create(
            timestamp=int(time.time()),
            author=author,
            operationId=operation_id,
        )
        obj.abcd_account.version += 1
        provisions = Provisions.by_segment(segment).update(obj.name, provisions, last_update, obj.abcd_account.version)
        if provisions:
            obj.abcd_account.last_update.timestamp = dt.datetime.fromtimestamp(last_update.timestamp)
            obj.abcd_account.last_update.author = last_update.author.staffLogin
            obj.abcd_account.last_update.operation_id = last_update.operationId
            obj.save()
    return provisions


class UpdateProvision(web.QuotaManagementRouteV1(v1.account.UpdateProvision)):
    @classmethod
    @check_auth
    @log_request
    def post(cls, _, account_id, body):
        segment = body.accountsSpaceKey.segmentation[0].segmentKey
        if segment == COMPUTE_SEGMENT:
            model = mapping.Group
        elif segment == STORAGE_SEGMENT:
            model = mapping.Bucket
        else:
            raise exceptions.NotAcceptable("Segment is not supported")

        abc_name = common_abc.abc_service_name(body.abcServiceId)
        obj = model.objects(abcd_account__id=account_id).first()
        if obj is None:
            raise exceptions.NotFound("Account not found")
        if not abc_name or obj.abc.lower() != abc_name.lower():
            raise exceptions.BadRequest("ABC service does not match the account")
        if body.folderId != obj.abcd_account.folder_id:
            raise exceptions.BadRequest("Folder id does not match the account")
        provisions = update_provision(
            obj, body.updatedProvisions, body.author, body.operationId, segment
        )
        return v1.schemas.provision.UpdateProvisionResponse.create(
            provisions=provisions,
            accountVersion=obj.abcd_account.version,
            accountsSpaceKey=body.accountsSpaceKey
        )


class MoveProvision(web.QuotaManagementRouteV1(v1.account.MoveProvision)):
    @classmethod
    @check_auth
    @log_request
    def post(cls, _, account_id, body):
        segment = body.accountsSpaceKey.segmentation[0].segmentKey
        if segment == COMPUTE_SEGMENT:
            model = mapping.Group
        elif segment == STORAGE_SEGMENT:
            model = mapping.Bucket
        else:
            raise exceptions.NotAcceptable("Segment is not supported")

        src_abc_name = common_abc.abc_service_name(body.sourceAbcServiceId)
        dst_abc_name = common_abc.abc_service_name(body.destinationAbcServiceId)
        src_obj = model.objects(abcd_account__id=account_id).first()
        if src_obj is None:
            raise exceptions.NotFound("Source account not found")
        dst_obj = model.objects(abcd_account__id=body.destinationAccountId).first()
        if dst_obj is None:
            raise exceptions.NotFound("Destination account not found")
        if not src_abc_name or src_obj.abc.lower() != src_abc_name.lower():
            raise exceptions.BadRequest("Source ABC service does not match the account")
        if not dst_abc_name or dst_obj.abc.lower() != dst_abc_name.lower():
            raise exceptions.BadRequest("Destination ABC service does not match the account")
        if body.sourceFolderId != src_obj.abcd_account.folder_id:
            raise exceptions.BadRequest("Source folder id does not match the account")
        if body.destinationFolderId != dst_obj.abcd_account.folder_id:
            raise exceptions.BadRequest("Destination folder id does not match the account")
        provisions = []
        for obj, updated_provisions in (
            (src_obj, body.updatedSourceProvisions),
            (dst_obj, body.updatedDestinationProvisions),
        ):
            provisions.append(update_provision(obj, updated_provisions, body.author, body.operationId, segment))
        src_provisions, dst_provisions = provisions
        return v1.schemas.provision.MoveProvisionResponse.create(
            sourceProvisions=src_provisions,
            destinationProvisions=dst_provisions,
            accountsSpaceKey=body.accountsSpaceKey,
            sourceAccountVersion=src_obj.abcd_account.version,
            destinationAccountVersion=dst_obj.abcd_account.version,
        )
