from __future__ import absolute_import

from sandbox.common import api
from ..api import Api

from . import common
from . import resource
from . import provision


class GetAccountRequest(Api.Schema):
    """ Get account request """
    withProvisions = api.Boolean("Include provisions within returned account")
    includeDeleted = api.Boolean("Return account even if it is deleted")
    folderId = common.FolderId()
    abcServiceId = common.AbcServiceId()
    accountsSpaceKey = resource.AccountsSpaceKey()


class Account(Api.Schema):
    """ Account """
    accountId = common.AccountId()
    key = api.String("Account key")
    displayName = api.String("Account display name")
    folderId = common.FolderId()
    deleted = api.Boolean("'Account is deleted' flag")
    provisions = api.Array(provision.Provision, "Account resource provisions")
    accountVersion = common.AccountVersion()
    accountsSpaceKey = resource.AccountsSpaceKey()
    lastUpdate = common.LastUpdate("Last update for this account")
    freeTier = api.Boolean("'Account is free tier' flag")


class CreateAccountRequest(Api.Schema):
    """ Create account request """
    key = api.String("Account key")
    displayName = api.String("Account display name")
    folderId = common.FolderId()
    abcServiceId = common.AbcServiceId()
    author = common.UserId()
    operationId = common.OperationId()
    accountsSpaceKey = resource.AccountsSpaceKey()
    freeTier = api.Boolean("'Account is free tier' flag")


class CreateAccountAndProvideRequest(CreateAccountRequest):
    """ Create account and provide request """
    provisions = api.Array(provision.ProvisionRequest, "Provided amounts")
    knownProvisions = api.Array(
        provision.KnownAccountProvisions,
        "Last known provided amounts as saved in quota management for each account in the folder"
    )


class DeleteAccountRequest(Api.Schema):
    """ Delete account request """
    folderId = common.FolderId()
    abcServiceId = common.AbcServiceId()
    author = common.UserId()
    operationId = common.OperationId()
    accountsSpaceKey = resource.AccountsSpaceKey()


class RenameAccountRequest(Api.Schema):
    """ Rename account request """
    folderId = common.FolderId()
    abcServiceId = common.AbcServiceId()
    displayName = api.String("Updated account display name")
    author = common.UserId()
    operationId = common.OperationId()
    accountsSpaceKey = resource.AccountsSpaceKey()


class MoveAccountRequest(Api.Schema):
    """ Move account request """
    sourceFolderId = common.FolderId("Source folder id")
    destinationFolderId = common.FolderId("Destination folder id")
    sourceAbcServiceId = common.AbcServiceId("Abc service id of the source folder")
    destinationAbcServiceId = common.AbcServiceId("Abc service id of the destination folder")
    knownSourceProvisions = api.Array(
        provision.KnownAccountProvisions,
        "Last known provided amounts as saved in quota management for all accounts in source folder"
    )
    knownDestinationProvisions = api.Array(
        provision.KnownAccountProvisions,
        "Last known provided amounts as saved in quota management for all accounts in destination folder"
    )
    author = common.UserId()
    operationId = common.OperationId()
    accountsSpaceKey = resource.AccountsSpaceKey()


class ListAccountsRequest(Api.Schema):
    """ List accounts by folder request """
    limit = api.Integer("Limit per page")
    pageToken = api.String("Page token")
    withProvisions = api.Boolean("Include provisions within returned accounts")
    includeDeleted = api.Boolean("Return deleted accounts also")
    accountsSpaceKey = resource.AccountsSpaceKey()
    folderId = common.FolderId()
    abcServiceId = common.AbcServiceId()


class ListAccountsResponse(Api.Schema):
    """ List accounts response """
    accounts = api.Array(Account, "Accounts")
    nextPageToken = api.String("Next page token")


class RevokeFreeTierRequest(Api.Schema):
    """ Free tier revoke request """
    folderId = common.FolderId()
    abcServiceId = common.AbcServiceId()
    author = common.UserId()
    operationId = common.OperationId()
    accountsSpaceKey = resource.AccountsSpaceKey()
