import six
import json
import math
import uuid
import httplib
import datetime as dt
import collections

import flask

from sandbox.common.types import misc as ctm
from sandbox.common.types import resource as ctr
from sandbox.common import abc as common_abc
from sandbox.common import mds as common_mds
from sandbox.common import tvm as common_tvm
from sandbox.common import auth as common_auth
from sandbox.common import hash as common_hash
from sandbox.common import rest as common_rest
from sandbox.common import config as common_config
from sandbox.common import errors as common_errors
from sandbox.common import patterns as common_patterns

from sandbox import sdk2

from sandbox.web.api import v1, v2
from sandbox.yasandbox import context, controller
from sandbox.yasandbox.database import mapping

from sandbox.serviceapi import mappers
from sandbox.serviceapi import constants as sa_consts
from sandbox.serviceapi.web import RouteV1, RouteV2, exceptions


def get_resource(resource_id):
    """
    Return resource object by ids ID
    :param resource_id: int
    :return: mapping.Resource
    :raises: exceptions.NotFound
    """

    # Resource hosts are requested separately by `ResourceLinks` if needed
    doc = mapping.Resource.objects.lite().exclude("hosts_states").with_id(resource_id)
    if doc is None:
        raise exceptions.NotFound("Resource #{} not found.".format(resource_id))

    return doc


class ResourceBase(object):
    @common_patterns.singleton_classproperty
    def legacy_client(cls):
        client = common_rest.Client(
            sa_consts.LegacyAPI.url + "api/v1.0", total_wait=0
        )
        client.RETRYABLE_CODES = ()
        return client

    @classmethod
    def resource_meta(cls, resource_type):
        if resource_type in sdk2.Resource:
            return sdk2.Resource[resource_type].__getstate__()
        try:
            return cls.legacy_client.resource.meta_list[resource_type].read()
        except cls.legacy_client.HTTPError as ex:
            if ex.status == httplib.SERVICE_UNAVAILABLE:
                raise exceptions.ServiceUnavailable("Legacy server temporary unavailable")
            if ex.status == httplib.NOT_FOUND:
                raise exceptions.BadRequest("Unknown resource type {}".format(resource_type))
            raise exceptions.BadRequest(str(ex))
        except cls.legacy_client.TimeoutExceeded:
            raise exceptions.ServiceUnavailable("Legacy server temporary unavailable")


class Resource(RouteV1(v1.resource.Resource), ResourceBase):
    @classmethod
    def get(cls, id_):
        # Resource hosts are requested separately by `ResourceLinks` if needed
        doc = get_resource(id_)
        if ctm.HTTPHeader.TOUCH_RESOURCE in flask.request.headers:
            controller.Resource.touch(doc.id)

        return mappers.resource.SingleResourceMapper().dump(doc)

    @classmethod
    def _update_data(cls, body):
        result = {}
        for name in body.__fields__.iterkeys():
            value = getattr(body, name)
            if value is not ctm.NotExists:
                result[name] = value
        return result

    @classmethod
    def put(cls, id_, body):
        resource = mapping.Resource.objects.with_id(id_)

        if resource is None:
            raise exceptions.NotFound("Resource #{} not found".format(id_))

        data = cls._update_data(body)
        if not data:
            controller.Resource.touch(id_)
            return "", httplib.NO_CONTENT

        full_update = any(_ not in controller.Resource.SHORT_UPDATE_FIELDS for _ in data)

        task_id, client_id = None, None
        if context.current.request.is_task_session:
            task_id, client_id = context.current.request.session.task_id, context.current.request.session.client_id
        elif context.current.user.super_user:
            task_id = resource.task_id

        if task_id is None:
            if not controller.user_has_permission(context.current.user, (resource.owner,)):
                raise exceptions.Forbidden(
                    "User `{}` is not permitted to modify resource #{}".format(context.current.user.login, id_)
                )

            if full_update:
                raise exceptions.Forbidden(
                    "It's not allowed to modify {} fields beyond task session scope. "
                    "Modification of {} is permitted.".format(
                        set(data) - set(controller.Resource.SHORT_UPDATE_FIELDS),
                        controller.Resource.SHORT_UPDATE_FIELDS
                    )
                )

        resource_meta = None
        if full_update:
            resource_meta = controller.Resource.resource_meta(resource)
            if resource_meta is None:
                resource.hashes, resource_meta = controller.Resource.update_meta_objects(
                    cls.resource_meta(resource.type)
                )

                resource_meta = resource_meta[0]
                resource.save()

        if task_id is not None and full_update and not context.current.user.super_user:
            allowed_ids = mapping.Task.objects(id=task_id).scalar("id", "parent_id")[0]
            if resource.task_id not in allowed_ids:
                raise exceptions.Forbidden(
                    "Resource #{} created by task #{} cannot be changed in scope of task #{}.".format(
                        resource.id, resource.task_id, task_id
                    )
                )

            if resource.state != ctr.State.NOT_READY:
                try:
                    updated = controller.Resource.update(resource, data, client_id, resource_meta, db_update=False)
                    if updated:
                        raise exceptions.BadRequest(
                            "Modification of resource #{} in state {!r} is not allowed.".format(
                                resource.id, resource.state
                            )
                        )
                    else:
                        context.current.logger.warning("Resource #%s re-modification detected.", resource.id)
                        return "", httplib.NO_CONTENT
                except (common_errors.TaskError, TypeError, ValueError) as ex:
                    raise exceptions.BadRequest(str(ex))

        try:
            controller.Resource.update(resource, data, client_id, resource_meta, full_update=full_update)
        except (common_errors.TaskError, TypeError, ValueError) as ex:
            raise exceptions.BadRequest(str(ex))
        return "", httplib.NO_CONTENT


class ResourceDataRsync(RouteV1(v1.resource.ResourceDataRsync)):
    @classmethod
    def get(cls, id_):
        doc = get_resource(id_)
        rl = controller.ResourceLinks([doc])
        return rl.rsync_verbose[id_]


class ResourceDataHttp(RouteV1(v1.resource.ResourceDataHttp)):
    @classmethod
    def get(cls, id_):
        doc = get_resource(id_)
        rl = controller.ResourceLinks([doc])
        return rl.http_verbose[id_]


class Cache(object):
    """ The class is a wrapper for resource cache entry """

    def __init__(self, query_kwargs):
        self.query_hash = common_hash.json_hash(query_kwargs, sort_lists=True)
        self.resource_id = None
        self.updated = None

        cache = (
            mapping.ResourceCache.objects
            .filter(query_hash=self.query_hash)
            .fast_scalar("resource_id", "updated")
            .first()
        )
        if cache:
            self.resource_id, self.updated = cache

    def update(self, resource_id, force=False):
        """
        Update or create cache entry for the given query

        :param resource_id: new resource ID
        :param force: ignore TTL of existing entry and update anyway
        """
        if resource_id == self.resource_id:
            return

        # Do not update recent cache entries unless forced to do so
        if not force and self.updated and self.updated > mapping.ResourceCache.ttl_threshold():
            return

        context.current.logger.debug(
            "[%s] Resource cache: update, resource_id=%r, force=%r", self.query_hash[:8], resource_id, force
        )

        mapping.ResourceCache.objects.filter(query_hash=self.query_hash).update(
            upsert=True, resource_id=resource_id, updated=dt.datetime.utcnow()
        )


class ResourceListBase(object):
    # Mapping of request_param -> (query_builder_param, ordering_param)
    # If `ordering_param` is None -- ordering by this field is not supported
    LIST_QUERY_MAP = controller.Resource.LIST_QUERY_MAP

    Result = collections.namedtuple("Result", ("query", "resources", "limit", "offset"))

    @classmethod
    def _prepare_resource_query(cls, query):
        query, offset, limit = cls.remap_query(query, save_query=True)

        # TODO: This validation should performed via schema
        if limit > 3000 and not context.current.user.super_user:
            raise exceptions.BadRequest("Too big amount of data requested.")

        try:
            return controller.Resource.prepare_api_query(query)
        except ValueError as ex:
            raise exceptions.BadRequest(str(ex))

    @classmethod
    def _get_by_query(cls, query):

        logger = context.current.logger

        # Parse query, convert fields names, etc.
        mongo_query, query_kwargs, offset, limit = cls._prepare_resource_query(query)

        # Trying to use cache if the query is for single most recent resource that meets the given parameters.
        # See SANDBOX-7322 for details and motivation.

        use_cache = (
            query_kwargs is not None
            and not (set(query_kwargs) & {"id", "id__in", "task_id__in", "task_id"})  # such queries are already fast
            and mongo_query._ordering == [("_id", -1)]
            and limit == 1 and offset == 0
        )

        if not use_cache:
            return cls.Result(resources=list(mongo_query), query=mongo_query, limit=limit, offset=offset)

        resources = None

        try:
            cache = Cache(query_kwargs)
        except Exception:
            logger.error("Resource cache: failed to hash query, fallback")
            return cls.Result(resources=list(mongo_query), query=mongo_query, limit=limit, offset=offset)

        # If cache entry exists, narrow the query down using the cached resource ID as inclusive lower bound
        if cache.resource_id:
            logger.debug("[%s] Resource cache: check resources with id >= %r", cache.query_hash[:8], cache.resource_id)
            resources = list(mongo_query.filter(id__gte=cache.resource_id))

            if resources:
                logger.debug("[%s] Resource cache: hit with id=%r", cache.query_hash[:8], resources[-1].id)
            else:
                logger.debug("[%s] Resource cache: miss, fallback to the original query", cache.query_hash[:8])

        force_cache_update = False

        if not resources:
            # Either
            # * cache entry was not found, or
            # * cache miss: the most recent resource ID is now older than the cached one.
            #   In this supposedly rare situation we have to fallback to the original query.
            resources = list(mongo_query)

            # Update the cache to prevent the miss in the future.
            force_cache_update = True

        if resources:
            cache.update(resources[-1].id, force=force_cache_update)

        return cls.Result(resources=resources, query=mongo_query, limit=limit, offset=offset)


class ResourceListV1(ResourceListBase, RouteV1(v1.resource.ResourceList), ResourceBase):
    @classmethod
    def get(cls, query):
        # Parse query, convert fields names, etc.
        result = cls._get_by_query(query)
        return v1.schemas.resource.ResourceList.create(
            offset=result.offset,
            limit=result.limit,
            total=result.query.count(),
            items=mappers.resource.ResourceListMapper().dump(result.resources),
        )

    @classmethod
    def post(cls, body):
        if not context.current.request.is_task_session:
            raise exceptions.Forbidden("The method is allowed only for task session scope.")

        if not body.arch:
            body.arch = ctm.OSFamily.ANY
        if not body.state:
            body.state = ctr.State.NOT_READY

        task_obj = controller.Resource.ShortTask(
            *mapping.Task.objects.fast_scalar("id", "author", "owner", "parent_id").with_id(
                context.current.request.session.task_id
            )
        )

        attributes = body.attributes
        for_parent = body.for_parent

        if for_parent:
            if not task_obj.parent_id:
                raise exceptions.BadRequest(
                    "Cannot create resource for parent of task #{}: task has no parent".format(
                        context.current.request.session.task_id
                    )
                )
            task_id = task_obj.parent_id
        else:
            task_id = task_obj.id

        resource_meta = (
            [dict(meta) for meta in body.resource_meta]
            if body.resource_meta else
            cls.resource_meta(body.type)
        )

        system_attributes = {}
        if body.system_attributes:
            for pl in ctm.OSFamily.Group.BUILDABLE:
                pl_name = pl + "_platform"
                path = getattr(body.system_attributes, pl_name, None)
                if path:
                    system_attributes[pl_name] = path

        try:
            resource = controller.Resource.create(
                body.description, body.file_name or "", body.md5, body.type, task_id, resource_meta, task_obj,
                skynet_id=body.skynet_id,
                attrs=attributes,
                state=body.state,
                arch=body.arch,
                mds=dict(body.mds) if body.mds else None,
                multifile=bool(body.multifile),
                executable=bool(body.executable),
                system_attributes=system_attributes
            )
        except common_errors.TaskError as ex:
            raise exceptions.BadRequest(str(ex))

        return mappers.resource.SingleResourceMapper().dump(resource)


class ResourceListV2(ResourceListBase, RouteV2(v2.resource.ResourceList)):
    @classmethod
    def get(cls, query):
        # Parse query, convert fields names, etc.
        result = cls._get_by_query(query)
        return v1.schemas.resource.ResourceList.create(
            offset=result.offset,
            limit=result.limit,
            total=0,
            items=mappers.resource.ResourceListMapper().dump(result.resources),
        )


class ResourceAttributeBase(object):
    @classmethod
    def get_short_resource(cls, resource_id):
        """
        Returns short resource view for attributes actions

        :param resource_id: Id of resource
        :return: List[Tuple]
        """
        doc = mapping.Resource.objects.fast_scalar("id", "attributes", "owner").with_id(resource_id)
        if doc is None:
            raise exceptions.NotFound("Resource #{} not found.".format(resource_id))
        return doc

    @classmethod
    def get_attrs(cls, resource):
        """
        Return a list of resource attributes as key-value tuples

        :param resource: tuple of rid, attributes and owner
        :return: List[Tuple]
        :raises: exceptions.NotFound
        """
        return [(a.get("k"), a.get("v")) for a in (resource[1] or [])]

    @classmethod
    def cast(cls, value):
        return value.encode("utf-8") if isinstance(value, unicode) else str(value)

    @classmethod
    def _is_valid_ttl(cls, value):
        if value:
            try:
                value = float(value)
            except ValueError:
                return False
            return math.isinf(value) or 1 <= value < 1000
        return False

    @classmethod
    def check_permissions(cls, resource):
        # TODO: SANDBOX-9063
        if context.current.request.is_task_session or context.current.user.super_user:
            return
        if not controller.user_has_permission(context.current.user, (resource[2],)):
            raise exceptions.Forbidden(
                "User `{}` is not permitted to modify resource #{}".format(context.current.user.login, resource[0])
            )


class ResourceAttribute(ResourceAttributeBase, RouteV1(v1.resource.ResourceAttribute)):
    @classmethod
    def get(cls, id_):
        return [
            v1.schemas.resource.ResourceAttribute.create(name=k, value=v)
            for k, v in cls.get_attrs(cls.get_short_resource(id_))
        ]

    @classmethod
    def post(cls, id_, body):
        doc = cls.get_short_resource(id_)
        cls.check_permissions(doc)
        attr_name = body.name.strip()
        attr_value = cls.cast(body.value).strip()

        if any(key == attr_name for key, _ in cls.get_attrs(doc)):
            raise exceptions.Conflict("Attribute '{}' already exists".format(attr_name))

        if attr_name == "ttl" and not cls._is_valid_ttl(attr_value):
            raise exceptions.BadRequest(
                "Attribute 'ttl' is not valid: should be either 'inf' or a positive number < 1000"
            )

        attr_doc = mapping.Resource.Attribute(key=attr_name, value=attr_value)
        updated = mapping.Resource.objects(
            id=id_,
            attributes__key__ne=attr_name,
        ).update_one(push__attributes=attr_doc, set__time__updated=dt.datetime.utcnow())

        if updated:
            context.current.logger.info("Attribute for resource #%s created: %s=%s", id_, attr_name, attr_value)
            controller.Resource.resource_audit("Add attribute {}={}".format(attr_name, attr_value), resource_id=id_)
        else:
            raise exceptions.Conflict("Attribute '{}' already exists".format(attr_name))

        return v1.schemas.resource.ResourceAttribute.create(name=attr_name, value=attr_value)


class ResourceAttributeName(ResourceAttributeBase, RouteV1(v1.resource.ResourceAttributeName)):
    @classmethod
    def put(cls, id_, name, body):
        doc = cls.get_short_resource(id_)
        cls.check_permissions(doc)
        attr_doc = None
        for key, value in cls.get_attrs(doc):
            if key == name:
                attr_doc = mapping.Resource.Attribute(key=key, value=value)
                break
        if not attr_doc:
            raise exceptions.NotFound("Attribute '{}' does not exist".format(name))
        if body.name:
            attr_doc.key = body.name
        if body.value is not ctm.NotExists:
            attr_doc.value = cls.cast(body.value)
        if attr_doc.key == "ttl" and not cls._is_valid_ttl(attr_doc.value):
            raise exceptions.BadRequest(
                "Attribute 'ttl' is not valid: should be either 'inf' or a positive number < 1000"
            )

        updated = mapping.Resource.objects(id=id_, attributes__key=name).update_one(
            set__attributes__S=attr_doc,
            set__time__updated=dt.datetime.utcnow()
        )
        if updated:
            context.current.logger.info("Attribute for resource #%s updated: %s=%s", id_, attr_doc.key, attr_doc.value)
            controller.Resource.resource_audit(
                "Update attribute {}={}".format(attr_doc.key, attr_doc.value),
                resource_id=id_
            )
        else:
            raise exceptions.NotFound("Attribute '{}' does not exist".format(name))

        return "", httplib.NO_CONTENT

    @classmethod
    def delete(cls, id_, attr_name):
        doc = cls.get_short_resource(id_)
        cls.check_permissions(doc)
        if attr_name not in (key for key, _ in cls.get_attrs(doc)):
            raise exceptions.NotFound("Attribute '{}' does not exist".format(attr_name))
        mapping.Resource.objects(
            id=id_,
            attributes__key=attr_name
        ).update_one(
            pull__attributes__key=attr_name,
            set__time__updated=dt.datetime.utcnow()
        )
        controller.Resource.resource_audit("Remove attribute {}".format(attr_name), resource_id=id_)
        return "", httplib.NO_CONTENT


class ResourceSourceV1(RouteV1(v1.resource.ResourceSource)):
    @classmethod
    def post(cls, id_, body):
        if context.current.request.is_task_session:
            host = context.current.request.session.client_id
            mds = None
        else:
            host = body.host
            mds = body.mds
        if not host and not mds:
            raise exceptions.BadRequest("New resource source must be defined.")

        resource = mapping.Resource.objects.with_id(id_)
        if resource is None:
            raise exceptions.NotFound("Resource #{} not found.".format(id_))
        if mds:
            controller.Resource.set_mds(resource, dict(mds))
            resource.save()
        else:
            controller.Resource.add_host(resource, host)

        return flask.current_app.response_class(
            status=httplib.NO_CONTENT,
            headers={"Location": flask.request.path}
        )

    @classmethod
    def delete(cls, id_, host):
        resource = get_resource(id_)
        if resource is None:
            raise exceptions.NotFound("Resource #{} not found.".format(id_))
        if not controller.user_has_permission(context.current.user, (resource.author, resource.owner)):
            raise exceptions.Forbidden(
                "User `{}` is not permitted to delete resource #{}".format(context.current.user.login, id_)
            )

        controller.Resource.remove_host(resource, host)

        return "", httplib.NO_CONTENT


class ResourceTouch(RouteV1(v1.resource.ResourceTouch)):
    @classmethod
    def post(cls, id_):
        controller.Resource.touch(id_)
        return "", httplib.NO_CONTENT


class ResourceLink(RouteV1(v1.resource.ResourceLink)):
    LIST_QUERY_MAP = {
        "resource_id": ("resource_id", "resource_id")
    }

    @classmethod
    def get(cls, query):
        query, offset, limit = cls.remap_query(query)
        order_by = query.pop("order_by")
        links = mapping.ResourceLink.objects(author=context.current.user.login, **query)
        total = links.count()
        if order_by:
            links = links.order_by(*order_by)

        links = links.limit(limit).skip(offset)

        return v1.schemas.resource.ResourceLinkList.create(
            items=[
                v1.schemas.resource.ResourceProxyLink.create(
                    id=link.id,
                    author=link.author,
                    accessed=link.accessed,
                    resource_id=link.resource_id,
                    link=controller.Resource.proxy_temporary_link(link)
                ) for link in links
            ],
            offset=offset,
            limit=limit,
            total=total,
        )

    @classmethod
    def post(cls, body):
        if not mapping.Resource.objects(id=body.resource_id).count():
            raise exceptions.NotFound("Resource {} not found.".format(body.resource_id))
        link = controller.Resource.create_temporary_link(body.resource_id, context.current.user.login)

        return v1.schemas.resource.ResourceProxyLink.create(
            id=link.id,
            author=link.author,
            accessed=link.accessed,
            resource_id=link.resource_id,
            link=controller.Resource.proxy_temporary_link(link),
        )

    @classmethod
    def put(cls, body):
        link = mapping.ResourceLink.objects.with_id(body.id)

        if link is None:
            raise exceptions.NotFound("Link {} not found.".format(body.id))

        controller.Resource.update_resource_temporary_link(link.id)

        return v1.schemas.resource.ResourceLink.create(
            id=link.id,
            author=link.author,
            accessed=link.accessed,
            resource_id=link.resource_id
        )


class ResourcesToBackup(RouteV1(v1.resource.ResourcesToBackup)):
    @classmethod
    def put(cls, client, body):
        backup_interval = common_config.Registry().common.resources.backup_interval
        till = dt.datetime.utcnow() + dt.timedelta(seconds=backup_interval * 4)
        updated = mapping.ResourceForBackup.objects(
            id__in=body.resources,
            lock=client,
        ).update(
            set__till=till,
        )
        if updated == len(body.resources):
            cancelled = []
        else:
            cancelled = list(
                mapping.ResourceForBackup.objects(id__in=body.resources, lock__ne=client).fast_scalar("id")
            )
        need_upload = list(
            mapping.ResourceForBackup.objects(sources=client, lock=None).order_by("id").fast_scalar("id")
        )
        return v1.schemas.resource.ResourcesToBackupResponse.create(
            resources=need_upload,
            cancelled=cancelled,
        )

    @classmethod
    def post(cls, client, body):
        backup_interval = common_config.Registry().common.resources.backup_interval
        till = dt.datetime.utcnow() + dt.timedelta(seconds=backup_interval * 4)
        updated = mapping.ResourceForBackup.objects(
            id__in=body.resources,
            lock=None,
        ).update(
            set__lock=client,
            set__till=till,
        )
        if updated == len(body.resources):
            locked = body.resources
        else:
            resources_in_queue = list(mapping.ResourceForBackup.objects(
                id__in=body.resources
            ).fast_scalar("id", "lock"))
            locked = [rid for rid, lock in resources_in_queue if lock == client]
            new = set(body.resources)
            new -= {rid for rid, _ in resources_in_queue}
            resources = list(mapping.Resource.objects(
                id__in=new,
                read_preference=mapping.ReadPreference.SECONDARY,
            ).fast_scalar("id", "state", "size", "hosts_states"))
            for rid, state, size, hosts_states in resources:
                if state == ctr.State.NOT_READY and not hosts_states:
                    hosts_states = [{"h": client, "st": ctr.HostState.OK}]
                hosts = {item["h"] for item in hosts_states or () if item["st"] == ctr.HostState.OK}
                try:
                    mapping.ResourceForBackup(
                        id=rid,
                        size=size,
                        sources=hosts,
                        lock=client,
                        till=till,
                    ).save()
                except mapping.NotUniqueError:
                    pass
                else:
                    locked.append(rid)
        return locked


class LockResourceForMds(RouteV1(v1.resource.LockResourceForMds)):
    @classmethod
    def post(cls, _id, body):
        qclient = controller.TaskQueue.qclient
        return v1.schemas.resource.ResourceLockForMdsResponse.create(result=(
            qclient.acquire_resource_lock(_id, body.host)
            if body.acquire else
            qclient.release_resource_lock(_id, body.host)
        ))


class BucketList(RouteV1(v1.resource.ResourceBuckets)):
    LIST_QUERY_MAP = {
        "abc": ("abc", None),
        "limit": ("limit", None),
        "offset": ("offset", None),
        "order": ("order_by", None),
    }

    @classmethod
    def get(cls, query):
        query, offset, limit = cls.remap_query(query)
        order = query.pop("order_by", None)
        abc = query.get("abc")
        if abc:
            query["abc"] = abc
        query = mapping.Bucket.objects(**query)
        total = query.count()
        if order:
            query = query.order_by(*order)
        docs = list((query if not offset else query.skip(offset)).limit(limit))
        return v1.schemas.resource.BucketList.create(
            limit=limit,
            offset=offset,
            total=total,
            items=list(map(mappers.resource.BucketMapper.dump, docs)),
        )

    @staticmethod
    def _get_folder(service_id):
        config = common_config.Registry().common.abcd
        d_tvm_service_id = config.d_tvm_service_id
        tvm_ticket = common_tvm.TVM().get_service_ticket([d_tvm_service_id])[d_tvm_service_id]
        tvm_auth = common_auth.TVMSession(tvm_ticket)
        d_api = common_rest.Client(config.d_api_url, auth=tvm_auth)
        return next(iter(filter(
            lambda _: _["displayName"] == "default",
            d_api.services[service_id].folders[:]["items"]
        )), None)

    @classmethod
    def post(cls, body):
        prefix, abc_id, suffix = (body.name.split("-", 2) + ["", None])[:3]
        if prefix != "sandbox" or not abc_id.isdigit():
            raise exceptions.BadRequest("Invalid bucket name: '{}'".format(body.name))
        abc_slug = common_abc.abc_service_name(abc_id)
        if common_mds.S3().bucket_stats(body.name) is None:
            raise exceptions.BadRequest("S3 bucket '{}' does not exist".format(body.name))
        folder = cls._get_folder(abc_id)
        if folder is None:
            raise exceptions.BadRequest("default folder for service {} ({}) not found".format(abc_slug, abc_id))
        bucket = mapping.Bucket(
            name=body.name,
            abcd_account=mapping.ABCDAccount(
                id=str(uuid.uuid4()),
                folder_id=folder["id"],
                last_update=mapping.ABCDAccount.LastUpdate(
                    author=context.current.user.login,
                ),
            ),
            abc=abc_slug,
            lru_reserve=body.lru_reserve,
        )
        try:
            bucket.save(force_insert=True)
        except mapping.NotUniqueError:
            raise exceptions.Conflict("Bucket with name '{}' already exists".format(body.name))
        bucket.reload()
        return flask.current_app.response_class(
            response=json.dumps(mappers.resource.BucketMapper.dump(bucket)),
            status=httplib.CREATED,
        )


class Bucket(RouteV1(v1.resource.ResourceBucket)):
    @classmethod
    def get(cls, name):
        bucket = mapping.Bucket.objects.with_id(name)
        if not bucket:
            raise exceptions.NotFound("Bucket with name {} is not found".format(name))
        return mappers.resource.BucketMapper.dump(bucket)

    @classmethod
    def put(cls, name, body):
        bucket = mapping.Bucket.objects.with_id(name)
        if not bucket:
            raise exceptions.NotFound("Bucket with name {} is not found".format(name))

        if not cls._is_allowed_to_change(context.current.user, bucket.abc):
            raise exceptions.Forbidden(
                "You must be hardware_resources_manager or quotas_manager "
                "to edit notification settings for the bucket"
            )

        for field in ("ignore_bucket_exhaust", "lru_threshold"):
            value = getattr(body, field, ctm.NotExists)
            if value is not ctm.NotExists:
                setattr(bucket, field, value)

        bucket.save()

        return "", httplib.NO_CONTENT

    @classmethod
    def _is_allowed_to_change(cls, user, abc_service):
        settings = common_config.Registry()
        if not settings.server.auth.enabled:
            return True
        if user and isinstance(user, six.string_types):
            if user == controller.User.anonymous.login:
                return False
            user = mapping.User.objects.with_id(user)
        if not user:
            return False
        if user.super_user:
            return True
        return user.login in common_abc.bucket_notification_recipient(abc_service)
