import httplib

from sandbox.web.api import v1

from sandbox.yasandbox.database import mapping

from sandbox.yasandbox import controller, context

from sandbox.serviceapi import mappers
from sandbox.serviceapi.web import RouteV1, exceptions


class Scheduler(RouteV1(v1.scheduler.Scheduler)):
    @classmethod
    def delete(cls, id_):
        scheduler = controller.Scheduler.load(id_)
        if not scheduler:
            raise exceptions.NotFound(
                "Scheduler #{} not found.".format(id_)
            )

        if not controller.user_has_permission(context.current.user, (scheduler.owner,)):
            raise exceptions.Forbidden(
                "User {} is not permitted to delete the scheduler.".format(context.current.user.login)
            )

        scheduler.delete()
        return "", httplib.NO_CONTENT


class SchedulerList(RouteV1(v1.scheduler.SchedulerList)):
    LIST_QUERY_MAP = {
        "id": ("id__in", "id"),
        "task_type": ("type__in", None),
        "status": ("status__in", None),
        "owner": ("owner", None),
        "author": ("author", None),
        "limit": ("limit", None),
        "offset": ("offset", None),
        "order": ("order_by", None),
        "tags": ("tags", "tags"),
        "all_tags": ("all_tags", None),
    }

    @classmethod
    def get(cls, query):
        query, offset, limit = cls.remap_query(query)
        order_by = query.pop("order_by")  # Empty list if not specified
        if not order_by:
            order_by = ["-id"]

        if "status__in" not in query:
            query["status__ne"] = "DELETED"

        tags = query.pop("tags", None)
        all_tags = query.pop("all_tags", False)
        if tags:
            tag_op = "__all" if all_tags else "__in"
            query["tags" + tag_op] = list(map(lambda x: str(x).upper(), tags))

        db_query = mapping.Scheduler.objects(**query)
        total = db_query.count()
        if order_by:
            db_query = db_query.order_by(*order_by)
        if offset:
            db_query = db_query.skip(offset)

        db_query = db_query.limit(limit)

        return v1.schemas.scheduler.SchedulerList.create(
            limit=limit,
            offset=offset,
            total=total,
            items=mappers.scheduler.SchedulerListMapper.dump(db_query)
        )
