import re
import httplib
import datetime as dt

import flask
import sandbox.common.types.misc as ctm

from sandbox.serviceapi import mappers
from sandbox.serviceapi.web import RouteV1, exceptions

from sandbox.yasandbox import context, controller
from sandbox.yasandbox.database import mapping

from sandbox.web.api import v1


class SemaphoreBase(object):
    @staticmethod
    def _check_owner_setup(owner, user, user_groups):
        if not controller.Group.exists(owner):
            raise exceptions.BadRequest("Owner must be an existing group")
        if not controller.Semaphore.can_modify(user, owner, user_groups):
            raise exceptions.Forbidden("User '{}' has no permissions to set '{}' as owner".format(user.login, owner))

    @staticmethod
    def _check_shared(shared):
        if shared and shared is not ctm.NotExists:
            try:
                controller.user.validate_credentials(shared, groups=True)
            except Exception as ex:
                raise exceptions.BadRequest("Validation error: {}".format(ex))

    @staticmethod
    def _check_user_permissions(doc, user, user_groups):
        if not controller.Semaphore.can_modify(user, doc.owner, user_groups):
            raise exceptions.Forbidden("User '{}' not permitted to modify semaphore".format(user.login))


class SemaphoreList(SemaphoreBase, RouteV1(v1.semaphore.SemaphoreList)):
    LIST_QUERY_MAP = {
        "name": ("name", "name"),
        "group": ("group", "group"),
        "owner": ("owner", "owner"),
        "shared": ("shared", None),
        "auto": ("auto", None),
        "limit": ("limit", None),
        "offset": ("offset", None),
        "order": ("order_by", None),
    }

    @classmethod
    def get(cls, query):
        query, offset, limit = cls.remap_query(query)
        order = query.pop("order_by", None)
        user_groups = controller.Group.get_user_groups(context.current.user)
        name = query.get("name")
        if name:
            try:
                query["name"] = re.compile(name, re.IGNORECASE)
            except re.error as error:
                raise exceptions.BadRequest("Incorrect regex: {} \n {}".format(name, error))

        group = query.pop("group", None)
        sem_ids, groups = None, []
        if group:
            sem_ids, groups = controller.TaskQueue.qclient.semaphore_group(group)
            if not sem_ids and not groups:
                raise exceptions.BadRequest("Semaphore group '{}' does not exist".format(group))
            if sem_ids:
                query["id__in"] = sem_ids

        if sem_ids is None or sem_ids:
            query = mapping.Semaphore.objects(**query)
            total = query.count()
            if order:
                query = query.order_by(*order)
            docs = list((query if not offset else query.skip(offset)).limit(limit))
            values = controller.TaskQueue.qclient.semaphore_values(map(lambda _: _.id, docs))
        else:
            docs, values, total = [], [], 0

        return v1.schemas.semaphore.SemaphoreList.create(
            limit=limit,
            offset=offset,
            total=total,
            items=[
                mappers.semaphore.SemaphoreListItemMapper.list_dump(
                    doc, value, user_groups
                ) for doc, value in zip(docs, values)
            ],
            groups=groups
        )

    @classmethod
    def post(cls, body):
        name, owner = body.name, body.owner
        user_groups = controller.Group.get_user_groups(context.current.user)

        cls._check_owner_setup(owner, context.current.user, user_groups)
        cls._check_shared(body.shared)

        fields = {"name": name, "owner": owner}
        if body.capacity is not None:
            fields["capacity"] = body.capacity
        if body.shared is not None:
            fields["shared"] = body.shared
        if body.public is not None:
            fields["public"] = body.public
        try:
            doc = controller.Semaphore.create(fields)
        except controller.Semaphore.Exception as ex:
            raise exceptions.BadRequest(str(ex))
        return mappers.semaphore.SemaphoreListItemMapper.dump(doc, 0, user_groups)


class Semaphore(SemaphoreBase, RouteV1(v1.semaphore.Semaphore)):
    @classmethod
    def get(cls, id_):
        doc = controller.Semaphore.get(id_)
        if not doc:
            raise exceptions.NotFound("Semaphore #{} not found".format(id_))
        user_groups = controller.Group.get_user_groups(context.current.user)
        value = controller.TaskQueue.qclient.semaphore_values([doc.id])[0]
        return mappers.semaphore.SemaphoreMapper.dump(doc, value, user_groups)

    @classmethod
    def put(cls, id_, body):
        doc = controller.Semaphore.get(id_)
        if not doc:
            raise exceptions.NotFound("Semaphore #{} not found".format(id_))
        user_groups = controller.Group.get_user_groups(context.current.user)
        cls._check_user_permissions(doc, context.current.user, user_groups)
        if body.owner is not ctm.NotExists:
            cls._check_owner_setup(body.owner, context.current.user, user_groups)
        cls._check_shared(body.shared)
        event = body.event

        fields = {}
        for field in ("owner", "shared", "capacity", "auto", "public"):
            value = getattr(body, field)
            if value is not ctm.NotExists:
                fields[field] = value

        try:
            doc = controller.Semaphore.update(doc.id, fields, event=event)
        except controller.Semaphore.NotExists as ex:
            raise exceptions.NotFound(str(ex))
        except (controller.Semaphore.Exception, TypeError, ValueError) as ex:
            raise exceptions.BadRequest(str(ex))
        value = controller.TaskQueue.qclient.semaphore_values([id_])[0]
        if ctm.HTTPHeader.WANT_UPDATED_DATA in flask.request.headers:
            return mappers.semaphore.SemaphoreMapper.dump(doc, value, user_groups)
        else:
            return "", httplib.NO_CONTENT

    @classmethod
    def delete(cls, id_):
        doc = controller.Semaphore.get(id_)
        if not doc:
            raise exceptions.NotFound("Semaphore #{} not found".format(id_))
        user_groups = controller.Group.get_user_groups(context.current.user)
        cls._check_user_permissions(doc, context.current.user, user_groups)

        if not controller.TaskQueue.qclient.delete_semaphore(doc.id):
            raise exceptions.Locked("Semaphore #{} is busy".format(id_))

        controller.Semaphore.audit(id_, "Deleted")
        return "", httplib.NO_CONTENT


class SemaphoreAudit(RouteV1(v1.semaphore.SemaphoreAudit)):
    @classmethod
    def get(cls, id_):
        ret = [
            v1.schemas.semaphore.SemaphoreAuditItem.create(
                description=doc.description,
                time=doc.time or dt.datetime.utcnow(),
                target=doc.target,
                author=doc.author,
                source=doc.source
            )
            for doc in mapping.SemaphoreAudit.objects(semaphore_id=id_).order_by("+time")
        ]
        return ret
