import json
import httplib
import datetime
import collections

import flask
import aniso8601

from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm

from sandbox.web.api import v1
from sandbox.yasandbox.database import mapping
from sandbox.yasandbox import context
from sandbox.yasandbox import controller
import sandbox.serviceq.types as qtypes

from sandbox.serviceapi.web import RouteV1
from sandbox.serviceapi.web import exceptions


class ServiceStatisticsTaskStatus(RouteV1(v1.service.ServiceStatisticsTaskStatus)):
    RENAMED_FIELDS = {"break": "break_"}

    @classmethod
    def get(cls, stat_type):
        stats = controller.Statistics.get_statistics(controller.Statistics.Keys.STATUS).get(stat_type)
        if not stats:
            raise exceptions.NotFound("Statistics type not found.")

        stats = {
            str(group).lower(): {k: stats[k] for k in map(lambda _: _.lower(), group)}
            for group in ctt.Status.Group
            if group.primary
        }

        for field, rfield in cls.RENAMED_FIELDS.iteritems():
            stats[rfield] = stats.pop(field)

        return v1.schemas.service.TaskStatusStatistics.create(**stats)


class ServiceStatisticsTaskTypesNotUsed(RouteV1(v1.service.ServiceStatisticsTaskTypesNotUsed)):
    @classmethod
    def get(cls, query):
        """ List of task types that do not run more than N days """
        days = query.get('days_ago')

        today = datetime.datetime.utcnow().replace(hour=0, minute=0, second=0, microsecond=0)
        threshold_date = today - datetime.timedelta(days=days)

        pipeline = [
            {"$group": {"_id": "$type", "last_run": {"$max": "$exc.time.st"}}},
            {"$match": {"last_run": {"$lte": threshold_date}}},
            {"$project": {"_id": False, "type": "$_id", "last_run": True}}
        ]
        res = mapping.Task.aggregate(pipeline)
        return [v1.schemas.service.TaskTypesNotUsedStatistics.create(**r) for r in res]


class ServiceStatusOperationMode(RouteV1(v1.service.ServiceStatusOperationMode)):
    @classmethod
    def get(cls):
        return controller.Settings().mode().lower()

    @classmethod
    def put(cls, mode):
        controller.Settings.set_mode((
            getattr(controller.Settings.OperationMode, mode.upper()),
            context.current.user.login
        ))

        return "", httplib.NO_CONTENT


class ServiceStatusQInstances(RouteV1(v1.service.ServiceStatusQInstances)):
    @classmethod
    def get(cls):
        qclient = controller.TaskQueue.qclient
        result = []
        for item in qclient.replication_info().keys():
            address, port = item.split(":")
            result.append(
                v1.schemas.service.QInstance.create(
                    address=address,
                    port=port,
                    status=qtypes.Status.SECONDARY
                )
            )

        primary_address, primary_port = qclient.primary_address.split(":")
        result.append(
            v1.schemas.service.QInstance.create(
                address=primary_address,
                port=primary_port,
                status=qtypes.Status.PRIMARY
            )
        )
        return result


ShardCPUUsage = collections.namedtuple(
    "ShardCPUUsage",
    ("shard_cpu_usage", "shard_cpu_updated", "shard_reloaded_ts")
)


class ServiceStatusDatabaseShards(RouteV1(v1.service.ServiceStatusDatabaseShards)):
    @classmethod
    def get(cls):
        data = controller.Statistics.db_shards_status()
        state = list(mapping.State.objects())
        shards_cpu_usages = collections.defaultdict(lambda: ShardCPUUsage(None, None, None))
        shards_cpu_usages.update(
            {
                shard_name: ShardCPUUsage(
                    shard_cpu_usage=shard_info.shard_cpu_usage,
                    shard_cpu_updated=shard_info.shard_cpu_updated,
                    shard_reloaded_ts=shard_info.shard_reloaded_ts,
                )
                for server in state
                for shard_name, shard_info in server.shards.iteritems()
            }
        )
        return [
            v1.schemas.service.DatabaseShard.create(
                id=shard["replicaset"],
                updated=aniso8601.parse_datetime(shard["date"].replace(" ", "T")) if shard["date"] else None,
                replicaset=[
                    v1.schemas.service.DatabaseShardReplicaInstance.create(
                        uptime=r["uptime"],
                        optime=r.get("optime"),  # arbiters don't have optime
                        ping=r.get("pingMs", 0),
                        elected=r.get("electionTime"),
                        state=ctm.MongoState.DOWN if "not reachable" in r["stateStr"] else r["stateStr"],
                        last_heartbeat=(
                            aniso8601.parse_datetime(r["lastHeartbeat"].replace(" ", "T"))
                            if r.get("lastHeartbeat") else
                            None
                        ),
                        id=r["name"],
                        can_vote=r.get("votes") != 0 or r.get("priority") != 0,
                        hidden=r.get("hidden", False),
                        shard_cpu_usage=shards_cpu_usages[r["name"]].shard_cpu_usage,
                        shard_cpu_updated=shards_cpu_usages[r["name"]].shard_cpu_updated,
                        shard_reloaded_ts=shards_cpu_usages[r["name"]].shard_reloaded_ts,
                    )
                    for r in shard["members"]
                ],
            )
            for shard in data
        ]


class ServiceStatusDatabaseSize(RouteV1(v1.service.ServiceStatusDatabaseSize)):
    @classmethod
    def get(cls):
        stat = controller.Statistics.db_size()
        return v1.schemas.service.DatabaseSize.create(**stat)


class ServiceStatusThreads(RouteV1(v1.service.ServiceStatusThreads)):
    @classmethod
    def get(cls):
        locks = common.zk.Zookeeper().start().list()
        # a tiny compatibility layer for correct lock displaying for servants
        # whose ZooKeeper lock path differs from their `self.name` property (for instance, TaskStateSwitcher)
        # TODO: remove this after TODO in services.base.service.SingletonService.__init__ is removed
        locks.update({
            common.utils.ident(servant_name).lower(): locks[servant_name]
            for servant_name in locks
        })
        times = {
            _.name: {
                "last": _.time.last_run,
                "next_": _.time.next_run,
            } for _ in mapping.Service.objects
        }
        def_run_time = {"last": None, "next_": None}
        return [
            v1.schemas.service.ServiceThreadStatus.create(
                id=key,
                locks=locks.get(key, []),
                run_time=v1.schemas.service.ServiceThreadStatusTime.create(**times.get(key, def_run_time))
            )
            for key in times.iterkeys()
        ]


class ServiceQ(RouteV1(v1.service.ServiceQ)):
    LIST_QUERY_MAP = {
        "client": ("client", "client"),
    }

    @classmethod
    def get(cls, query):
        # Parse query arguments and form them as keyword arguments to database query builder
        query, offset, limit = cls.remap_query(query)
        client = query.get("client")

        items, count = [], -1
        p = ctt.Priority()
        base_url = context.current.request.host_url
        for tid, pr in mapping.Task.objects(
            mapping.Q(execution__status__in=list(ctt.Status.Group.NONWAITABLE)) &
            (mapping.Q(lock_host=client) | mapping.Q(execution__host=client))
        ).order_by("+id").fast_scalar("id", "priority"):
            count += 1
            if count < offset or count >= offset + limit:
                continue

            class_, subclass = p.__setstate__(pr).__getstate__()

            items.append(v1.schemas.service.QueuedTask.create(
                id=tid,
                url="{}/task/{}".format(base_url, tid),
                priority=v1.schemas.task.TaskPriority.create(class_=class_, subclass=subclass),
                clients=[[None, client]],
            ))

        data = controller.TaskQueue.qclient.queue_by_host(client, secondary=True)
        for task_id, priority, score in data:
            count += 1
            if count < offset or count >= offset + limit:
                continue

            class_, subclass = p.__setstate__(priority).__getstate__()

            items.append(v1.schemas.service.QueuedTask.create(
                id=task_id,
                url="{}/task/{}".format(base_url, task_id),
                priority=v1.schemas.task.TaskPriority.create(class_=class_, subclass=subclass),
                clients=[[score, client]],
            ))

        return v1.schemas.service.QueuedTasksList.create(
            items=items,
            limit=limit,
            offset=offset,
            total=count + 1,
        )


class ServiceUINotificationList(RouteV1(v1.service.ServiceUINotificationList)):
    @classmethod
    def get(cls):
        return [
            v1.schemas.service.UINotification.create(
                id=str(doc.id),
                severity=doc.severity,
                content=doc.content,
            )
            for doc in mapping.UINotification.objects()
        ]


class ServiceResources(RouteV1(v1.service.ServiceResources)):
    @classmethod
    def get(cls, query):
        resource_type = query["type"]
        platform = query["platform"]
        version = query["version"]
        platform = common.platform.get_platform_alias(platform)
        context = mapping.Service.objects.fast_scalar("context").with_id("UpdateSandboxResources")
        if not context:
            return flask.current_app.response_class(
                status=httplib.OK,
                content_type="application/json; charset=utf-8",
            )
        cache = context.get("cache")
        if not cache:
            return flask.current_app.response_class(
                status=httplib.OK,
                content_type="application/json; charset=utf-8",
            )
        return flask.current_app.response_class(
            response=json.dumps(json.loads(cache).get(resource_type, {}).get(platform, {}).get(version)),
            status=httplib.OK,
            content_type="application/json; charset=utf-8",
        )


class ServiceTimeCurrent(RouteV1(v1.service.ServiceTimeCurrent)):
    @classmethod
    def get(cls):
        return common.api.DateTime.encode(datetime.datetime.utcnow())


class ServiceUnavailable(RouteV1(v1.service.ServiceUnavailable)):
    @classmethod
    def get(cls):
        raise exceptions.ServiceUnavailable("They asked me to do it!")


class ServiceTVM(RouteV1(v1.service.ServiceTVM)):
    @classmethod
    def get(cls):
        headers = context.current.request.get_tvm_headers("yav", need_user_ticket=True)
        return flask.current_app.response_class(
            response=json.dumps(list(headers)),
            status=httplib.OK,
            content_type="application/json; charset=utf-8",
        )


class ServiceProxyBannedNetworks(RouteV1(v1.service.ServiceProxyBannedNetworks)):
    @classmethod
    def get(cls):
        network_macros_resolver = mapping.Service.objects.with_id("network_macros_resolver")
        if network_macros_resolver is None or "banned_networks" not in network_macros_resolver.context:
            return {}
        return network_macros_resolver.context["banned_networks"]


class ServiceProxyBannedResources(RouteV1(v1.service.ServiceProxyBannedResources)):
    @classmethod
    def get(cls):
        network_macros_resolver = mapping.Service.objects.with_id("network_macros_resolver")
        if network_macros_resolver is None or "banned_resources" not in network_macros_resolver.context:
            return {}
        return network_macros_resolver.context["banned_resources"]


class ServiceProxyWhitelistResources(RouteV1(v1.service.ServiceProxyWhitelistResources)):
    @classmethod
    def get(cls):
        network_macros_resolver = mapping.Service.objects.with_id("network_macros_resolver")
        if network_macros_resolver is None or "whitelist_resources" not in network_macros_resolver.context:
            return {}
        return network_macros_resolver.context["whitelist_resources"]


class ServiceTasksEnqueuedCpuPreferences(RouteV1(v1.service.ServiceTasksEnqueuedCpuPreferences)):
    @classmethod
    def get(cls):
        return controller.Statistics.get_statistics(key=controller.Statistics.Keys.ENQUEUED_TASKS)


class StorageSizeStatistics(RouteV1(v1.service.ServiceStatisticsStorageSize)):
    @classmethod
    def get(cls):
        return v1.schemas.service.StorageSizeStatistics.create(
            **controller.Statistics.get_statistics(key=controller.Statistics.Keys.STORAGE)
        )


class ServiceTasksEnqueuedQueueTime(RouteV1(v1.service.ServiceTasksEnqueuedQueueTime)):
    @classmethod
    def get(cls):
        data = controller.Statistics.get_statistics(key=controller.Statistics.Keys.ENQUEUE_TIME)
        total_time = sum(map(lambda x: (x.values() or [0])[0], data.get("last_hour_timings", [])))
        return v1.schemas.service.TasksEnqueuedQueueTime.create(enqueue_time_in_last_hour=total_time / 3600)


class ServiceStatusDatabaseCurrentOp(RouteV1(v1.service.ServiceStatusDatabaseCurrentOp)):
    @classmethod
    def get(cls):
        return v1.schemas.service.DatabaseCurrentOp.create(
            **controller.Statistics.current_db_operations_statistics()
        )


class ServiceTelegraBotOwnerLastVisit(RouteV1(v1.service.ServiceTelegraBotOwnerLastVisit)):
    @classmethod
    def get(cls):
        return common.api.DateTime.encode(
            common.format.str2dt(mapping.Service.objects.with_id("telegram_bot").context["bot_owner_last_visit"])
        )


class ServiceManageApiQuota(RouteV1(v1.service.ServiceManageApiQuota)):
    @classmethod
    def get(cls, user_name):
        qc = controller.TaskQueue.qclient
        quota = qc.get_api_quota(user_name)
        return v1.schemas.service.QuotaApi.create(
            quota_value=quota,
        )

    @classmethod
    def put(cls, user_name, body):
        if not context.current.user.super_user:
            raise exceptions.Forbidden("Only administrator can manage api quota.")
        if body.quota_value < 0:
            raise exceptions.BadRequest("Quota must be non-negative.")
        qc = controller.TaskQueue.qclient
        qc.set_api_quota(user_name, body.quota_value)
        return "", httplib.NO_CONTENT
