import json
import httplib

import flask

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.user as ctu
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
import sandbox.common.types.scheduler as cts

from sandbox.serviceapi.mules import suggest_cache

from sandbox.taskbox.model import bridge as model_bridge

from sandbox.web.api import v1
from sandbox.common import config
from sandbox.common import platform
from sandbox.yasandbox import context
from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping

from sandbox.serviceapi.web import RouteV1
from sandbox.serviceapi.web import exceptions
from sandbox.serviceapi import constants as sa_consts


try:
    import uwsgi
except ImportError:
    uwsgi = None


registry = config.Registry()


class CachedSuggestBase(object):
    @classmethod
    def get_cached_suggest_data(cls, type_):
        value = uwsgi.cache_get(type_, sa_consts.Suggest.UWSGI_CACHE_NAME)
        if value is not None:
            return value

        try:
            # Get full suggest info from legacy server
            r = suggest_cache.Suggest.suggest(type_)
        except Exception:
            context.current.logger.exception("Failed to get suggest data from cache")
            raise exceptions.ServiceUnavailable("Temporary error, please, try again")
        return r

    @classmethod
    def get_suggest_response(cls, suggest_type, object_type):
        data = cls.get_cached_suggest_data(suggest_type)

        if object_type:
            object_type = object_type.strip().upper()

            for item in json.loads(data):
                if item["type"] == object_type:
                    data = json.dumps([item])
                    break
            else:
                raise exceptions.BadRequest("No task type '{}' found.".format(object_type))

        ret = flask.current_app.response_class(
            response=data,
            status=httplib.OK,
            content_type="application/json; charset=utf-8",
        )

        return ret


class SuggestTask(CachedSuggestBase, RouteV1(v1.suggest.SuggestTask)):
    @classmethod
    def get(cls, query):
        resource_id = query.get("resource_id")
        if resource_id is not None:
            resource = controller.Template.tasks_resource_meta(tasks_resource_id=resource_id)
            if resource is None:
                raise exceptions.NotFound("Resource not found")
            if controller.Template.is_taskboxed_task(resource):
                if resource.age >= 7:
                    bridge = model_bridge.RemoteTaskBridge(
                        controller.Template.empty_task(resource), context.current.request.id
                    )
                else:
                    raise exceptions.BadRequest(
                        "Age of resource with binary tasks {} < 7.".format(resource.age)
                    )
                return bridge.tasks_suggest(query["type"])
        return cls.get_suggest_response("task", query["type"])


class SuggestResource(CachedSuggestBase, RouteV1(v1.suggest.SuggestResource)):
    @classmethod
    def get(cls, query):
        return cls.get_suggest_response("resource", query["type"])


class SuggestEnumBase(object):
    @classmethod
    def get_suggest_for_enum(cls, enum):
        return v1.schemas.suggest.SuggestEnum.create(
            items=[
                v1.schemas.suggest.SuggestEnumItem.create(
                    name=str(item),
                    description="" if isinstance(item, basestring) else (item.__doc__ or "")
                )
                for item in enum
            ],
            groups=[
                v1.schemas.suggest.SuggestEnumGroup.create(
                    name=str(group),
                    description=group.__doc__ or "",
                    primary=group.primary,
                    items=map(str, group),
                )
                for group in getattr(enum, "Group", [])
            ]
        )


class SuggestReleaseStatus(SuggestEnumBase, RouteV1(v1.suggest.SuggestReleaseStatus)):
    @classmethod
    def get(cls):
        return cls.get_suggest_for_enum(ctt.ReleaseStatus)


class SuggestSchedulerStatus(SuggestEnumBase, RouteV1(v1.suggest.SuggestSchedulerStatus)):
    @classmethod
    def get(cls):
        return cls.get_suggest_for_enum(cts.Status)


class SuggestClientTag(SuggestEnumBase, RouteV1(v1.suggest.SuggestClientTag)):
    @classmethod
    def get(cls):
        return cls.get_suggest_for_enum(ctc.Tag)


class SuggestTaskReqDNS(SuggestEnumBase, RouteV1(v1.suggest.SuggestTaskReqDNS)):
    @classmethod
    def get(cls):
        return cls.get_suggest_for_enum(ctm.DnsType)


class SuggestTaskStatus(SuggestEnumBase, RouteV1(v1.suggest.SuggestTaskStatus)):
    @classmethod
    def get(cls):
        return cls.get_suggest_for_enum(ctt.Status)


class SuggestResourceStatus(SuggestEnumBase, RouteV1(v1.suggest.SuggestResourceStatus)):
    @classmethod
    def get(cls):
        return cls.get_suggest_for_enum(ctr.State)


class SuggestGroup(RouteV1(v1.suggest.SuggestGroup)):
    @classmethod
    def get_url_for(cls, group):
        return "{}/group/{}".format(context.current.request.base_url.rsplit("/", 3)[0], group.name)

    @classmethod
    def get_priority_for(cls, group, attr):
        pl = group.priority_limits or controller.Group.regular.priority_limits

        if getattr(pl, attr) is None:
            priority = getattr(ctu.DEFAULT_PRIORITY_LIMITS, attr)
        else:
            priority = ctt.Priority().__setstate__(getattr(pl, attr))

        class_, subclass = priority.__getstate__()

        return v1.schemas.task.TaskPriority.create(
            class_=class_,
            subclass=subclass
        )

    @classmethod
    def get(cls):
        ret = [
            v1.schemas.suggest.SuggestGroup.create(
                name=g.name,
                url=cls.get_url_for(g),
                priority_limits=v1.schemas.group.GroupPriorityLimits.create(
                    ui=cls.get_priority_for(g, "ui"),
                    api=cls.get_priority_for(g, "api"),
                )
            )
            for g in mapping.Group.objects.order_by("+id")
        ]
        return ret


def suggest_tags(query, limit):
    tags_query = mapping.TaskTagCache.objects(tag__startswith=query.upper())
    docs = tags_query.order_by("-hits", "-accessed").limit(limit).fast_scalar("tag", "hits")
    return [v1.schemas.suggest.SuggestTaskTag.create(tag=tag, hits=hits) for tag, hits in docs]


class SuggestTaskTags(RouteV1(v1.suggest.SuggestTaskTags)):

    @staticmethod
    def get(query):
        return suggest_tags(query["query"], query["limit"])


class SuggestTaskTagsByPrefix(RouteV1(v1.suggest.SuggestTaskTagsByPrefix)):

    @staticmethod
    def get(prefix, query):
        return suggest_tags(prefix, query["limit"])


class SuggestQuicksearch(RouteV1(v1.suggest.SuggestQuicksearch)):
    @classmethod
    def get(cls, query, q):
        query = query.strip()
        try:
            integer = int(query.replace(' ', ''))
        except (TypeError, ValueError):
            integer = None

        res = []
        # Cut-off this URL suffix to determine base URL.
        base_url = context.current.request.base_url.rsplit('/', 3)[0]

        if integer:
            # In case some integer detected, try to search for task and/or resource objects.
            task = mapping.Task.objects.with_id(integer)
            if task:
                res.append({
                    "task": {
                        "url": "{}/task/{}".format(base_url, task.id),
                        "status": task.execution.status,
                        "owner": task.owner,
                        "type": task.type,
                        "id": task.id
                    }
                })
            resource = mapping.Resource.objects.with_id(integer)
            if resource:
                res.append({
                    "resource": {
                        "url": "{}/resource/{}".format(base_url, resource.id),
                        "status": resource.state,
                        "owner": resource.owner,
                        "type": resource.type,
                        "id": resource.id
                    }
                })

        return res


class SuggestTemplate(CachedSuggestBase, RouteV1(v1.suggest.SuggestTemplate)):
    @classmethod
    def get(cls):
        return cls.get_suggest_response("template", None)


class SuggestPlatforms(SuggestEnumBase, RouteV1(v1.suggest.SuggestPlatforms)):
    @classmethod
    def get(cls):
        return cls.get_suggest_for_enum(sorted(platform.PLATFORM_TO_TAG))
