import httplib

import sandbox.common.types.template as ctte

from sandbox.web.api import v1
from sandbox.yasandbox import context
from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping

from sandbox.yasandbox.api.json import mappers as api_mappers

from sandbox.serviceapi.web import RouteV1, exceptions


class TemplateList(RouteV1(v1.template.TemplateList)):
    LIST_QUERY_MAP = {
        "alias": ("alias", "alias"),
        "author": ("author", "author"),
        "shared_with": ("shared_with", None),
        "favorites": ("favorites", None),
        "task_type": ("task__type__in", "task__type__in"),
        "status": ("status__in", None),
        "limit": ("limit", None),
        "offset": ("offset", None),
        "order": ("order_by", None),
    }

    @classmethod
    def get(cls, query):
        fields = query.pop("fields", None)
        query, offset, limit = cls.remap_query(query)

        if ctte.Status.DELETED not in query.get("status__in", []):
            query["status__nin"] = [ctte.Status.DELETED]

        order_by = query.pop("order_by")  # Empty list if not specified
        if not order_by and "alias" not in query:
            order_by = ["alias"]

        mongo_query = mapping.TaskTemplate.objects(**query)

        if order_by:
            mongo_query = mongo_query.order_by(*order_by)
        if offset:
            mongo_query = mongo_query.skip(offset)

        # `limit` is required and thus always present
        mongo_query = mongo_query.limit(limit)
        mapper = api_mappers.template.TemplateMapper(fields, context.current.user)
        return v1.schemas.template.TemplateList.create(
            offset=offset,
            limit=limit,
            total=mongo_query.count(),
            items=[mapper.dump(template, legacy=False) for template in mongo_query]
        )


class Template(RouteV1(v1.template.Template)):
    @classmethod
    def get(cls, alias, query):
        fields = query.get("fields")
        template = mapping.TaskTemplate.objects.with_id(alias)
        if template is None:
            raise exceptions.NotFound("Template {} not found.".format(alias))
        return v1.schemas.template.Template.create(
            **api_mappers.template.TemplateMapper(fields, context.current.user).dump(template, legacy=False)
        )

    @classmethod
    def delete(cls, alias):
        template = mapping.TaskTemplate.objects.with_id(alias)
        if template is None:
            raise exceptions.NotFound("Template {} not found.".format(alias))
        user = context.current.user
        if not controller.user_has_permission(user, template.shared_with):
            raise exceptions.Forbidden(
                "User {} has no permissions to remove template {}".format(user.login, template.alias)
            )
        if template.status == ctte.Status.DELETED:
            raise exceptions.BadRequest("Template {} already deleted".format(alias))

        audit = mapping.TemplateAudit(
            template_alias=alias, author=user.login,
            properties={"status": mapping.TemplateAudit.ChangedProperty(old=template.status, new=ctte.Status.DELETED)}
        )

        template.status = ctte.Status.DELETED
        template.save()
        audit.save()
        return "", httplib.NO_CONTENT


class TemplateAuditList(RouteV1(v1.template.TemplateAuditList)):
    LIST_QUERY_MAP = {
        "template_alias": ("template_alias", "template_alias"),
        "author": ("author", "author"),
        "date": ("date", "date__at"),
        "limit": ("limit", None),
        "offset": ("offset", None),
        "order": ("order_by", None),
    }

    @classmethod
    def get(cls, query):
        query, offset, limit = cls.remap_query(query)
        order_by = query.pop("order_by", None)
        date = query.get("date")

        if not any((query.get(field) for field in ("date", "template_alias", "author"))):
            raise exceptions.BadRequest("Required one of query fields: date, template_alias ot author")

        if date:
            query["date__gt"] = date[0]
            query["date__lt"] = date[1]

        db_query = mapping.TemplateAudit.objects(**query)
        total = db_query.count()
        if order_by:
            db_query = db_query.order_by(*order_by)
        if offset:
            db_query = db_query.skip(offset)
        db_query = db_query.limit(limit)
        mapper = api_mappers.template.TemplateAuditMapper(legacy=False)
        return v1.schemas.template.TemplateAuditList.create(
            total=total,
            limit=limit,
            offset=offset,
            items=[mapper.dump(doc) for doc in db_query]
        )


class TemplateFavorites(RouteV1(v1.template.TemplateFavorites)):
    @classmethod
    def _template(cls, alias):
        template = mapping.TaskTemplate.objects.with_id(alias)
        if template is None:
            raise exceptions.NotFound("Template {} not found".format(alias))
        return template

    @classmethod
    def post(cls, alias):
        user = context.current.user.login
        template = cls._template(alias)
        if user in template.favorites:
            raise exceptions.BadRequest("Template {} already in favorites".format(alias))
        mapping.TaskTemplate.objects(alias=alias).update(add_to_set__favorites=user)
        return "", httplib.NO_CONTENT

    @classmethod
    def delete(cls, alias):
        user = context.current.user.login
        template = cls._template(alias)
        if user not in template.favorites:
            raise exceptions.BadRequest("Template {} already not in favorites".format(alias))
        mapping.TaskTemplate.objects(alias=alias).update(pull__favorites=user)
        return "", httplib.NO_CONTENT
