import httplib

import flask

import sandbox.common.types.user as ctu

from sandbox.web.api import v1
from sandbox.yasandbox import context
from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping

from sandbox.serviceapi.web import RouteV1
from sandbox.serviceapi.web import exceptions

import sandbox.serviceapi.handlers.common


class UserBase(object):
    @staticmethod
    def get(login=None):
        model = controller.User.get(login) if login else context.current.user
        if not model:
            raise exceptions.NotFound("User with login '{}' not found.".format(login))
        if model.super_user:
            role = ctu.Role.ADMINISTRATOR
        elif model == controller.User.anonymous:
            role = ctu.Role.ANONYMOUS
        else:
            role = ctu.Role.REGULAR

        ret = v1.schemas.user.User.create(
            login=model.login,
            role=role,
            quota=sandbox.serviceapi.handlers.common.quota_info(model.login)
        )

        return ret


class User(UserBase, RouteV1(v1.user.User)):
    pass


class UserCurrent(UserBase, RouteV1(v1.user.UserCurrent)):
    pass


class UserGroupBase(object):
    @staticmethod
    def get_url_for(group):
        return "{}/group/{}".format(flask.request.url.rsplit("/", 3)[0], group.name)

    @staticmethod
    def get_priority_for(group, attr):
        priority = controller.Group.allowed_priority(None, group.name, context.current.user, attr, False)
        class_, subclass = priority.__getstate__()

        return v1.schemas.task.TaskPriority.create(
            class_=class_,
            subclass=subclass
        )

    @classmethod
    def get_quotas(cls, groups):
        try:
            quota_consumption = controller.TaskQueue.multiple_owners_quota(owners=[g.name for g in groups])
        except:
            return dict()
        return quota_consumption

    @classmethod
    def _to_quota_info_short(cls, quotas, group):
        if isinstance(quotas, v1.schemas.user.QuotaInfoShortInPool):
            return quotas
        default_pool_quota = quotas.get(None, {}).get(group)
        if default_pool_quota:
            quota_info = v1.schemas.user.QuotaInfoShort.create(
                consumption=v1.schemas.user.QuotaConsumption.create(
                    real=default_pool_quota.real_consumption,
                    future=default_pool_quota.future_consumption,
                ),
                limit=default_pool_quota.limit,
            )
        else:
            quota_info = v1.schemas.user.QuotaInfoShort.create(
                consumption=v1.schemas.user.QuotaConsumption.create(
                    real=0,
                    future=0,
                ),
                limit=0,
            )
        pools = []
        for pool, quotas in quotas.items():
            if pool is None:
                continue
            pool_quota = quotas.get(group)
            if not pool_quota:
                continue
            pools.append(v1.schemas.user.QuotaInfoShortInPool.create(
                pool=pool,
                consumption=v1.schemas.user.QuotaConsumption.create(
                    real=pool_quota.real_consumption,
                    future=pool_quota.future_consumption,
                ),
                limit=pool_quota.limit,
            ))
        quota_info.pools = pools
        return quota_info

    @classmethod
    def get(cls, login=None):
        login = login or context.current.user.login
        user_groups = controller.Group.get_user_groups(login, True, include_public=False)
        quotas = cls.get_quotas(user_groups)

        ret = [
            v1.schemas.suggest.SuggestGroupWithQuota.create(
                name=g.name,
                url=cls.get_url_for(g),
                priority_limits=v1.schemas.group.GroupPriorityLimits.create(
                    ui=cls.get_priority_for(g, "ui"),
                    api=cls.get_priority_for(g, "api"),
                ),
                quota=cls._to_quota_info_short(quotas, g.name)
            )
            for g in user_groups
        ]
        return sorted(ret, key=lambda x: x.name)


class UserCurrentGroup(UserGroupBase, RouteV1(v1.user.UserCurrentGroups)):
    pass


class UserGroup(UserGroupBase, RouteV1(v1.user.UserGroups)):
    pass


class UserPreferencesBase(object):
    @classmethod
    def _model(cls, _):
        return context.current.user

    @classmethod
    def get(cls, login, subnode):
        model = cls._model(login)
        ret = model.preferences.get(subnode)
        if ret is None:
            raise exceptions.NotFound("User '{}' preferences subnode '{}' not found.".format(model.login, subnode))

        return ret

    @classmethod
    def put(cls, login, subnode, data):
        model = cls._model(login)
        try:
            model.preferences[subnode] = data
            model.save()
        except mapping.InvalidDocumentError as ex:
            raise exceptions.BadRequest("Unable to encode data: {}".format(ex))

        return flask.current_app.response_class(
            status=httplib.CREATED,
            headers={"Location": flask.request.path}
        )


class UserCurrentPreferences(UserPreferencesBase, RouteV1(v1.user.UserCurrentPreferences)):
    @classmethod
    def get(self, subnode):
        return super(UserCurrentPreferences, self).get(None, subnode)

    @classmethod
    def put(self, subnode, data):
        return super(UserCurrentPreferences, self).put(None, subnode, data)


class UserPreferences(UserPreferencesBase, RouteV1(v1.user.UserPreferences)):
    @classmethod
    def _model(cls, login):
        model = controller.User.get(login)
        if model != context.current.user and not context.current.user.super_user:
            raise exceptions.Forbidden("Only administrator can fetch other user's preferences.")
        return model


class UserCurrentRobots(RouteV1(v1.user.UserCurrentRobots)):

    @classmethod
    def get(cls):
        login = context.current.user.login
        robots = mapping.RobotOwner.objects.fast_scalar("robots").with_id(login) or []
        robots.sort()
        ret = [v1.schemas.user.Robot.create(login=robot) for robot in robots]
        return ret


class UserCurrentPreferencesGroupAbcLink(RouteV1(v1.user.UserCurrentPreferencesGroupAbcLink)):
    @classmethod
    def get(cls):
        group_names = context.current.user.preferences.get("group_abc_link", [])
        groups = mapping.Group.objects(name__in=group_names).fast_scalar("name", "abc")
        return [name for name, abc in groups if abc is None]
