import re
import httplib
import itertools as it
import collections

import flask

from sandbox import common
import sandbox.common.types.misc as ctm

from sandbox.web.api import v1
from sandbox.yasandbox.database import mapping
from sandbox.yasandbox import context, controller

from sandbox.serviceapi import mappers
from sandbox.serviceapi.web import RouteV1, exceptions


class VaultBase(object):
    @classmethod
    def _check_owner_setup(cls, owner, user, user_groups):
        if not user.super_user and owner not in user_groups:
            raise exceptions.Forbidden(
                "User {} has no permissions to set '{}' as owner for vault.".format(user.login, owner)
            )

    @classmethod
    def _check_allowed_users(cls, data):
        allowed_users = data.shared
        if allowed_users:
            try:
                controller.user.validate_credentials(allowed_users)
            except Exception as ex:
                raise exceptions.BadRequest("User validation error: {}".format(ex))

    @classmethod
    def _check_user_permissions(cls, vault, user, user_groups):
        if not (user.super_user and not user.robot) and vault.owner not in user_groups:
            raise exceptions.Forbidden("User {} not permitted to make changes in vault.".format(user.login))


class VaultList(VaultBase, RouteV1(v1.vault.VaultList)):
    # Mapping of request_param -> (query_builder_param, ordering_param)
    # If `ordering_param` is None -- ordering by this field is not supported
    LIST_QUERY_MAP = {
        "name": ("name", "name"),
        "owner": ("owner", "owner"),
        "shared": ("allowed_users", None),
        "description": ("description", None),
        "limit": ("limit", None),
        "offset": ("offset", None),
        "order": ("order_by", None),
    }

    @classmethod
    def __url(cls, vault):
        return "{}/{}".format(context.current.request.base_url, vault.id)

    @classmethod
    def get(cls, query):
        query, offset, limit = cls.remap_query(query)
        order_by = query.pop("order_by")
        mquery = mapping.Vault.objects(**query)
        total = mquery.count()
        if order_by:
            mquery = mquery.order_by(*order_by)
        mquery = mquery.limit(limit).skip(offset)

        task = None
        task_owner = None
        task_author = None
        if context.current.request.is_task_session:
            task = mapping.Task.objects.fast_scalar("owner", "author").with_id(context.current.request.session.task_id)
            if task is None:
                raise exceptions.NotFound("Not found task {}".format(context.current.request.session.task_id))
            task_owner, task_author = task

        if task is None and context.current.user.super_user:
            groups_and_login = None
        else:
            for_uog = task_owner if task else context.current.user.login
            groups_and_login = set(common.utils.chain(
                controller.Group.get_user_groups(for_uog),
                for_uog
            ))
            if task is not None:
                groups_and_login.update(common.utils.chain(
                    controller.Group.get_user_groups(task_author),
                    task_author
                ))

        result = []
        for vault in list(mquery):
            result.append(mappers.vault.VaultMapper.dump_list_item(context, vault, groups_and_login, cls.__url(vault)))

        result = filter((lambda vault: vault.rights) if context.current.request.is_task_session else None, result)

        return v1.vault.schemas.vault.VaultList.create(
            offset=offset,
            limit=limit,
            total=total,
            items=result,
        )

    @classmethod
    def post(cls, data):
        user_groups = set(it.chain(
            controller.Group.get_user_groups(context.current.user), (context.current.user.login,)
        ))

        cls._check_owner_setup(data.owner, context.current.user, user_groups)
        cls._check_allowed_users(data)

        try:
            vault = controller.Vault.create(
                mapping.Vault(
                    owner=data.owner, name=data.name, allowed_users=data.shared,
                    data=str(data.data), description=data.description
                )
            )
        except controller.Vault.AlreadyExists as ex:
            raise exceptions.BadRequest(str(ex))

        return flask.current_app.response_class(
            mappers.vault.VaultMapper.dump(context, vault, user_groups, cls.__url(vault)),
            httplib.CREATED,
            {"Location": "{}/{}".format(flask.request.base_url, vault.id)}
        )


class Vault(VaultBase, RouteV1(v1.vault.Vault)):
    @classmethod
    def get(cls, id_):
        vault = mapping.Vault.objects.with_id(id_)
        if vault is None:
            raise exceptions.NotFound("Vault with ID '{}' not found.".format(id_))

        user_groups_and_login = {context.current.user.login}
        user_groups_and_login.update(controller.Group.get_user_groups(context.current.user.login))

        cls._check_user_permissions(vault, context.current.user, user_groups_and_login)
        return mappers.vault.VaultMapper.dump(context, vault, user_groups_and_login, context.current.request.base_url)

    @classmethod
    def put(cls, id_, data):
        vault = mapping.Vault.objects.with_id(id_)
        if vault is None:
            raise exceptions.NotFound("Document {} not found".format(id_))

        user_groups = set(it.chain(
            controller.Group.get_user_groups(context.current.user.login), (context.current.user.login,)))

        cls._check_user_permissions(vault, context.current.user, user_groups),
        cls._check_owner_setup(data.owner, context.current.user, user_groups) if data.owner is not None else None,
        cls._check_allowed_users(data)

        updatable_fields = ["shared", "name", "owner", "description"]

        vault_data = data.data
        if vault_data:
            updatable_fields.append("data")
            data.data = vault_data.encode("utf-8") if isinstance(vault_data, unicode) else vault_data

        for name, alias in it.izip_longest(updatable_fields, ("allowed_users",)):
            if getattr(data, name) is not None:
                setattr(vault, (alias or name), getattr(data, name))
        controller.Vault.update(context.current.user, vault, encrypt=bool(vault_data))

        if ctm.HTTPHeader.WANT_UPDATED_DATA in flask.request.headers:
            return mappers.vault.VaultMapper.dump(context, vault, user_groups, flask.request.base_url)
        else:
            return "", httplib.NO_CONTENT

    @classmethod
    def delete(cls, id_):
        vault = mapping.Vault.objects.with_id(id_)
        if vault is None:
            raise exceptions.NotFound("Vault with ID '{}' not found.".format(id_))

        user_groups_and_login = {context.current.user.login}
        user_groups_and_login.update(controller.Group.get_user_groups(context.current.user.login))

        cls._check_user_permissions(vault, context.current.user, user_groups_and_login)
        controller.Vault.delete(context.current.user, vault)
        return "", httplib.NO_CONTENT


class VaultDataByQuery(RouteV1(v1.vault.VaultDataByQuery)):
    TaskCreatorsInfo = collections.namedtuple(
        "TaskCreatorsInfo", ("id", "owner", "author")
    )
    # Special name syntax for Yav secrets: `sec-a1b@ver-a1b[key]` or `sec-a1b[key]`
    YAV_NAME = re.compile(r"^(sec-[a-z0-9]+)(?:@(ver-[a-z0-9]+))?\[([\w\-.]+)\]$")

    @classmethod
    def try_yav_secret(cls, name):
        """
        If the vault name indicates a yav secret, fetch its value and return the encrypted response.
        Return `None` otherwise.
        """
        match = cls.YAV_NAME.match(name)
        if not match:
            return

        from sandbox.serviceapi.handlers import yav as yav_handler

        secret_uuid, version_uuid, key = match.groups()
        secret = common.yav.Secret.create(secret_uuid, version_uuid)

        data = yav_handler.Data.get_secret_values([secret])[secret]
        try:
            value = data.value[key]
            value = value.encode("utf-8")
        except KeyError:
            raise exceptions.BadRequest("Key `{}` is not in {}".format(key, secret))

        encryption_key = context.current.request.session.vault
        encrypted_value = common.crypto.AES(encryption_key).encrypt(value, use_base64=True)

        return v1.schemas.vault.VaultData.create(
            name=name,
            owner=context.current.user.login,
            shared=[],
            description="",
            id=str(common.yav.Secret.create(secret_uuid, version_uuid, key)),
            data=encrypted_value,
        )

    @classmethod
    def get(cls, query):
        name = query["name"]
        owner = query["owner"]

        response = cls.try_yav_secret(name)
        if response:
            return response

        if owner is None:
            del query["owner"]
        task = cls.TaskCreatorsInfo(*mapping.Task.objects.fast_scalar(
            "id", "owner", "author"
        ).with_id(context.current.request.session.task_id))
        groups_and_logins = set(common.utils.chain(
            controller.Group.get_user_groups(task.owner),
            controller.Group.get_user_groups(task.author),
            task.owner,
            task.author
        ))

        vault_docs = list(mapping.Vault.objects(**query))

        vaults = []

        for doc in vault_docs:
            if (
                context.current.user.super_user and not context.current.user.robot or
                doc.owner in groups_and_logins or
                doc.allowed_users and any(g_or_l in doc.allowed_users for g_or_l in groups_and_logins)
            ):
                vaults.append(doc)

        if len(vaults) > 1:
            raise exceptions.NotFound(
                "There are more than one records satisfy owner '{}' and name '{}'.".format(owner, name)
            )

        if not len(vaults):
            if not vault_docs:
                raise exceptions.NotFound("No record was found for owner '{}' and name '{}'.".format(owner, name))
            else:
                raise exceptions.Forbidden(
                    "Permission denied to records for owner '{}' and name '{}'.".format(owner, name)
                )

        doc = vaults[0]
        try:
            data = controller.Vault.encrypt_data(doc, task, context.current.request.session.vault, True)
        except controller.Vault.NotAllowed as ex:
            raise exceptions.Forbidden(str(ex))
        return v1.schemas.vault.VaultData.create(
            name=doc.name,
            owner=doc.owner,
            shared=doc.allowed_users,
            description=doc.description,
            id=str(doc.id),
            data=data,
        )
