from __future__ import unicode_literals

from sandbox.common import yav as common_yav
from sandbox.common import config as common_config
from sandbox.common import patterns as common_patterns
from sandbox.common.types import misc as ctm

from sandbox.web.api import v1 as routes
from sandbox.web.api.v1.schemas import yav as schemas

from sandbox.yasandbox import context
from sandbox.yasandbox.database import mapping
from sandbox.yasandbox.controller import yav as controllers

from sandbox.serviceapi import web
from sandbox.serviceapi.web import exceptions


@common_patterns.singleton
def yav_consumer():
    """
    Return a version-based identifier for Yav API quota accounting (see SANDBOX-7024)
    """
    from library.python import svn_version
    revision = svn_version.svn_revision()
    installation = str(common_config.Registry().common.installation).lower()
    return "sandbox.{}.{}".format(installation, revision)


class Status(web.RouteV1(routes.yav.Status)):

    @classmethod
    def post(cls, data):
        """ Check if the secrets are delegated to Sandbox. """

        secrets = []
        for secret in data.secrets:
            try:
                secrets.append(common_yav.Secret.create(secret.id))
            except ValueError as exc:
                raise exceptions.BadRequest(exc.args[0])

        secret_uuids = {s.secret_uuid for s in secrets}
        delegated = set(
            mapping.YavToken.objects
            .fast_scalar("secret_uuid")
            .filter(secret_uuid__in=secret_uuids)
        )

        statuses = [
            schemas.SecretStatus.create(
                secret=secret.secret_uuid,
                delegated=(secret.secret_uuid in delegated)
            )
            for secret in secrets
        ]

        return schemas.SecretStatusList.create(items=statuses)


class Tokens(web.RouteV1(routes.yav.Tokens)):

    @classmethod
    def post(cls, data):
        """ Delegate secrets to Sandbox. """

        secrets = []
        for secret in data.secrets:
            try:
                secrets.append(common_yav.Secret.create(secret.id))
            except ValueError as exc:
                raise exceptions.BadRequest(exc.args[0])

        secret_uuids = {s.secret_uuid for s in secrets}
        delegated = {
            secret[0]: secret for secret in
            mapping.YavToken.objects
            .fast_scalar("secret_uuid", "signature", "token")
            .filter(secret_uuid__in=secret_uuids, author=context.current.user.login)
        }

        tickets = context.current.request.get_tvm_headers("yav", need_user_ticket=True)
        if ctm.HTTPHeader.USER_TICKET not in tickets:
            raise exceptions.Forbidden("Could not get TVM user-ticket")

        yav = controllers.Yav(
            service_ticket=tickets.get(ctm.HTTPHeader.SERVICE_TICKET),
            user_ticket=tickets.get(ctm.HTTPHeader.USER_TICKET),
            consumer=yav_consumer(),
        )

        revoked = []
        for secret in secrets:
            if secret.secret_uuid in delegated:
                yav_token = delegated[secret.secret_uuid]
                if yav.check_revoked(secret, yav_token):
                    revoked.append(secret.secret_uuid)
                    delegated.pop(secret.secret_uuid)

        if revoked:
            context.current.logger("Remove revoked tokens for secrets: %s", ", ".join(revoked))
            mapping.YavToken.objects(secret_uuid__in=revoked, author=context.current.user.login).delete()

        statuses = []

        for secret in secrets:
            status = schemas.DelegationResult.create(secret=secret.secret_uuid)
            statuses.append(status)

            if secret.secret_uuid in delegated:
                status.delegated, status.message = True, "ok"
                continue

            try:
                token = yav.create_token(secret)
                token.to_model(context.current.user.login).save()
                status.delegated, status.message = True, "ok"

            except controllers.Yav.Error as exc:
                status.delegated, status.message = False, exc.args[0]

            except Exception as exc:
                context.current.logger.exception("could not delegate a secret")
                status.delegated, status.message = False, str(exc)

        return schemas.DelegationResultList.create(items=statuses)


class Data(web.RouteV1(routes.yav.Data)):

    @staticmethod
    def get_secret_values(secrets):  # type: (list[common_yav.Secret]) -> dict[str, common_yav.Data]
        try:
            tokens = controllers.Token.get(secrets)
        except controllers.Token.Error as exc:
            raise exceptions.Forbidden(exc.args[0])

        tickets = context.current.request.get_tvm_headers("yav", need_user_ticket=False)
        if ctm.HTTPHeader.SERVICE_TICKET not in tickets:
            raise exceptions.ServiceUnavailable("Could not get TVM service_ticket for 'yav'")

        yav = controllers.Yav(
            service_ticket=tickets[ctm.HTTPHeader.SERVICE_TICKET],
            consumer=yav_consumer(),
        )

        uid = None  # no user validation on local Sandbox
        if common_config.Registry().common.installation in ctm.Installation.Group.NONLOCAL:
            if context.current.request.is_task_session:
                last_action_user = mapping.Task.objects(
                    id=context.current.request.session.task_id
                ).fast_scalar("last_action_user").first()
                uid = mapping.User.objects(login=last_action_user).fast_scalar("uid").first()
            elif context.current.request.is_external_session:
                uid = context.current.request.user.uid
            if not uid:  # sanity check, it is not a normal situation
                raise exceptions.Forbidden("No staff uid found for user {}".format(context.current.user.login))

        try:
            return yav.get_by_tokens(tokens, uid=uid)
        except controllers.Yav.Error as exc:
            raise exceptions.Forbidden(exc.args[0])

    @classmethod
    def post(cls, data):
        """ Get encrypted secret values by secret uuid and an optional version. Task session is required. """

        secrets = []
        for secret in data.secrets:
            try:
                secrets.append(common_yav.Secret.create(secret.id, secret.version))
            except ValueError as exc:
                raise exceptions.BadRequest(exc.args[0])

        secrets_data = cls.get_secret_values(secrets)
        key = context.current.request.session.vault
        results = []

        for secret in secrets:
            secret_data = secrets_data[secret]
            encrypted_data = secret_data.encrypt(key)

            results.append(schemas.EncryptedSecretData.create(
                id=secret.secret_uuid, version=secret_data.version_uuid, data=encrypted_data
            ))

        return schemas.EncryptedSecretDataList.create(items=results)


class SecretsData(web.RouteV1(routes.yav.SecretsData)):

    @classmethod
    def post(cls, data):
        """ Get secret values by secret uuid and an optional version. Task session is required. """

        secrets = []
        for secret in data.secrets:
            try:
                secrets.append(common_yav.Secret.create(secret.id, secret.version))
            except ValueError as exc:
                raise exceptions.BadRequest(exc.args[0])

        secrets_data = Data.get_secret_values(secrets)

        results = []
        for secret in secrets:
            secret_data = secrets_data[secret]  # type: common_yav.Data
            secret_values = []
            for secret_value in secret_data.secret_values:
                secret_values.append(schemas.SecretValue.create(
                    key=secret_value["key"],
                    value=secret_value["value"],
                    encoding=secret_value.get("encoding", ""),
                ))

            results.append(schemas.SecretData.create(
                id=secret.secret_uuid,
                version=secret_data.version_uuid,
                values=secret_values,
            ))

        return schemas.SecretDataList.create(items=results)


class Secrets(web.RouteV1(routes.yav.Secrets)):

    @classmethod
    def get(cls, query):
        """ List of secrets matching the query for the current user """

        tickets = context.current.request.get_tvm_headers("yav", need_user_ticket=True)
        if ctm.HTTPHeader.USER_TICKET not in tickets:
            raise exceptions.Forbidden("Could not get TVM user-ticket")

        yav = controllers.Yav(
            service_ticket=tickets.get(ctm.HTTPHeader.SERVICE_TICKET),
            user_ticket=tickets.get(ctm.HTTPHeader.USER_TICKET),
        )

        try:
            secrets = [
                dict(filter(lambda _: _[0] != "versions", s)) for s in yav.suggest(
                    query=query["query"], uuids=query["id"], limit=query["limit"], page=query["page"]
                )
            ]
        except controllers.Yav.Error as exc:
            raise exceptions.BadRequest(exc.args[0])

        return schemas.SecretSuggestList.create(items=secrets, page=query["page"], limit=query["limit"])


class Secret(web.RouteV1(routes.yav.Secret)):

    @classmethod
    def get(cls, id_):
        """ Detailed info for the given secret, including the list of its versions """

        tickets = context.current.request.get_tvm_headers("yav", need_user_ticket=True)
        if ctm.HTTPHeader.USER_TICKET not in tickets:
            raise exceptions.Forbidden("Could not get TVM user-ticket")

        yav = controllers.Yav(
            service_ticket=tickets.get(ctm.HTTPHeader.SERVICE_TICKET),
            user_ticket=tickets.get(ctm.HTTPHeader.USER_TICKET),
        )

        try:
            secret = yav.secrets([id_], versions=True, delegation=True)[id_]
        except controllers.Yav.Error as exc:
            raise exceptions.BadRequest(exc.args[0])

        return schemas.SecretInfo.create(dict(secret))
