# coding: utf-8

import abc
import collections
import datetime as dt
import itertools as it

import sandbox.common.types.user as ctu
import sandbox.common.types.client as ctc

from sandbox import common
from sandbox.web import api
from sandbox.common import config

from sandbox.yasandbox.database import mapping
from sandbox.yasandbox import controller, context

registry = config.Registry()


class ClientMapperBase(object):
    __metaclass__ = abc.ABCMeta

    # Base client fields
    BASE_FIELDS = (
        "id", "uuid", "url", "os", "ram", "ncpu", "cpu", "platform", "lxc", "fqdn", "fileserver", "dc", "alive",
        "availability", "disk", "tags", "last_activity", "net_unreachable", "pending_commands", "msg", "owners",
        "platforms", "tasks", "task"
    )

    # All client fields
    ALL_FIELDS = BASE_FIELDS + ("user_tags",)

    ClientTask = collections.namedtuple("ClientTask", ["id", "owner", "type"])

    def __init__(self, fields=None):
        self.fields = fields if fields is not None else self.BASE_FIELDS
        self.running_tasks = None
        self.user_tags_map = None

    @classmethod
    def get_running_tasks(cls, clients):
        cut = len(ctu.TokenSource.CLIENT + ":")
        q = mapping.OAuthCache.objects(source__in=[
            ":".join((ctu.TokenSource.CLIENT, c.hostname)) for c in clients
        ]).lite()
        cid2tids = {src[cut:]: [s.task_id for s in g] for src, g in it.groupby(q, lambda _: _.source)}
        q = mapping.Task.objects(
            id__in=it.chain.from_iterable(cid2tids.itervalues())
        ).fast_scalar("id", "owner", "type")
        tasks = {row[0]: cls.ClientTask(*row) for row in q}
        return {_: filter(None, map(tasks.get, cid2tids[_])) for _ in cid2tids.iterkeys()}

    @classmethod
    def get_user_tags(cls, clients):
        user_tags = set(tag for client in clients for tag in client.tags if tag in ctc.Tag.Group.USER)
        q = mapping.Group.objects(user_tags__name__in=list(user_tags))
        tags_group_mapping = {}
        for group in q:
            for tag in group.user_tags:
                if tag.name in user_tags:
                    tags_group_mapping[tag.name] = group

        result = {}

        user_groups = controller.Group.get_user_groups(context.current.user)

        for client in clients:
            user_tags_result = {
                "allowed": collections.defaultdict(list),
                "others": collections.defaultdict(list)
            }
            for tag in client.tags:
                if tag not in ctc.Tag.Group.USER:
                    continue
                group = tags_group_mapping.get(tag)
                if group is None:
                    continue
                edit_group_type = (
                    user_tags_result["allowed"]
                    if group.name in user_groups or context.current.user.super_user else
                    user_tags_result["others"]
                )
                edit_group_type[group.name].append(tag)
            result[client.hostname] = user_tags_result
        return result

    @abc.abstractproperty
    def base_url(self):
        pass

    def _dump(self, client):
        result_dict = {field: getattr(self, field)(client) for field in self.fields}
        return api.v1.schemas.client.ClientEntity.create(__selected_fields__=self.fields, **result_dict)

    def system_info(self, client):
        return client.info.get("system", {})

    def id(self, client):
        return client.hostname

    def uuid(self, client):
        return client.info.get("uuid")

    def url(self, client):
        return "{}/client/{}".format(self.base_url, client.hostname)

    def os(self, client):
        return api.v1.schemas.client.ClientOS.create(
            name=self.system_info(client).get("arch"),
            version=self.system_info(client).get("os_version"),
        )

    def ram(self, client):
        return client.hardware.ram << 20

    def ncpu(self, client):
        return client.hardware.cpu.cores

    def cpu(self, client):
        return client.hardware.cpu.model

    def platform(self, client):
        return common.platform.get_platform_alias(self.system_info(client).get("platform"))

    def lxc(self, client):
        return client.lxc

    def fqdn(self, client):
        return self.system_info(client).get("fqdn")

    def fileserver(self, client):
        return self.system_info(client).get("fileserver", None)

    def dc(self, client):
        return self.system_info(client).get("dc", "unk")

    def alive(self, client):
        return client.alive

    def availability(self, client):
        return client.available

    def disk(self, client):
        return api.v1.schemas.client.DiskInfo.create(
            status=self.system_info(client).get("disk_status", ctc.DiskStatus.OK),
            total_space=self.system_info(client).get("total_space", 0) << 20,
            free_space=max(0, self.system_info(client).get("free_space", 0) << 20),  # Don't return negative values
        )

    def tags(self, client):
        return list(client.pure_tags)

    def last_activity(self, client):
        return client.updated

    def net_unreachable(self, client):
        net_unreach_ts = client.info.get("net_unreach_ts", None)
        return net_unreach_ts and dt.datetime.utcfromtimestamp(net_unreach_ts)

    def pending_commands(self, client):
        return sorted(controller.Client.pending_service_commands(client))

    def msg(self, client):
        return client.info.get("msg", "")

    def owners(self, client):
        return client.info.get("owners", [])

    def platforms(self, client):
        if client.lxc:
            return common.platform.LXC_PLATFORMS
        else:
            return []

    def tasks(self, client):
        tasks = self.running_tasks.get(client.hostname)
        if tasks:
            return [
                api.v1.schemas.client.ClientTask.create(
                    id=task.id,
                    type=task.type,
                    owner=task.owner,
                    url="{}/task/{}".format(self.base_url, task.id),
                )
                for task in tasks
            ]
        else:
            return []

    def task(self, client):
        tasks = self.running_tasks.get(client.hostname)
        if tasks:
            return api.v1.schemas.client.ClientTaskOne.create(
                task_id=tasks[0].id,
                task_type=tasks[0].type,
                owner=tasks[0].owner,
                url="{}/task/{}".format(self.base_url, tasks[0].id)
            )
        else:
            return None

    def user_tags(self, client):
        return self.user_tags_map.get(client.hostname)


class SingleClientMapper(ClientMapperBase):
    @common.utils.classproperty
    def base_url(self):
        return context.current.request.base_url.rsplit("/", 2)[0]

    def dump(self, client, running_tasks=None):
        if running_tasks is not None:
            self.running_tasks = running_tasks
        elif self.fields is None or "tasks" in self.fields or "task" in self.fields:
            self.running_tasks = self.get_running_tasks([client])
        if "user_tags" in self.fields:
            self.user_tags_map = self.get_user_tags([client])
        return self._dump(client)


class ClientListMapper(ClientMapperBase):
    @common.utils.classproperty
    def base_url(self):
        return context.current.request.base_url.rsplit("/", 1)[0]

    def dump(self, clients, offset, limit, running_tasks=None):
        if running_tasks is not None:
            self.running_tasks = running_tasks
        elif self.fields is None or "tasks" in self.fields or "task" in self.fields:
            self.running_tasks = self.get_running_tasks(clients)
        if "user_tags" in self.fields:
            self.user_tags_map = self.get_user_tags(clients)
        return [
            self._dump(client)
            for client in it.islice(clients, offset, offset + limit)
        ]
