import collections

import six

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.user as ctu

from sandbox.web.api import v1
from sandbox.yasandbox import context, controller

import sandbox.serviceapi.handlers.common


class GroupMapperBase(object):
    BASE_FIELDS = (
        "name", "parent", "url", "abc", "email", "members", "rights", "priority_limits", "sources", "sync",
        "quota", "user_tags", "messenger_chat_id", "telegram_chat_id", "mds_transfer_resources",
        "mds_strong_mode", "juggler_settings", "abcd_account",
    )
    ALL_FIELDS = BASE_FIELDS + ("mds_quota", "users",)

    def __init__(self, fields=None, quotas=None):
        self.fields = fields or self.BASE_FIELDS
        self.quotas = quotas

    @common.utils.classproperty
    def base_url(cls):
        pass

    def name(self, group):
        return group.name

    def parent(self, group):
        return group.parent

    def url(self, group):
        return "{}/{}".format(self.base_url, group.id)

    def abc(self, group):
        return group.abc

    def email(self, group):
        return group.email

    def members(self, group):
        return sorted(group.get_users())

    def users(self, group):
        users = collections.defaultdict(set)
        for source in group.get_sources():
            for content in source.content:
                if len(content.logins) == 1 and content.logins[0] == content.name:
                    users["USER"].add(content.name)
                else:
                    users["{}:{}".format(source.source, content.name)].update(content.logins)
        return {k: v for k, v in users.items()}

    def rights(self, group):
        return ctu.Rights.get(context.current.user.login in group.users or context.current.user.super_user)

    def priority_limits(self, group):
        pl = group.priority_limits or controller.Group.regular.priority_limits
        return {
            attr: dict(
                zip(("class_", "subclass"), (
                    getattr(
                        ctu.DEFAULT_PRIORITY_LIMITS,
                        attr
                    )
                    if getattr(pl, attr) is None else
                    ctt.Priority().__setstate__(getattr(pl, attr))
                ).__getstate__()))
            for attr in ("ui", "api")
        }

    @staticmethod
    def sources(group):
        return [
            v1.schemas.group.GroupSyncSource.create(
                source=source.source,
                group=source.group,
                content=[
                    v1.schemas.group.GroupSyncSourceContent.create(name=content.name, logins=content.logins)
                    for content in source.content
                ]
            )
            for source in group.get_sources()
        ]

    @staticmethod
    def sync(group):  # TODO: compatibility with old UI (SANDBOX-5061)
        return {
            attr: getattr(group.sources[0], attr) for attr in ("source", "group")
        } if group.sources else None

    def quota(self, group):
        quotas = self.quotas
        if quotas is None:
            quotas = sandbox.yasandbox.controller.TaskQueue.owners_rating(owner=group.name)
        return sandbox.serviceapi.handlers.common.quota_info(group.name, quotas=quotas)

    @staticmethod
    def mds_quota(group):
        if common.config.Registry().common.installation not in ctm.Installation.Group.LOCAL:
            if group.bucket:
                namespace, bucket_stats = common.mds.S3.check_bucket(bucket=group.bucket)
            else:
                abc_service_id = common.abc.abc_service_id(group.abc)
                namespace, bucket_stats = common.mds.S3.check_bucket(abc_service_id)
            if namespace:
                return v1.schemas.group.GroupMDSQuota.create(
                    name=bucket_stats["name"],
                    used_space=bucket_stats["used_space"],
                    max_size=bucket_stats["max_size"],
                )
        return None

    @staticmethod
    def user_tags(group):
        return [_.name for _ in group.user_tags or []]

    def messenger_chat_id(self, group):
        return group.messenger_chat_id if self.rights(group) == ctu.Rights.WRITE else None

    def telegram_chat_id(self, group):
        return group.telegram_chat_id if self.rights(group) == ctu.Rights.WRITE else None

    @staticmethod
    def mds_transfer_resources(group):
        return group.mds_transfer_resources

    @staticmethod
    def mds_strong_mode(group):
        return group.mds_strong_mode

    @staticmethod
    def juggler_settings(group):
        return {
            "default_host": group.juggler_settings.default_host,
            "default_service": group.juggler_settings.default_service,
            "checks": {
                name: {"host": check.host, "service": check.service}
                for name, check in six.iteritems(group.juggler_settings.checks)
            }
        } if group.juggler_settings else None

    @staticmethod
    def abcd_account(group):
        return v1.schemas.abcd.Account.create(
            id=group.abcd_account.id,
            folder_id=group.abcd_account.folder_id,
            version=group.abcd_account.version,
            last_update=v1.schemas.abcd.LastUpdate.create(
                author=group.abcd_account.last_update.author,
                timestamp=group.abcd_account.last_update.timestamp,
                operation_id=group.abcd_account.last_update.operation_id,
            ) if group.abcd_account else None
        ) if group.abcd_account else None

    def dump(self, group):
        result_dict = {field: getattr(self, field)(group) for field in self.fields}
        return v1.schemas.group.Group.create(__selected_fields__=self.fields, **result_dict)


class GroupMapper(GroupMapperBase):
    def __init__(self, fields=None, quotas=None):
        fields = fields or self.ALL_FIELDS
        super(GroupMapper, self).__init__(fields, quotas)

    @common.utils.classproperty
    def base_url(cls):
        return context.current.request.base_url.rsplit("/", 1)[0]


class GroupListMapper(GroupMapperBase):
    @common.utils.classproperty
    def base_url(cls):
        return context.current.request.base_url
