import abc
import datetime as dt
import re

import flask

from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.common.types import user as ctu
from sandbox import common
from sandbox.web.api import v1
from sandbox.yasandbox.database import mapping
from sandbox.yasandbox import context, controller


class ResourceMetaMapper(object):
    @classmethod
    def dump(cls, resource_meta):
        meta = resource_meta.resource_meta
        return v1.schemas.resource.ResourceMeta.create(
            name=meta.name,
            parent=meta.parent,
            releaseable=meta.releaseable,
            releasable=meta.releasable,
            any_arch=meta.any_arch,
            releasers=meta.releasers,
            executable=meta.executable,
            auto_backup=meta.auto_backup,
            calc_md5=meta.calc_md5,
            on_restart=meta.on_restart,
            release_subscribers=meta.release_subscribers,
            share=meta.share,
            default_attribute=meta.default_attribute,
            attributes=meta.attributes
        )


class ResourceMapperBase(object):
    __metaclass__ = abc.ABCMeta

    @abc.abstractproperty
    def base_schema(self):
        pass

    @abc.abstractproperty
    def base_url(self):
        pass

    @common.utils.singleton_property
    def lightweight(self):
        if ctm.HTTPHeader.LINKS in flask.request.headers:
            return False

        if ctm.HTTPHeader.NO_LINKS in flask.request.headers:
            return True

        # TODO: TESTENV-1416: A special crutch to speedup requests from TestEnv
        return (
            common.config.Registry().server.auth.enabled and
            not context.current.request.session and context.current.user.login in ("robot-testenv", "guest")
        )

    @property
    def shuffle_links(self):
        # HTTP/rsync links are normally reordered on each request to balance the load.
        # We disable it for the UI so that the links do not jump around because of autoupdate.
        return context.current.request.source != ctt.RequestSource.WEB

    def dump_resource_base(self, resource):
        attributes = {a.key: a.value for a in resource.attributes}
        try:
            ttl = attributes.get("ttl", 14)
            ttl = int(ttl) if ttl != "inf" else None
        except (TypeError, ValueError):
            ttl = 14

        mds = None
        if resource.mds:
            mds = v1.schemas.resource.MDS.create(
                key=resource.mds.key,
                namespace=resource.mds.namespace,
                url=common.mds.s3_link(resource.mds.key, resource.mds.namespace),
                backup_url=(
                    common.mds.s3_link(resource.mds.key, resource.mds.backup_namespace)
                    if resource.mds.backup_namespace else
                    ""
                ),
                __filter_empty__=True,
            )

        TAR_ARCHIVES_REGEX = re.compile("\w+\.t((gz)|(ar(\.gz)?))$")

        observable = resource.multifile or bool(TAR_ARCHIVES_REGEX.search(resource.path))

        ret = self.base_schema.create(
            id=resource.id,
            rights=ctu.Rights.get(
                False if self.lightweight else controller.Resource.user_has_permission(resource, context.current.user)
            ),
            url="{}/{}".format(self.base_url, resource.id),
            type=resource.type,
            owner=resource.owner,
            arch=resource.arch,
            state=resource.state,
            description=resource.name,
            size=resource.size << 10,
            time=v1.schemas.resource.ResourceTime.create(
                created=resource.time.created,
                accessed=resource.time.accessed,
                expires=(resource.time.accessed + dt.timedelta(days=ttl)) if ttl else None,
                updated=resource.time.updated or resource.time.created
            ),
            task=v1.schemas.resource.ResourceTask.create(
                id=resource.task_id,
                url="{}/task/{}".format(self.base_url.rsplit("/", 1)[0], resource.task_id),
                status=None,  # will be filled later
            ),
            http=v1.schemas.resource.ResourceDataLinks.create(
                links=[],
                proxy="",
            ),
            attributes=attributes,
            skynet_id=resource.skynet_id,
            md5=resource.md5 or None,
            file_name=resource.path,
            mds=mds,
            multifile=resource.multifile,
            executable=resource.executable,
            observable=observable,
            system_attributes=v1.schemas.resource.SystemAttributes.create(
                linux_platform=resource.system_attributes.linux_platform,
                osx_platform=resource.system_attributes.osx_platform,
                osx_arm_platform=resource.system_attributes.osx_arm_platform,
                win_nt_platform=resource.system_attributes.win_nt_platform,
            ) if resource.system_attributes else None
        )

        return ret

    @abc.abstractmethod
    def dump(self, *args, **kwargs):
        pass


class SingleResourceMapper(ResourceMapperBase):
    @property
    def base_schema(self):
        return v1.schemas.resource.Resource

    @property
    def base_url(self):
        return context.current.request.base_url.rsplit("/", 1)[0]

    def dump(self, resource):
        rl = controller.ResourceLinks([resource], shuffle=self.shuffle_links)
        proxy_links = rl.proxy

        if self.lightweight:
            sources = rl.all_sources
            http_links = {}
            rsync_links = {}
        else:
            sources = rl.sources
            http_links = rl.http
            rsync_links = rl.rsync

        ret = self.dump_resource_base(resource)
        ret.task.status = mapping.Task.objects(id=resource.task_id).fast_scalar("execution__status").first() or ""

        ret.rsync = v1.schemas.resource.ResourceDataLinks.create(links=rsync_links.get(resource.id, []))
        ret.sources = sources[resource.id]
        ret.http.proxy = proxy_links.get(resource.id, "")
        ret.http.links = http_links.get(resource.id, [])
        ret.resource_meta = []

        resource_meta = []
        if ctm.HTTPHeader.RESOURCE_META in flask.request.headers:
            resource_meta = controller.Resource.resource_meta_list(resource) or []

        for meta in resource_meta:
            ret.resource_meta.append(ResourceMetaMapper.dump(meta))

        return ret


class ResourceListMapper(ResourceMapperBase):
    @property
    def base_schema(self):
        return v1.schemas.resource.ResourceListItem

    @property
    def base_url(self):
        return context.current.request.base_url

    def dump(self, docs):
        rets = []

        docs = list(docs)
        rl = controller.ResourceLinks(docs, shuffle=self.shuffle_links)

        proxy_links = rl.proxy
        if self.lightweight:
            http_links = {}
        else:
            http_links = rl.http

        task_query = mapping.Task.objects(id__in=set(doc.task_id for doc in docs))
        task_statuses = {tid: status for (tid, status) in task_query.fast_scalar("id", "execution__status")}

        for doc in docs:
            ret = self.dump_resource_base(doc)
            ret.task.status = task_statuses.get(doc.task_id, "")
            ret.http.proxy = proxy_links.get(doc.id, "")
            ret.http.links = http_links.get(doc.id, [])
            rets.append(ret)

        if ctm.HTTPHeader.RESOURCE_META in flask.request.headers:
            hashes = set()
            for resource in docs:
                if resource.resource_meta:
                    hashes.update(resource.resource_meta)
            resource_meta = {obj.hash: obj for obj in mapping.ResourceMeta.objects(hash__in=hashes)}

            for idx, ret in enumerate(rets):
                ret.resource_meta = [
                    ResourceMetaMapper.dump(resource_meta[meta_hash]) for meta_hash in docs[idx].resource_meta
                ]

        return rets


class BucketMapper(object):
    @classmethod
    def dump(cls, bucket):
        return v1.schemas.resource.Bucket.create(
            name=bucket.name,
            abc=bucket.abc,
            abcd_account=v1.schemas.abcd.Account.create(
                id=bucket.abcd_account.id,
                folder_id=bucket.abcd_account.folder_id,
                version=bucket.abcd_account.version,
                last_update=v1.schemas.abcd.LastUpdate.create(
                    author=bucket.abcd_account.last_update.author,
                    timestamp=bucket.abcd_account.last_update.timestamp,
                    operation_id=bucket.abcd_account.last_update.operation_id,
                ),
            ),
            ignore_bucket_exhaust=bucket.ignore_bucket_exhaust,
            lru_threshold=bucket.lru_threshold,
        )
