import collections

import sandbox.common.types.user as ctu
import sandbox.common.types.scheduler as cts

from sandbox import common
from sandbox.web.api import v1
from sandbox.yasandbox.database import mapping
from sandbox.yasandbox import context, controller


class SchedulerMapperBase(object):
    @classmethod
    def get_schedule(cls, scheduler):
        repetition = retry = None

        repetition_mode = scheduler.plan.repetition
        if repetition_mode == cts.Repetition.INTERVAL:
            repetition = {"interval": scheduler.plan.interval}
        elif repetition_mode == cts.Repetition.WEEKLY:
            days_of_week = scheduler.plan.days_of_week
            repetition = {"weekly": [i for i in xrange(7) if days_of_week & (1 << i)]}
        retry_mode = scheduler.plan.retry
        if retry_mode == cts.Retry.NO:
            retry = {"ignore": True}
        elif retry_mode == cts.Retry.INTERVAL:
            retry = {"interval": scheduler.plan.retry_interval}

        start_time = None
        if scheduler.plan.start_mode == cts.StartMode.SET:
            start_time = scheduler.plan.start_time
        return v1.scheduler.schemas.scheduler.SchedulerInterval.create(
            repetition=repetition,
            retry=retry,
            fail_on_error=(retry_mode == cts.Retry.FAILURE),
            sequential_run=scheduler.plan.sequential_run,
            start_time=start_time
        )

    @classmethod
    def get_time(cls, scheduler):
        return v1.schemas.scheduler.SchedulerTime.create(
            created=scheduler.time.created,
            updated=scheduler.time.updated,
            next=scheduler.time.next_run,
            last=scheduler.time.last_run
        )


class SchedulerListMapper(SchedulerMapperBase):
    SchedulerTask = collections.namedtuple("SchedulerTask", ["id", "status"])

    @classmethod
    def get_last_task(cls, last_task):
        if last_task is not None:
            return v1.scheduler.schemas.scheduler.SchedulerLastTask.create(
                id=last_task.id,
                url="{}/task/{}".format(context.current.request.base_url.rsplit("/", 1)[0], last_task.id),
                status=last_task.status,
            )
        else:
            return None

    @classmethod
    def get_task(cls, scheduler, last_task):
        return v1.schemas.scheduler.BaseTaskConfig.create(
            type=scheduler.type,
            description=common.utils.force_unicode_safe(scheduler.description),
            tags=scheduler.tags or [],
            last=cls.get_last_task(last_task),
        )

    @classmethod
    def dump_scheduler(cls, scheduler, last_task):
        write_access = controller.user_has_permission(context.current.request.user, [scheduler.owner])
        return v1.schemas.scheduler.SchedulerListItem.create(
            id=scheduler.id,
            rights=ctu.Rights.WRITE if write_access else ctu.Rights.READ,
            url="{}/{}".format(context.current.request.base_url, scheduler.id),
            scheduler_notifications=[
                v1.schemas.scheduler.SchedulerNotifications.create(
                    transport=notification.transport,
                    statuses=notification.statuses,
                    recipients=notification.recipients,
                    check_status=notification.check_status,
                    juggler_tags=notification.juggler_tags
                ) for notification in (
                    scheduler.scheduler_notifications.notifications if scheduler.scheduler_notifications else []
                )
            ],
            author=scheduler.author,
            owner=scheduler.owner,
            status=scheduler.status,
            schedule=cls.get_schedule(scheduler),
            time=cls.get_time(scheduler),
            task=cls.get_task(scheduler, last_task)
        )

    @classmethod
    def dump(cls, schedulers):
        schedulers_task = mapping.Task.last_task_per_scheduler([o.id for o in schedulers])
        tasks_statuses = dict(mapping.Task.objects(
            id__in=schedulers_task.values()).fast_scalar("id", "execution__status"))

        schedulers_last_tasks = {
            key: cls.SchedulerTask(value, tasks_statuses[value])
            for key, value in schedulers_task.iteritems()
        }
        return [
            cls.dump_scheduler(scheduler, schedulers_last_tasks.get(scheduler.id))
            for scheduler in schedulers
        ]
