import os
import itertools as it

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc

from sandbox.yasandbox import controller


SOLOMON_AGENT = "solomon_agent"  # samogon servant name of solomon agent
SALMON = "salmon_sysmond"  # samogon servant name of salmon sysmond


class TaskAuditMapperRaw(object):
    @classmethod
    def dump(cls, doc):
        return {
            # Required fields
            "time": doc["dt"].isoformat() + "Z",
            "target": doc["h"],
            "task_id": doc["ti"],
            # May be absent
            "iface": doc.get("sc", None),
            "status": doc.get("st", None),
            "author": doc.get("a", None),
            "source": doc.get("ip", None),
            "message": doc.get("ct", None),
            "request_id": doc.get("ri", None),
            "wait_targets": doc.get("wait_targets", {}),
        }

    @classmethod
    def dump_list(cls, docs):
        return [cls.dump(doc) for doc in docs]


class AuditHostsListItemEntry(dict):
    __fqdns = {}
    __partitions = {}
    __clients = {}

    def __set_till(self, item, value):
        item["till"] = value
        item["cpu_usage"]["e"] = value
        item["virt_mem"]["e"] = value
        item["disk_usage"]["e"] = value

    def __new__(cls, items, doc):
        self = dict.__new__(cls)
        self.__doc = doc
        if items and not items[-1].get("till"):
            if doc.status == ctt.Status.STOPPED:
                self.__set_till(
                    items[-1],
                    items[-1].__date
                    if items[-1].__doc.status == ctt.Status.STOPPING else
                    self.__date
                )
            elif doc.status not in it.chain(
                ctt.Status.Group.REALEXECUTE,
                (ctt.Status.FINISHING, ctt.Status.STOPPING)
            ):
                self.__set_till(items[-1], self.__date)
        elif doc.status in ctt.Status.Group.REALEXECUTE:
            self["host"] = self.__host
            self["from_"] = self.__date
            self["partition"] = self.__partition
            self["cpu_usage"] = self.__cpu_usage
            self["virt_mem"] = self.__virt_mem
            self["disk_usage"] = self.__disk_usage
            self["client_id"] = self.__doc.client
            items.append(self)
        if items:
            items[-1].__doc = doc
        return items

    @property
    def __client(self):
        if self.__doc.client:
            if self.__doc.client in self.__clients:
                return self.__clients[self.__doc.client]
            client = controller.Client.get(self.__doc.client)
            self.__clients[self.__doc.client] = client
            return client
        return None

    @property
    def __partition(self):
        if self.__doc.client:
            partition = self.__partitions.get(self.__doc.client, ctm.NotExists)
            if partition is not ctm.NotExists:
                return partition
            client = self.__client
            tasks_dir = client.info["system"]["tasks_dir"]
            partition = self.__partitions[self.__doc.client] = os.sep.join(tasks_dir.split(os.sep)[:2])
            return partition
        return os.sep

    @property
    def __host(self):
        if self.__doc.client:
            fqdn = self.__fqdns.get(self.__doc.client, ctm.NotExists)
            if fqdn is not ctm.NotExists:
                return fqdn
            client = controller.Client.get(self.__doc.client)
            fqdn = self.__fqdns[self.__doc.client] = client.info.get("system", {}).get("fqdn") or self.__doc.remote_ip
            return fqdn
        return self.__doc.remote_ip

    @property
    def __date(self):
        return self.__doc.date

    @property
    def __base_info(self):
        if self.__doc.client:
            return {
                "project": common.config.Registry().common.solomon.pull.project,
                "cluster": common.config.Registry().common.solomon.pull.cluster,
                "service": common.config.Registry().common.solomon.pull.service,
                "b": self.__date,
                "e": self.get("till"),
                "host": self.__doc.client
            }
        return {}

    def __osx_client(self):
        return set(ctc.Tag.Group.OSX) & set(self.__client.tags)

    @property
    def __cpu_usage(self):
        result = {}
        if self.__client:
            if self.__osx_client():
                result = {
                    "path": "/System/UserTime",
                    "servant": SOLOMON_AGENT,
                }
            else:
                result = {
                    "info": "cpu_usage",
                    "partition": "load"
                }
            result.update(self.__base_info)
        return result

    @property
    def __virt_mem(self):
        result = {}
        if self.__client:
            if self.__osx_client():
                result = {
                    "path": "/Memory/Active",
                    "servant": SOLOMON_AGENT,
                }
            else:
                result = {
                    "partition": "total_rss"
                }
            result.update(self.__base_info)
        return result

    @property
    def __disk_usage(self):
        result = {}
        if self.__client:
            if self.__osx_client():
                result = {
                    "path": "/Filesystem/UsedB",
                    "servant": SOLOMON_AGENT,
                }
            else:
                result = {
                    "servant": SALMON,
                    "sensor": "/Filesystem/UsedB"
                }
            result.update(self.__base_info)
        return result
