import sys
import time
import logging
import requests
import threading

import struct
import msgpack

from sandbox import common
import sandbox.common.types.misc as ctm

from sandbox.serviceapi import mules
from sandbox.serviceapi import constants as sa_consts

from sandbox.serviceq import client as qclient


__all__ = ("BaseAggregator", "run_safe")


logger = logging.getLogger(__name__)


class BaseAggregator(object):
    EMPTY_SERIALIZED_BANNED_LIST = msgpack.dumps([[], []], use_bin_type=True)
    UPDATE_INTERVAL = 1
    API_CONSUMPTION_TABLE_UPDATE_INTERVAL = 10
    __metaclass__ = common.utils.ThreadSafeSingletonMeta
    _thread = None
    _delta = []
    _length_struct = struct.Struct("i")
    revision = 0

    def __init__(self):
        self.logger = logging.getLogger("aggregator")
        self.qclient = qclient.Client()

    @common.utils.singleton_property
    def _lock(self):
        return threading.Lock()

    @common.utils.singleton_property
    def _wakeup(self):
        return threading.Event()

    @classmethod
    def send_msg(cls, delta):
        if len(delta) > 0:
            import uwsgi
            uwsgi.mule_msg(msgpack.dumps(delta, use_bin_type=True), 3)


class Aggregator(BaseAggregator):
    _revision_thread = None
    _api_quota_thread = None
    _update_banned_list = int(time.time())

    @common.utils.singleton_classproperty
    def _revision_wakeup(self):
        return threading.Event()

    @common.utils.singleton_classproperty
    def _api_wakeup(self):
        return threading.Event()

    def squeezed_delta(self):
        with self._lock:
            delta = self._delta
            self._delta = []

        squeezed_delta = []
        for login, timestamp, consumption, web_consumption in delta:
            if not len(squeezed_delta) or squeezed_delta[-1][:2] != (login, timestamp):
                squeezed_delta.append((login, timestamp, consumption, web_consumption))
            else:
                squeezed_delta[-1] = (
                    login, timestamp, consumption + squeezed_delta[-1][2],
                    web_consumption + squeezed_delta[-1][3]
                )
        return squeezed_delta

    def send(self, delta):
        return self.qclient.push_api_quota(delta)

    def update_banned_list(self):
        delta = self.squeezed_delta()
        try:
            serialized_banned_list = self.send(delta)
        except Exception as ex:
            self.logger.error(
                "Error in communication with serviceq. Run next interval without api quotas.", exc_info=ex
            )
            serialized_banned_list = self.EMPTY_SERIALIZED_BANNED_LIST

        import uwsgi
        uwsgi.sharedarea_wlock(sa_consts.UWSGIKey.API_QUOTAS)
        banned_list_length = len(serialized_banned_list)
        revision = self.revision
        logger.info("Write banned list with length %s and revision %s", banned_list_length, revision)
        self._update_banned_list = int(time.time())
        uwsgi.sharedarea_memoryview(sa_consts.UWSGIKey.API_QUOTAS)[0:4] = self._length_struct.pack(revision)
        uwsgi.sharedarea_memoryview(sa_consts.UWSGIKey.API_QUOTAS)[4:8] = self._length_struct.pack(banned_list_length)
        uwsgi.sharedarea_memoryview(sa_consts.UWSGIKey.API_QUOTAS)[8:8 + banned_list_length] = serialized_banned_list
        uwsgi.sharedarea_unlock(sa_consts.UWSGIKey.API_QUOTAS)

    def update_revision(self):
        try:
            response = requests.get("".join((sa_consts.LegacyAPI.url, "http_check")))
            self.revision = int(response.headers[ctm.HTTPHeader.TASKS_REVISION])
        except:
            pass
        if int(time.time()) - self._update_banned_list > 120:
            import uwsgi
            self.logger.warning("Too long update banned list, unlock sharedarea.")
            uwsgi.sharedarea_unlock(sa_consts.UWSGIKey.API_QUOTAS)

    def update_api_quota_table(self):
        import uwsgi
        try:
            consumption_table = self.qclient.get_api_quotas_table()
            self.logger.info("Update quota consumptions")
            uwsgi.cache_update(
                sa_consts.ApiConsumption.CONSUMPTION_KEY_NAME,
                consumption_table,
                sa_consts.ApiConsumption.UWSGI_CACHE_TTL,
                sa_consts.ApiConsumption.UWSGI_CACHE_NAME
            )
        except Exception as ex:
            self.logger.error(
                "Error in communication with serviceq. Run next interval without api quota table update.",
                exc_info=ex
            )

    def timer_proc(self, func, wakeup, sleep_time):
        def proc():
            while True:
                wakeup.wait(sleep_time)
                wakeup.clear()
                func()

        return proc

    def start_timer_daemon(self, func, wakeup, sleep_time):
        thread = threading.Thread(target=self.timer_proc(func, wakeup, sleep_time))
        thread.daemon = True
        thread.start()
        return thread

    def run(self):
        import uwsgi
        logger.info("Start Aggregator mule")
        self._revision_thread = self.start_timer_daemon(
            self.update_revision, self._revision_wakeup, self.UPDATE_INTERVAL
        )
        self._thread = self.start_timer_daemon(self.update_banned_list, self._wakeup, self.UPDATE_INTERVAL)
        self._api_quota_thread = self.start_timer_daemon(
            self.update_api_quota_table, self._api_wakeup, self.API_CONSUMPTION_TABLE_UPDATE_INTERVAL
        )
        self._update_banned_list = int(time.time())

        while True:
            try:
                message = msgpack.loads(uwsgi.mule_get_msg())
            except Exception:
                logger.error("Can't load message, skip it")
                continue

            with self._lock:
                self._delta.extend(message)


def run_safe():
    mules.setup_mule_logging()
    sys.modules["serviceq.errors"] = sys.modules["sandbox.serviceq.errors"]
    logging.getLogger("urllib3.connectionpool").setLevel(logging.WARNING)

    try:
        Aggregator().run()
    except:
        logger.exception("Got exception in Aggregator mule!")
        raise
