import time
import logging
import threading
import cPickle as pickle

from sandbox.common import config, utils
from sandbox.serviceapi import mules


__all__ = ("send_msg", "run_safe", "Sensor")


logger = logging.getLogger(__name__)

registry = config.Registry()


def send_msg(method, args):
    import uwsgi
    uwsgi.mule_msg(pickle.dumps((method, args)), 2)


def _cache_updater(metrics):
    import uwsgi

    while True:
        uwsgi.cache_update(
            "solomon", pickle.dumps(metrics.to_json()),
            0,  # never expire
            "metrics",  # cache_name
        )
        time.sleep(5)


def run():
    import uwsgi
    logger.info("Start Metrics mule")

    metrics = Metrics()

    updater_thread = threading.Thread(target=_cache_updater, args=(metrics,))
    updater_thread.daemon = True
    updater_thread.start()

    while True:
        try:
            message = pickle.loads(uwsgi.mule_get_msg())
        except pickle.UnpicklingError:
            logger.error("Can't unpickle message, skip it")
            continue

        try:
            method, (name, labels, kind) = message
        except ValueError:
            logger.error("Invalid message, skip it: %r", message)
            continue

        sensor = metrics.get_sensor(name, labels, kind)

        if method == Sensor.Op.INC:
            sensor.value += 1
        elif method == Sensor.Op.DEC:
            sensor.value -= 1
        else:
            logger.error("Unknown method '%s'", method)


def run_safe():
    mules.setup_mule_logging()

    try:
        run()
    except:
        logger.exception("Got exception in Metrics mule!")
        raise


class Sensor(object):
    class Kind(utils.Enum):
        DGAUGE = None
        RATE = None

    class Op(utils.Enum):
        INC = None
        DEC = None

    def __init__(self, name, labels, kind=Kind.DGAUGE):
        self.name = name
        self.labels = labels
        self.value = 0
        self.kind = kind

    def to_json(self):
        return {
            "value": self.value,
            "labels": dict({"sensor": self.name}, **self.labels),
            "kind": str(self.kind)
        }


class Metrics(object):
    def __init__(self):
        self._sensors = {}

    def _get_sensor_key(self, name, labels):
        key = frozenset(labels.items()).union({name})
        return key

    def get_sensor(self, name, labels, kind):
        key = self._get_sensor_key(name, labels)

        if key not in self._sensors:
            self._sensors[key] = Sensor(name, labels, kind)

        return self._sensors[key]

    def to_json(self):
        return [sensor.to_json() for sensor in self._sensors.values()]
