import json
import time
import logging

from sandbox import common

from sandbox.serviceapi import mules
from sandbox.serviceapi import constants as sa_consts

from sandbox.yasandbox.database import mapping

__all__ = ("run_safe",)


logger = logging.getLogger(__name__)


class Suggest(object):
    SUGGEST_TYPES = ("task", "resource", "client", "template")  # Types of suggest for cache

    @common.patterns.singleton_classproperty
    def legacy_client(cls):
        return common.rest.Client(sa_consts.LegacyAPI.url + "api/v1.0", max_timeout=60, total_wait=120, max_interval=60)

    @classmethod
    def add_suggest(cls, suggest_type, value):
        import uwsgi
        uwsgi.cache_update(
            suggest_type, value, sa_consts.Suggest.UWSGI_CACHE_TTL, sa_consts.Suggest.UWSGI_CACHE_NAME
        )

    @classmethod
    def legacy_request(cls, suggest_type):
        r = json.dumps(cls.legacy_client.suggest[suggest_type].read())
        cls.add_suggest(suggest_type, r)
        return r

    @classmethod
    def template_suggest(cls, suggest_type):
        r = json.dumps(
            [
                {"alias": alias, "task_type": task_type, "description": description}
                for alias, task_type, description in
                mapping.TaskTemplate.objects.fast_scalar("alias", "task.type", "description")
            ]
        )
        cls.add_suggest(suggest_type, r)
        return r

    HANDLERS = {
        "task": legacy_request.__func__,
        "resource": legacy_request.__func__,
        "client": legacy_request.__func__,
        "template": template_suggest.__func__
    }

    @classmethod
    def suggest(cls, suggest_type):
        return cls.HANDLERS[suggest_type](cls, suggest_type)


def run_safe():
    mules.setup_mule_logging()
    mapping.ensure_connection()

    try:
        while True:
            for suggest_type in Suggest.HANDLERS.iterkeys():
                Suggest.suggest(suggest_type)
            time.sleep(sa_consts.Suggest.CACHE_SLEEP_INTERVAL)

    except:
        logger.exception("Got exception in Aggregator mule!")
        raise
