import httplib
import urlparse
import traceback

import flask
import requests
import distutils.util
from sandbox.common import api
from sandbox.common.types import misc as ctm
from sandbox.serviceapi.web import exceptions, routes
from sandbox.web.api.v2.schemas import batch as batch_schemas

__all__ = ("init_plugin",)

bp = flask.Blueprint("batch", __name__)


PRIMITIVE_TYPES = (basestring, int, long, float)

# These headers propagate from batch endpoint to its subrequests
COMMON_REQUEST_HEADERS = (
    "Authorization",
    "Cookie",
)

# Only these headers are preserved in subresponses
ALLOWED_RESPONSE_HEADERS = frozenset((
    "location",
    "x-request-metrics",
    "x-request-duration",
))

# Non-standard code for cancelled requests
HTTP_NO_RESPONSE = 424  # Failed Dependency


@bp.route("/api/v2/batch", accept_all_methods=True)
def batch_endpoint(path=None):
    from sandbox.serviceapi import plugins

    try:
        fail_fast = flask.request.args.get('fail_fast', '0')
        fail_fast = bool(distutils.util.strtobool(fail_fast))
    except ValueError as ex:
        raise exceptions.BadRequest("Invalid `fail_fast` value: {}".format(ex))

    try:
        raw_data = flask.request.json
        subrequests = api.Array(batch_schemas.BatchSubRequest, collection_format=api.ArrayFormat.MULTI).decode(raw_data)
    except ValueError as ex:
        raise exceptions.BadRequest("Unable to parse input data: {}".format(ex))

    common_headers = {}
    for header in COMMON_REQUEST_HEADERS:
        value = flask.request.headers.get(header)
        if value is not None:
            common_headers[header] = value

    subresponses = []

    cancel_with_status = None

    for req_data in subrequests:

        if cancel_with_status:
            subresponses.append({
                "status": cancel_with_status,
                "response": None,
                "headers": {},
            })
            continue

        params = req_data.params or {}

        for key, value in params.iteritems():
            if isinstance(value, PRIMITIVE_TYPES):
                continue
            if isinstance(value, list) and all(isinstance(x, PRIMITIVE_TYPES) for x in value):
                continue
            raise exceptions.BadRequest("params[{}] is not primitive or array of primitives".format(key))

        headers = req_data.headers or {}
        headers.update(common_headers)

        # The prepared request is used to normalise input data, such as query parameters and JSON body
        req = requests.Request(
            req_data.method,
            url=urlparse.urljoin("http://dummy", req_data.path),
            params=params,
            json=req_data.data or None,
            headers=headers,
        ).prepare()

        context = dict(
            method=req.method,
            path=req.path_url,
            data=req.body,
            headers=headers,
            base_url=flask.request.host_url,
            environ_base=flask.request.environ,
            environ_overrides={
                # Getting real IP via `wsgi.input` does not work for subrequests.
                # `request.remote_addr` is used as a fallback instead.
                # Here we make sure that the real IP is propagated to `request.remote_addr`.
                "REMOTE_ADDR": plugins.context.get_request_remote_ip()
            }
        )

        with flask.current_app.app_context():
            flask.g.is_batch = True
            with flask.current_app.test_request_context(**context):
                try:
                    rv = flask.current_app.preprocess_request()
                    if rv is None:
                        rv = flask.current_app.dispatch_request()
                except Exception as e:
                    rv = flask.current_app.handle_user_exception(e)

                response = flask.current_app.make_response(rv)
                # Post process Request
                response = flask.current_app.process_response(response)

        resp_headers = {k: v for k, v in response.headers if k.lower() in ALLOWED_RESPONSE_HEADERS}

        subresponses.append({
            "status": response.status_code,
            "response": response.json,
            "headers": resp_headers,
        })

        # Redirect
        if 301 <= response.status_code < 400 and ctm.HTTPHeader.LOCATION in resp_headers:
            # the client will resolve the redirect and proceed with the remaining requests
            cancel_with_status = httplib.SERVICE_UNAVAILABLE

        # Early failure
        elif fail_fast and response.status_code >= 400:
            cancel_with_status = HTTP_NO_RESPONSE

    return flask.jsonify(subresponses), httplib.MULTI_STATUS,


def init_plugin(app):
    # Cannot use `routes.http_error_handler` as it requires context, which is undefined at this point
    def base_error_handler(ex):
        body = {"reason": str(ex), "error": type(ex).__name__, "traceback": traceback.format_exc()}
        return flask.jsonify(body), httplib.INTERNAL_SERVER_ERROR

    routes.apply_max_concurrent_requests_guard(bp)
    bp.errorhandler(exceptions.HttpError)(routes.http_error_handler)
    bp.errorhandler(Exception)(base_error_handler)
    app.register_blueprint(bp)
