import uuid
import socket
import logging

import flask

from sandbox import common
import sandbox.common.types.misc as ctm

from sandbox.yasandbox import context


try:
    import uwsgi
except ImportError:
    uwsgi = None


__all__ = ("init_plugin",)


class ReqidLogger(logging.LoggerAdapter):
    def process(self, msg, kwargs):
        reqid = self.extra["reqid"]
        subreq_id = self.extra["subreq_id"]
        if subreq_id:
            reqid = '{}:{}'.format(reqid, subreq_id)
        return "{{{}}} {}".format(reqid, msg), kwargs


def get_request_remote_ip():
    # First try to get remote IP from X-Forwarded-For and alike headers
    for addr in reversed(flask.request.access_route):
        if addr == "0.0.0.0":
            continue
        return addr

    # Determine peer's address via request's file descriptor number
    try:
        fd = flask.request.environ["wsgi.input"].fileno()
        sock = socket.fromfd(fd, socket.AF_INET6, socket.SOCK_STREAM)
        return sock.getpeername()[0]
    except Exception:
        pass  # Normal situation for batch subrequests where `wsgi.input` is simulated

    # Fallback for flask default
    return flask.request.remote_addr


def remove_ipv6_prefix(addr):
    if addr.startswith("::ffff:"):
        return addr[7:]
    return addr


def before_request():
    # Create request-specific logger and base context with it
    req_id = flask.request.headers.get(ctm.HTTPHeader.REQUEST_ID) or ("_" + uuid.uuid4().hex[1:])
    subreq_id = flask.request.headers.get(ctm.HTTPHeader.SUBREQUEST_ID)
    if subreq_id is not None:
        subreq_id = subreq_id[:8]
    logger = ReqidLogger(logging.getLogger(__name__), {"reqid": req_id[:8], "subreq_id": subreq_id})
    context.set_current(context.Context(logger, None))

    flask.request.req_id = req_id
    # Try to get real remote IP for the current request
    flask.request.remote_ip = remove_ipv6_prefix(get_request_remote_ip())

    if uwsgi:
        uwsgi.set_logvar("req_id", req_id[:8])
        uwsgi.set_logvar("remote_ip", flask.request.remote_ip)
        uwsgi.set_logvar("component", flask.request.headers.get(ctm.HTTPHeader.COMPONENT, "x"))


def teardown_request(exception=None):
    context.set_current(None)


def init_plugin(app):
    logging.getLogger().addHandler(common.log.ExceptionSignalsSenderHandler())
    app.before_request(before_request)
    app.teardown_request(teardown_request)
