import json
import flask
import httplib
import datetime
import requests

from sandbox.serviceapi.web import exceptions

from sandbox.common import config
import sandbox.common.types.misc as ctm

from sandbox.yasandbox import context, controller
from sandbox.yasandbox.database import mapping
from sandbox.serviceapi import metrics
from sandbox.serviceapi.web import routes, aggregator, statistics
from sandbox.serviceapi import constants as sa_consts


__all__ = ("init_plugin",)


# This headers will be installed by uwsgi
EXCLUDE_HEADERS = (
    "content-encoding",
    "content-length",
    "transfer-encoding",
    "connection",
)

# Endpoints for which ServiceAPI requests TVM2 tickets and passes them to legacy server via headers.
TVM2_ENABLED_PATHS = ()

bp = flask.Blueprint("legacy", __name__)


def make_error_response(error, code):
    body = {"reason": httplib.responses[code], "error": error}
    return flask.jsonify(body), code


def shall_enable_tvm(path):
    is_local = config.Registry().common.installation in ctm.Installation.LOCAL
    if is_local and not config.Registry().server.services.tvmtool.enabled:
        # This means that local Sandbox has not been reconfigured to use tvmtool
        return False
    path = path.rstrip('/')
    return any(map(path.endswith, TVM2_ENABLED_PATHS))


# Unknown API-requests
@bp.route("/api/v1.0/<path:path>", accept_all_methods=True)
@bp.route("/api/json/<path:path>", accept_all_methods=True)
@bp.route("/sandbox/xmlrpc", accept_all_methods=True)
# Legacy UI handlers
@bp.route("/sandbox/<path:path>", accept_all_methods=True)
def legacy_routes(path=None):
    context.current.logger.debug(
        "Proxy request to legacy web-server: %s %s",
        flask.request.method.upper(), flask.request.full_path
    )

    request = context.current.request

    request_headers = {k: v for k, v in flask.request.headers.iteritems() if k.lower() != "host"}
    request_headers.pop(ctm.HTTPHeader.CURRENT_USER, None)
    if context.current.user != controller.User.anonymous:
        request_headers[ctm.HTTPHeader.CURRENT_USER] = context.current.user.login
    request_headers[ctm.HTTPHeader.REQUEST_SOURCE] = request.source
    if ctm.HTTPHeader.FORWARDED_FOR not in request_headers:
        request_headers[ctm.HTTPHeader.FORWARDED_FOR] = flask.request.remote_ip
    request_headers[ctm.HTTPHeader.REQUEST_TARGET] = flask.request.host_url
    if request_headers.get(ctm.HTTPHeader.TRANSFER_ENCODING, "") == "chunked":
        request_headers.pop(ctm.HTTPHeader.TRANSFER_ENCODING)
    if ctm.HTTPHeader.REQUEST_ID not in request_headers:
        request_headers[ctm.HTTPHeader.REQUEST_ID] = request.id
    if ctm.HTTPHeader.PROFILER in request_headers and request.profiler is None:
        request_headers.pop(ctm.HTTPHeader.PROFILER, None)

    if path and shall_enable_tvm(path):
        try:
            tvm_headers = request.get_tvm_headers("yav")
            request_headers.update(tvm_headers)
        except Exception:
            context.current.logger.warning("Could not fetch TVM headers")

    request.proxy_started = datetime.datetime.utcnow()
    try:
        resp = requests.request(
            method=flask.request.method,
            url=flask.request.url.replace(flask.request.host_url, sa_consts.LegacyAPI.url, 1),
            headers=request_headers,
            data=flask.request.get_data(),
            cookies=flask.request.cookies,
            allow_redirects=False,
        )
    except EnvironmentError as ex:
        return make_error_response(
            str(ex),
            httplib.SERVICE_UNAVAILABLE
        )
    finally:
        request.proxy_finished = datetime.datetime.utcnow()

    # Count all legacy redirects
    metrics.rate_inc("legacy_requests")

    return flask.current_app.response_class(
        response=resp.content,
        status=resp.status_code,
        headers={
            k: v
            for k, v in resp.headers.items()
            if k.lower() not in EXCLUDE_HEADERS
        },
    )


def after_request(r):
    statistics.StatRecorder.record(r, legacy=True)

    mapping.base.tls.request = None
    if flask.request.rejected or flask.request.rejected_in_progress:
        # Request is rejected due to high load by serviceapi, do nothing
        return r

    if ctm.HTTPHeader.REQUEST_ID not in r.headers:
        r.headers[ctm.HTTPHeader.REQUEST_ID] = flask.request.req_id

    ctx = context.current
    if ctx.user is None:
        ctx.logger.error("Current user is None, before_request has failed")
        return r

    if config.Registry().server.api.quotas.enabled:
        aggregator.Aggregator().add_delta()

    if ctx.request.profiler is not None:
        legacy_result = True
        try:
            response = json.loads(r.response[0])
        except:
            legacy_result = False
            response = r.response
        return routes.profiler_response(response, legacy_result=legacy_result)

    if r.status_code == httplib.SERVICE_UNAVAILABLE:
        # Request is rejected due to high load, do nothing
        return r

    if ctx.request.is_authenticated and ctx.request.need_reset_cookie:
        r.headers[ctm.HTTPHeader.RESET_SESSION] = "true"

    aggregator.Aggregator().update_api_consumption_headers(r.headers, ctx.user.login)
    return r


def init_plugin(app):
    routes.apply_max_concurrent_requests_guard(bp)
    bp.before_request(routes.before_request)
    routes.support_transfer_encoding(bp)
    bp.after_request(after_request)
    bp.errorhandler(exceptions.HttpError)(routes.http_error_handler)
    bp.errorhandler(Exception)(routes.base_error_handler)
    app.register_blueprint(bp)
