import httplib
import threading

import flask
import requests

import sandbox.common.types.misc as ctm

from sandbox import common
from sandbox.yasandbox import context
from sandbox.yasandbox.database import mapping
from sandbox.serviceapi import metrics
from sandbox.serviceapi.web import routes, exceptions, request


try:
    import uwsgi
except ImportError:
    uwsgi = None


__all__ = ("init_plugin",)


# Thread-local for requests session to MDS storage
tls = threading.local()
tls.session = None


bp = flask.Blueprint("mds", __name__)


@bp.route("/", methods=[ctm.RequestMethod.GET, ctm.RequestMethod.HEAD])
@bp.route("/<path:path>", methods=[ctm.RequestMethod.GET, ctm.RequestMethod.HEAD])
def mds_handler(path=None):
    path = flask.request.path.lstrip("/")
    if common.config.Registry().common.installation in ctm.Installation.Group.NONLOCAL:
        if not path.startswith("favicon"):
            path = "index.html"
            preferences = context.current.user.preferences.get("prefs", {})
            if preferences.get("interface", common.types.misc.Interface.NEW) == common.types.misc.Interface.NEW:
                if common.config.Registry().common.installation == ctm.Installation.PRODUCTION:
                    path = "index.next.html"
                else:
                    path = "index.next.prestable.html"
        else:
            if common.config.Registry().common.installation == ctm.Installation.PRE_PRODUCTION:
                basename, ext = path.split(".")
                path = "{}.{}.{}".format(basename, common.config.Registry().common.installation.lower(), ext)

    if common.config.Registry().common.installation == ctm.Installation.LOCAL:
        url = "https://sandbox-prestable.yandex-team.ru/" + path
    else:
        url = "{}/{}".format(common.config.Registry().server.web.static.mds_s3_url, path)
    context.current.logger.info("Proxy %s from %r", flask.request.method, url)

    if tls.session is None:
        tls.session = requests.Session()
    method = tls.session.head if flask.request.method == "HEAD" else tls.session.get

    headers = {k.lower(): v for k, v in flask.request.headers}
    headers.pop("host", None)
    # `requests` implicitly sets "Accept-Encoding: gzip, deflate", unless specified otherwise.
    # Set value to `None` by default, which means header absence.
    headers.setdefault("accept-encoding", None)

    # Make request to MDS
    try:
        r = method(url, headers=headers, timeout=10, stream=True)
        r.raw.decode_content = False
        resp_content = r.raw.read()
    except requests.exceptions.RequestException as e:
        if common.config.Registry().common.installation == ctm.Installation.LOCAL:
            return (
                "Possible there is no access to sandbox-prestable.yandex-team.ru."
                "Please, make new rule in puncher from your host or net to sandbox-prestable.yandex-team.ru:443."
            ), httplib.SERVICE_UNAVAILABLE
        else:
            context.current.logger.error("Failed to make request: %s", e)
            return str(e), httplib.SERVICE_UNAVAILABLE

    # Count mds redirects
    metrics.rate_inc("mds_requests")
    metrics.rate_inc("serviceapi_requests")

    resp_headers = {k.lower(): v for k, v in r.headers.items()}
    resp_headers.pop("connection", None)
    resp_headers.pop("transfer-encoding", None)
    resp_headers["Cache-Control"] = "no-cache, no-store, must-revalidate"
    resp_headers["Pragma"] = "no-cache"
    resp_headers["Expires"] = 0

    return flask.current_app.response_class(
        status=r.status_code,
        response=resp_content,
        headers=resp_headers,
    )


def before_request():
    mapping.base.tls.request = None
    req = request.Request(flask.request, context.current.logger)
    flask.request.rejected_in_progress = False
    ctx = routes.RequestContext(req)  # Create new context and replace base one
    context.set_current(ctx)
    if uwsgi:
        uwsgi.set_logvar("login", context.current.user.login)


def init_plugin(app):
    routes.apply_max_concurrent_requests_guard(bp)
    bp.before_request(before_request)
    bp.errorhandler(exceptions.HttpError)(routes.http_error_handler)
    app.register_blueprint(bp)
