import cPickle as pickle

import flask

from sandbox import common
from sandbox.serviceapi.mules import metrics
from sandbox.serviceapi.web import aggregator


__all__ = ("init_plugin",)


bp = flask.Blueprint("solomon", __name__)


def collect_sensors():
    # Import here because `uwsgi` module only available when app is running under uwsgi server.
    import uwsgi

    yield {
        "labels": {"sensor": "total_requests"},
        "value": uwsgi.total_requests(),
        "kind": metrics.Sensor.Kind.RATE,
    }

    def metric(name, sensor=None, deriv=False):
        if sensor is None:
            sensor = name
        ret = {
            "labels": {"sensor": sensor},
            "value": uwsgi.metric_get(name),
        }
        if deriv:
            ret["kind"] = metrics.Sensor.Kind.RATE
        return ret

    # Custom metrics
    yield metric("requests_in_progress")

    # Builtin metrics
    yield metric("core.busy_workers", "workers_busy")
    yield metric("core.idle_workers", "workers_idle")
    yield metric("core.total_rss", "total_rss_bytes")
    yield metric("core.total_vsz", "total_vsz_bytes")
    yield metric("core.total_tx", "total_tx_bytes", deriv=True)
    yield metric("core.total_running_time", "total_running_time", deriv=True)

    avg_response_times = [float(worker["avg_rt"]) for worker in uwsgi.workers() if worker["status"] != "cheap"]
    avg_response_time = sum(avg_response_times) / len(avg_response_times)

    yield {
        "labels": {"sensor": "avg_response_time"},
        "value": avg_response_time // 1000,  # Convert to milliseconds
    }

    for sensor in pickle.loads(uwsgi.cache_get("solomon", "metrics")):
        yield sensor

    revision = aggregator.Aggregator().revision

    if revision:
        yield {
            "labels": {"sensor": "tasks_revision"},
            "value": revision,
            "kind": str(metrics.Sensor.Kind.DGAUGE)
        }

    mask = uwsgi.sharedarea_memoryview(0)[:(common.config.Registry().server.api.workers + 8) / 8]
    requests_in_progress = 0
    index = 0

    for submask in mask:
        for i in range(8):
            if index > common.config.Registry().server.api.workers:
                break
            index += 1
            requests_in_progress += ((ord(submask) >> i) & 1)

    yield {
        "labels": {"sensor": "requests_in_progress"},
        "value": requests_in_progress,
        "kind": str(metrics.Sensor.Kind.DGAUGE)
    }


@bp.route("/solomon/sensors.json")
def solomon_sensors():
    payload = {"sensors": [sensor for sensor in collect_sensors()]}
    return flask.jsonify(payload)


def init_plugin(app):
    app.register_blueprint(bp)
