import socket
import httplib

import sandbox.common.joint.errors
import sandbox.common.joint.socket

import sandbox.serviceq.errors

from sandbox.yasandbox.database import mapping


class HttpError(Exception):
    def __init__(self, code_or_message, message=None):
        if isinstance(code_or_message, int):
            self._code = code_or_message
            self.message = message
        else:
            self._code = None
            self.message = code_or_message
            assert message is None

    @property
    def code(self):
        assert self._code
        return self._code

    @property
    def error(self):
        return httplib.responses.get(self.code, "Unknown error")


class BadRequest(HttpError):
    code = httplib.BAD_REQUEST


class Unauthorized(HttpError):
    code = httplib.UNAUTHORIZED


class Forbidden(HttpError):
    code = httplib.FORBIDDEN


class NotFound(HttpError):
    code = httplib.NOT_FOUND


class Gone(HttpError):
    code = httplib.GONE


class UserDismissed(HttpError):
    code = 451
    error = "User dismissed"


class ServiceUnavailable(HttpError):
    code = httplib.SERVICE_UNAVAILABLE


class Conflict(HttpError):
    code = httplib.CONFLICT


class NotAcceptable(HttpError):
    code = httplib.NOT_ACCEPTABLE


class RequestTimeout(HttpError):
    code = httplib.REQUEST_TIMEOUT


class Locked(HttpError):
    code = httplib.LOCKED


# Response with code 503 if these exceptions occurred
RETRIABLE_EXCEPTIONS = (
    mapping.ConnectionError,
    mapping.AutoReconnect,
    mapping.OperationFailure,
    mapping.OperationError,
    sandbox.serviceq.errors.QNeedValidation,
    sandbox.serviceq.errors.QTimeout,
    sandbox.serviceq.errors.QRetry,
    sandbox.serviceq.errors.QSemaphoresTemporaryUnavailable,
    sandbox.serviceq.errors.QAcquireError,
    sandbox.common.joint.errors.Timeout,
    sandbox.common.joint.errors.HandshakeError,
    sandbox.common.joint.errors.CallError,
    sandbox.common.joint.errors.ServerUnavailable,
    sandbox.common.joint.socket.EOF,
    socket.error,
)
