import httplib

import flask

import sandbox.common.types.user as ctu
import sandbox.common.types.misc as ctm

from sandbox.common import utils
from sandbox.yasandbox import context
from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping

from . import exceptions


def check_for_readonly(allow_ro):
    """
    Check if cluster is read-only and request is allowed to proceed.

    :param allow_ro: whether request is allowed to execute when cluster is read-only
    """
    if (
        flask.request.method != ctm.RequestMethod.GET and not allow_ro and
            controller.Settings.mode() == controller.Settings.OperationMode.READ_ONLY
    ):
        raise exceptions.ServiceUnavailable("Sandbox is in READ_ONLY mode")


def check_security_scope(scope):
    """
    Check for permissions for current request and user.

    :param scope: required permission scope
    """
    req = context.current.request
    # Set individual reply for each condition if it's the only one to check
    cond_reply = {
        ctu.Restriction.ANY: (True, (None, None)),  # Just a stub
        ctu.Restriction.AUTHENTICATED: (req.is_authenticated, (httplib.UNAUTHORIZED, "Unauthorized")),
        ctu.Restriction.ADMIN: (req.user.super_user, (httplib.FORBIDDEN, "Forbidden")),
        ctu.Restriction.TASK: (req.is_task_session, (httplib.FORBIDDEN, "Allowed only with task session token")),
        ctu.Restriction.STATISTICS: (req.is_statistics_sender, (httplib.FORBIDDEN, "Forbidden")),
        ctu.Restriction.TVM_SERVICE: (req.tvm_service, (httplib.FORBIDDEN, "Forbidden")),
        ctu.Restriction.SESSION_MAKER: (
            req.is_authenticated and controller.User.has_roles(req.user, [ctu.Role.SESSION_MAKER]),
            (httplib.FORBIDDEN, "Forbidden to create session without SESSION_MAKER role")
        ),
        ctu.Restriction.EXTERNAL_SESSION: (
            req.is_external_session, (httplib.FORBIDDEN, "Allowed with external session token")
        )
    }
    restrictions = list(utils.chain(scope))

    for restriction in restrictions:
        cond, _ = cond_reply[restriction]
        if cond:
            return

    if len(restrictions) == 1:
        _, (code, message) = cond_reply[restrictions[0]]
    else:
        code = httplib.FORBIDDEN
        message = "Your request matches none of these conditions: {}".format(
            ", ".join(r.lower() for r in restrictions)
        )

    raise exceptions.HttpError(code, message)


def wrap_temporary_errors(func):
    def raise_bad_request(e):
        context.current.logger.exception("Can't complete operation in database")
        raise exceptions.BadRequest(str(e))

    def raise_temporary(e):
        context.current.logger.exception("Temporary error during request")
        if isinstance(e, mapping.AutoReconnect):
            controller.Settings.on_master_lost()
        raise exceptions.ServiceUnavailable(str(e))

    def temporary_errors_wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except exceptions.RETRIABLE_EXCEPTIONS as e:
            if mapping.is_query_error(e):
                raise_bad_request(e)
            else:
                raise_temporary(e)

    return temporary_errors_wrapper


def wrap_after_request_temporary_errors(func):
    def return_bad_request(e):
        context.current.logger.exception("Can't complete operation in database on request postprocessing")
        return flask.current_app.response_class(
            response=str(e),
            status=httplib.BAD_REQUEST
        )

    def return_temporary_error(e):
        context.current.logger.exception("Temporary error during request postprocessing")
        if isinstance(e, mapping.AutoReconnect):
            controller.Settings.on_master_lost()
        return flask.current_app.response_class(
            response=str(e),
            status=httplib.SERVICE_UNAVAILABLE
        )

    def temporary_errors_wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except exceptions.RETRIABLE_EXCEPTIONS as e:
            if mapping.is_query_error(e):
                return return_bad_request(e)
            else:
                return return_temporary_error(e)

    return temporary_errors_wrapper
