import json
import datetime

import flask

from sandbox.common import utils
from sandbox.common import config
from sandbox.common import statistics as common_statistics
import sandbox.common.types.misc as ctm
import sandbox.common.types.statistics as ctss
from sandbox.yasandbox import context
from sandbox.serviceapi.web import request


class StatRecorder(object):
    REQID_LENGTH = 8

    @utils.classproperty
    def current_request(cls):
        try:
            return context.current.request
        except AttributeError:
            return request.Request(flask.request, context.current.logger, authenticate=False)

    @classmethod
    def __inner_record(cls, resp, legacy=False):
        ctx = context.current
        registry = config.Registry()
        req = cls.current_request
        """:type req: sandbox.serviceapi.web.request.Request"""

        request_duration = (datetime.datetime.utcnow() - req.request_started).total_seconds()
        request_duration_ms = request_duration * 1000

        total_duration = resp.headers.get(ctm.HTTPHeader.TOTAL_DURATION, default=0, type=int)
        task_id = req.session.task_id if req.session else None

        login = ctx.user.login if ctx.user is not None else ""
        statistics = {
            "type": ctss.SignalType.API_CALL,
            "date": req.request_started,
            "timestamp": req.request_started,
            "user": login,
            "source": req.source,
            "server": registry.this.fqdn,
            "read_preference": req.read_preference,
            "response_code": resp.status_code,
            "duration": request_duration_ms,
            "reqid": req.id,
            "task_id": task_id,
            "query_string": req.query_string,
            "quota_owner": req.quota_owner,
            "total_duration": total_duration + request_duration_ms,
            "retry_reason": resp.headers.get(ctm.HTTPHeader.RETRY_REASON),
            "proxy_duration": (req.proxy_finished - req.proxy_started).total_seconds() * 1000,
            "processing_duration": 0
        }
        if legacy:
            measures = resp.headers.pop(ctm.HTTPHeader.REQUEST_MEASURES, None)
            if measures:
                measures = json.loads(measures)
                statistics.update({
                    "processing_duration": float(resp.headers.get(ctm.HTTPHeader.REQ_DURATION, 0)) * 1000,
                    "method": measures["method"],
                    "mongodb_duration": measures["mongodb_duration"],
                    "serviceq_duration": measures["serviceq_duration"],
                })
                common_statistics.Signaler().push(statistics)
        else:
            statistics.update({
                "method": "{} [{}]".format(req.req.url_rule.rule, req.req.method),
                "mongodb_duration": ctx.spans.total_duration_for("mongodb_op") * 1000,
                "serviceq_duration": ctx.spans.total_duration_for("serviceq_call") * 1000,
            })
            common_statistics.Signaler().push(statistics)

    @classmethod
    def record(cls, resp, legacy=False):
        """
        :param resp: ServiceAPI response object
        :type resp: flask.Response
        :param legacy: don't record statistics for calls proxied to the old web server
        """
        try:
            cls.__inner_record(resp, legacy=legacy)
        except Exception as exc:
            expected = isinstance(exc, (ValueError, TypeError))
            message = (
                "Failed to record API call stats: %s"
                if expected else
                "Unexpected exception while recording API call stats: %s"
            )
            if context.current.logger is not None:
                context.current.logger.error(message, exc, exc_info=not expected)
