import os
import mmap
import struct
import collections

import msgpack

from sandbox import common
import sandbox.serviceq.types as qtypes

ST_POSITION = struct.Struct("<I")
ST_SIZE = struct.Struct("<I")
ST_OPERATION_ID = struct.Struct("<Q")


class Operation(
    collections.namedtuple(
        "Operation",
        ("operation_id", "method", "args", "prev_checksum", "checksum")
    ),
    qtypes.Serializable
):
    pass

Operation.__new__.__defaults__ = (1, 0, (), 0, 0)


class OperationJournal(object):
    DEFAULT_SIZE = 16 << 20  # journal default size in bytes

    class Header(object):
        TAIL_OFFSET = 0
        TAIL_LENGTH = ST_POSITION.size
        COUNTER_OFFSET = TAIL_NEXT_OFFSET = TAIL_OFFSET + TAIL_LENGTH
        COUNTER_LENGTH = ST_SIZE.size
        OPERATION_ID_OFFSET = COUNTER_NEXT_OFFSET = COUNTER_OFFSET + COUNTER_LENGTH
        OPERATION_ID_LENGTH = ST_OPERATION_ID.size
        DATA_OFFSET = OPERATION_ID_NEXT_OFFSET = OPERATION_ID_OFFSET + OPERATION_ID_LENGTH

        def __init__(self, mm):
            self.__mm = mm
            self.__tail = None
            self.__counter = None
            self.__operation_id = None

        @property
        def tail(self):
            if self.__tail is not None:
                return self.__tail
            self.__tail = ST_POSITION.unpack(self.__mm[self.TAIL_OFFSET:self.TAIL_NEXT_OFFSET])[0]
            return self.__tail

        @tail.setter
        def tail(self, value):
            self.__tail = value
            self.__mm[self.TAIL_OFFSET:self.TAIL_NEXT_OFFSET] = ST_POSITION.pack(value)

        @property
        def counter(self):
            if self.__counter is not None:
                return self.__counter
            self.__counter = ST_SIZE.unpack(self.__mm[self.COUNTER_OFFSET:self.COUNTER_NEXT_OFFSET])[0]
            return self.__counter

        @counter.setter
        def counter(self, value):
            self.__counter = value
            self.__mm[self.COUNTER_OFFSET:self.COUNTER_NEXT_OFFSET] = ST_SIZE.pack(value)

        @property
        def operation_id(self):
            if self.__operation_id is not None:
                return self.__operation_id
            self.__operation_id = ST_OPERATION_ID.unpack(
                self.__mm[self.OPERATION_ID_OFFSET:self.OPERATION_ID_NEXT_OFFSET]
            )[0]
            return self.__operation_id

        @operation_id.setter
        def operation_id(self, value):
            self.__operation_id = value
            self.__mm[self.OPERATION_ID_OFFSET:self.OPERATION_ID_NEXT_OFFSET] = ST_OPERATION_ID.pack(value)

    def __init__(self, path, file_size=DEFAULT_SIZE, force_create=False, logger=None):
        self.__path = path
        self.__newly_created = force_create or not os.path.exists(path)
        self.__logger = logger
        if self.__newly_created:
            if logger:
                logger.info("Creating new operation journal of size %s: %s", common.utils.size2str(file_size), path)
            self.__file_size = file_size
            with open(path, "wb") as f:
                f.seek(self.__file_size - 1)
                f.write("\0")
        else:
            self.__file_size = os.path.getsize(path)
            if logger:
                logger.info("Opening existing journal of size %s: %s", common.utils.size2str(self.__file_size), path)
        self.__file = open(path, "r+b")
        self.__mm = mmap.mmap(self.__file.fileno(), 0)
        self.__header = self.Header(self.__mm)
        if self.__newly_created:
            self.__header.tail = self.__header.DATA_OFFSET
            self.__header.counter = 0
            self.__header.operation_id = 0

    def __iter__(self):
        next_pos = self.__header.DATA_OFFSET
        while next_pos < self.__header.tail:
            data_pos = next_pos + ST_SIZE.size
            size = ST_SIZE.unpack(self.__mm[next_pos:next_pos + ST_SIZE.size])[0]
            next_pos = data_pos + size
            yield self.unpack_operation(self.__mm[data_pos:next_pos])

    def __fix_overflow(self, pos):
        if pos >= self.__file_size:
            self.__file_size <<= 1
            self.__mm.resize(self.__file_size)
        return pos

    @property
    def newly_created(self):
        return self.__newly_created

    @property
    def counter(self):
        return self.__header.counter

    @property
    def operation_id(self):
        return self.__header.operation_id

    @property
    def data_size(self):
        return self.__header.tail

    @property
    def file_size(self):
        return self.__file_size

    @property
    def path(self):
        return self.__path

    @staticmethod
    def pack_operation(operation):
        return msgpack.packb(operation.encode())

    @staticmethod
    def unpack_operation(data):
        return Operation.decode(msgpack.unpackb(data))

    def add(self, operation, operation_id, raw=False):
        data = operation if raw else self.pack_operation(operation)
        size_pos = self.__header.tail
        data_pos = self.__fix_overflow(size_pos + ST_SIZE.size)
        new_tail = self.__fix_overflow(data_pos + len(data))
        self.__mm[size_pos:data_pos] = ST_SIZE.pack(len(data))
        self.__mm[data_pos:new_tail] = data
        self.__header.tail = new_tail
        self.__header.counter += 1
        self.__header.operation_id = operation_id
        return data

    def flush(self):
        self.__mm.flush()
